
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "../utilities/openglheader.h"

#include "../utilities/utilities.h"
#include "scanner.h"
#include "readsmf.h"

/* ////////////////////////////////////////////////////////////////////////// */
#define SMF_FIRST_KEY              100
#define SMF_KEY_BEGIN              100
#define SMF_KEY_BIND               101
#define SMF_KEY_C                  102
#define SMF_KEY_E                  103
#define SMF_KEY_END                104
#define SMF_KEY_F                  105
#define SMF_KEY_FACE               106
#define SMF_KEY_FN                 107
#define SMF_KEY_FT                 108
#define SMF_KEY_G                  109
#define SMF_KEY_N                  110
#define SMF_KEY_Q                  111
#define SMF_KEY_R                  112
#define SMF_KEY_ROT                113
#define SMF_KEY_SCALE              114
#define SMF_KEY_SET                115
#define SMF_KEY_T                  116
#define SMF_KEY_T_SCALE            117
#define SMF_KEY_T_TRANS            118
#define SMF_KEY_TEX                119
#define SMF_KEY_TRANS              120
#define SMF_KEY_V                  121
#define SMF_KEY_VERTEX             122
#define SMF_KEY_VERTEX_CORRECTION  123
#define SMF_KEY_VN                 124
#define SMF_KEY_VT                 125
#define SMF_KEY_X                  126
#define SMF_KEY_Y                  127
#define SMF_KEY_Z                  128
#define SMF_NUM_KEYWORDS (SMF_KEY_Z-SMF_FIRST_KEY+1)

const char *smf_keyword[SMF_NUM_KEYWORDS] =
  { "begin",
    "bind",
    "c",
    "e",
    "end",
    "f",
    "face",
    "fn",
    "ft",
    "g",
    "n",
    "q",
    "r",
    "rot",
    "scale",
    "set",
    "t",
    "t_scale",
    "t_trans",
    "tex",
    "trans",
    "v",
    "vertex",
    "vertex_correction",
    "vn",
    "vt",
    "x",
    "y",
    "z" };

typedef struct {
    FILE    *f;
    int     nvert, ntr, ntrv, vertc;
    GLfloat *vc;
    GLint   *trv;
  } smf_data;

/* ////////////////////////////////////////////////////////////////////////// */
void smf_GetNextSymbol ( scanner *sc )
{
  int ns;

  ns = GetNextSymbol ( sc );
  if ( ns == SYMB_IDENT )
    sc->nextsymbol = BinSearchNameTable ( SMF_NUM_KEYWORDS,
                       SMF_FIRST_KEY, smf_keyword, sc->nextname );
} /*smf_GetNextSymbol*/

void smf_SkipToLineEnd ( scanner *sc )
{
  while ( sc->nextchar != '\n' && sc->nextchar != EOF )
    GetNextChar ( sc );
  if ( sc->nextchar == '\n' ) {
    GetNextChar ( sc );
    smf_GetNextSymbol ( sc );
  }
} /*smf_SkipToLineEnd*/

static int _smf_ReadInputFile ( void *userdata, int buflength, char *buffer )
{
  smf_data *data;

  data = (smf_data*)userdata;
  return fread ( buffer, 1, buflength, data->f );
} /*_smf_ReadInputFile*/

char smf_OpenInputFile ( scanner *sc, smf_data *data, const char *filename )
{
  if ( !(data->f = fopen ( filename, "r+" )) )
    return false;
  if ( InitScanner ( sc, 0, NULL, MAX_NAME_LENGTH, NULL, '#', '\n',
                     _smf_ReadInputFile, (void*)data ) ) {
    smf_GetNextSymbol ( sc );
    return true;
  }
  else {
    sc->userdata = (void*)data;
    fclose ( data->f );
    return false;
  }
} /*smf_OpenInputFile*/

int smf_GetSign ( scanner *sc )
{
  int sign;

  sign = 1;
  if ( sc->nextsymbol == SYMB_MINUS ) {
    smf_GetNextSymbol ( sc );
    sign = -1;
  }
  else if ( sc->nextsymbol == SYMB_PLUS )
    smf_GetNextSymbol ( sc );
  return sign;
} /*smf_GetSign*/

int smf_GetSignedInt ( scanner *sc )
{
  int sign, num;

  sign = smf_GetSign ( sc );
  if ( sc->nextsymbol == SYMB_INTEGER ) {
    num = sc->nextinteger;
    smf_GetNextSymbol ( sc );
    return sign*num;
  }
  else {
    sc->nextsymbol = SYMB_ERROR;
    return 0;
  }
} /*smf_GetSignedInt*/

double smf_GetSignedFloat ( scanner *sc )
{
  double sign, num;

  sign = (double)smf_GetSign ( sc );
  if ( sc->nextsymbol == SYMB_INTEGER || sc->nextsymbol == SYMB_FLOAT ) {
    num = sc->nextfloat;
    smf_GetNextSymbol ( sc );
    return sign*num;
  }
  else {
    sc->nextsymbol = SYMB_ERROR;
    return 0.0;
  }
} /*smf_GetSignedFloat*/

char smf_CmdSequence ( scanner *sc, GLfloat mm[16] )
{
  int              i, j, fvn;
  GLfloat          tm[16], tr[16], tt[16];
  GLfloat          va[3], phi;
  smf_data *data;

  data = (smf_data*)sc->userdata;
  fvn = data->nvert;
  memcpy ( tr, mm, 16*sizeof(GLfloat) );
  for (;;) {
    switch ( sc->nextsymbol ) {
case SMF_KEY_BEGIN:
      smf_GetNextSymbol ( sc );
      if ( smf_CmdSequence ( sc, tr ) ) {
        if ( sc->nextsymbol == SMF_KEY_END ) {
          smf_GetNextSymbol ( sc );
          return true;
        }
      }
      return false;

case SMF_KEY_END:
case SYMB_EOF:
      return true;

case SYMB_ERROR:
      return false;

case SYMB_PERCENT:  /* komentarz */
case SYMB_COLON:    /* symbole danych ignorowanych */
case SMF_KEY_BIND:
case SMF_KEY_C:
case SMF_KEY_E:
case SMF_KEY_FACE:
case SMF_KEY_FN:
case SMF_KEY_FT:
case SMF_KEY_G:
case SMF_KEY_N:
case SMF_KEY_R:
case SMF_KEY_TEX:
case SMF_KEY_T_SCALE:
case SMF_KEY_T_TRANS:
case SMF_KEY_VN:
case SMF_KEY_VT:
case SMF_KEY_VERTEX:
case SMF_KEY_VERTEX_CORRECTION:
      smf_SkipToLineEnd ( sc );
      break;

case SMF_KEY_F:  /* sciana trojkatna */
case SMF_KEY_T:
      if ( data->ntr >= MAXFAC ) return false;  /* za duzo scian */
      smf_GetNextSymbol ( sc );
      for ( i = 0; i < 3; i++ ) {
        if ( sc->nextsymbol == SYMB_INTEGER ) {
          data->trv[data->ntrv++] = sc->nextinteger + fvn + data->vertc - 1;
          smf_GetNextSymbol ( sc );
        }
        else return false;  /* blad skladni */
      }
      data->ntr ++;
      break;

case SMF_KEY_Q:  /* sciana czworokatna - zamieniam na 2 trojkaty */
      if ( data->ntr-1 >= MAXFAC ) return false;  /* za duzo scian */
      smf_GetNextSymbol ( sc );
      for ( i = 0, j = data->ntrv;  i < 4;  i++ ) {
        if ( sc->nextsymbol == SYMB_INTEGER ) {
          data->trv[data->ntrv++] = sc->nextinteger + fvn + data->vertc - 1;
          smf_GetNextSymbol ( sc );
        }
        return false;  /* blad skladni */
      }
      data->trv[data->ntrv++] = data->trv[j++];
      data->trv[data->ntrv++] = data->trv[j++];
      data->ntr += 2;
      break;

case SMF_KEY_ROT:
      smf_GetNextSymbol ( sc );
      i = sc->nextsymbol;
      smf_GetNextSymbol ( sc );
      phi = smf_GetSignedFloat ( sc );
      if ( sc->nextsymbol != SYMB_ERROR )
        phi *= PI/180.0;
      else
        return false;
      switch ( i ) {
    case SMF_KEY_X:
        M4x4RotateXf ( tt, phi );
        goto rotate;
    case SMF_KEY_Y:
        M4x4RotateYf ( tt, phi );
        goto rotate;
    case SMF_KEY_Z:
        M4x4RotateZf ( tt, phi );
rotate:
        M4x4Multf ( tm, tr, tt );
        memcpy ( tr, tm, 16*sizeof(GLfloat) );
        break;
    default:
        return false;  /* blad skladni */
      }
      break;

case SMF_KEY_SCALE:
      smf_GetNextSymbol ( sc );
      for ( i = 0; i < 3; i++ ) {
        va[i] = smf_GetSignedFloat ( sc );
        if ( sc->nextsymbol == SYMB_ERROR )
          return false;
      }
      M4x4Scalef ( tt, va[0], va[1], va[2] );
      M4x4Multf ( tm, tr, tt );
      memcpy ( tr, tm, 16*sizeof(GLfloat) );
      break;

case SMF_KEY_TRANS:
      smf_GetNextSymbol ( sc );
      for ( i = 0; i < 3; i++ ) {
        va[i] = smf_GetSignedFloat ( sc );
        if ( sc->nextsymbol == SYMB_ERROR )
          return false;
      }
      M4x4Translatef ( tt, va[0], va[1], va[2] );
      M4x4Multf ( tm, tr, tt );
      memcpy ( tr, tm, 16*sizeof(GLfloat) );
      break;

case SMF_KEY_SET:
      smf_GetNextSymbol ( sc );
      if ( sc->nextsymbol == SMF_KEY_VERTEX_CORRECTION ) {
        smf_GetNextSymbol ( sc );
        data->vertc = smf_GetSignedInt ( sc );
        if ( sc->nextsymbol == SYMB_ERROR )
          return false;
      }
      else
        smf_SkipToLineEnd ( sc );
      break;

case SMF_KEY_V:
      if ( data->nvert >= MAXVERT ) return false;  /* za duzo wierzcholkow */
      smf_GetNextSymbol ( sc );
      for ( i = 0; i < 3; i++ ) {
        va[i] = smf_GetSignedFloat ( sc );
        if ( sc->nextsymbol == SYMB_ERROR )
          return false;
      }
      M4x4MultMP3f ( &data->vc[3*data->nvert], tr, va );
      data->nvert ++;
      break;

default:
      printf ( "SYMBOL %d (%c) at line %d, column %d\n",
               sc->nextsymbol, sc->nextsymbol,
               sc->linenum+1, sc->colnum+1 );
      sc->nextsymbol = SYMB_ERROR;
      return false;
    }
  }
} /*smf_CmdSequence*/

char ReadSMFFile ( char *fn, int *nv, GLfloat **vc, int *ntr, GLint **trv )
{
  scanner  sc;
  smf_data data;
  GLfloat  tr[16];
  int      i;

  data.nvert = data.vertc = data.ntr = data.ntrv = 0;
  *vc = NULL;  *trv = NULL;
  if ( !smf_OpenInputFile ( &sc, &data, fn ) )
    return false;

  data.vc = malloc ( MAXVERT*3*sizeof(GLfloat) );
  data.trv = malloc ( MAXFAC*3*sizeof(GLint) );
  if ( data.vc && data.trv ) {
    M4x4Identf ( tr );
    smf_CmdSequence ( &sc, tr );
    ShutDownScanner ( &sc );
    fclose ( data.f );
    printf ( "nvert = %d, ntr = %d\n", data.nvert, data.ntr );
    if ( data.ntr > 0 ) {
      for ( i = 0; i < 3*data.ntr; i++ )
        if ( data.trv[i] < 0 || data.trv[i] >= data.nvert )
          goto failure;
      *vc = data.vc;    *nv = data.nvert;
      *trv = data.trv;  *ntr = data.ntr;
      return true;
    }
  }
failure:
  if ( data.vc ) free ( data.vc );
  if ( data.trv ) free ( data.trv );
  return false;
} /*ReadSMFFile*/

