
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "../utilities/util-debug.h"
#include "../utilities/utilities.h"
#include "app4.h"
#include "app4proc.h"
#include "trans.h"
#include "lights.h"
#include "balance.h"
#include "app4struct.h"
#include "myscene.h"
#include "room.h"
#include "table.h"
#include "chair.h"
#include "bookcase.h"
#include "galleon.h"
#include "candle.h"

#define MIN_ZOOM 0.1
#define MAX_ZOOM 5.0
#define ZOOM_FCT 1.05

static const float zenith[3] = {0.0,0.0,1.0};

AppData appdata;


static void SetupModelMatrix ( AppData *ad )
{
  M4x4Identf ( ad->trans.mm );
  LoadMMatrix ( &ad->trans, NULL );
} /*SetupModelMatrix*/

static void LoadViewMatrix ( AppData *ad )   
{
/*
printf ( "pos = (%f,%f,%f)\n",
 ad->trans.eyepos[0], ad->trans.eyepos[1], ad->trans.eyepos[2] );
printf ( "angk = %f, rotv = (%f,%f,%f)\n",
 ad->camera.viewer_rangle, ad->camera.viewer_rvec[0],
 ad->camera.viewer_rvec[2], ad->camera.viewer_rvec[2] );
*/
  M4x4InvTranslatefv ( ad->trans.vm, ad->trans.eyepos );
  M4x4RotateVMfv ( ad->trans.vm, ad->camera.viewer_rvec,
                   -ad->camera.viewer_rangle );
  LoadVPMatrix ( &ad->trans );
} /*LoadViewMatrix*/

static void Verticalise ( float vk[3], double *angk )
{
  float  R[16], s, c, vr[3];
  double theta, ang;

  M4x4RotateVfv ( R, vk, *angk );
  s = -V3DotProductf ( &R[0], zenith );
  c = V3DotProductf ( &R[4], zenith );
  theta = atan2 ( s, c );
  V3CompRotationsf ( vr, &ang, &R[8], theta, vk, *angk );
  memcpy ( vk, vr, 3*sizeof(float) );
  *angk = ang;
} /*Verticalise*/

static void InitViewMatrix ( AppData *ad )
{
  const float viewer_pos0[4] = {0.6,-0.8,0.0,1.0};
  const float viewer_rvec0[3] = {0.6,0.8,0.0};

  memcpy ( ad->trans.eyepos, viewer_pos0, 4*sizeof(GLfloat) );
  memcpy ( ad->camera.viewer_rvec, viewer_rvec0, 3*sizeof(float) );
  ad->camera.viewer_rangle = 0.4*PI;
  ad->vertical = true;
  Verticalise ( ad->camera.viewer_rvec, &ad->camera.viewer_rangle );
  LoadViewMatrix ( ad );
} /*InitViewMatrix*/

static void _RotateViewer ( AppData *ad,
                            double delta_xi, double delta_eta, char origin )
{
  float  vi[3], lgt, vk[3], vpos[4], z;
  double angi, angk;
  Camera *cam;

  if ( delta_xi == 0 && delta_eta == 0 )
    return;  /* natychmiast uciekamy - nie chcemy dzielic przez 0 */
  cam = &ad->camera;
  vi[0] = (float)delta_eta*(cam->top-cam->bottom)/(float)cam->win_height;
  vi[1] = (float)delta_xi*(cam->right-cam->left)/(float)cam->win_width;
  vi[2] = 0.0;
  lgt = sqrt ( V3DotProductf ( vi, vi ) );
  vi[0] /= lgt;  vi[1] /= lgt;
  z = pow ( ZOOM_FCT, ad->camera.zoomexp );
  angi = 0.0087266463*z;  /* 0.5 stopnia */
  if ( ad->vertical )
    angi *= 0.01 + 0.99*fabs ( ad->trans.vm[1]*zenith[0] +
             ad->trans.vm[5]*zenith[1] + ad->trans.vm[9]*zenith[2] );
  V3CompRotationsf ( vk, &angk, cam->viewer_rvec, cam->viewer_rangle, vi, angi );
  if ( ad->vertical )
    Verticalise ( vk, &angk );
  memcpy ( cam->viewer_rvec, vk, 3*sizeof(float) );
  cam->viewer_rangle = angk;
  if ( origin ) {
    M4x4MultMTV3f ( vk, ad->trans.vm, vi );
    M4x4RotateVfv ( ad->trans.vm, vk, angi );
    M4x4MultMVf ( vpos, ad->trans.vm, ad->trans.eyepos );
    memcpy ( ad->trans.eyepos, vpos, 4*sizeof(float) );
  }
  LoadViewMatrix ( ad );
} /*_RotateViewer*/

void RotateViewer ( double delta_xi, double delta_eta, char origin )
{
  _RotateViewer ( &appdata, delta_xi, delta_eta, origin );
} /*RotateViewer*/

static char _MoveViewer ( AppData *ad, char dir )
{
  static float
    stepb[3] = {0.0,0.0,-0.01}, stepf[3] = {0.0,0.0,0.01},
    stepd[3] = {0.0,-0.01,0.0}, stepu[3] = {0.0,0.01,0.0},
    stepl[3] = {-0.01,0.0,0.0}, stepr[3] = {0.01,0.0,0.0};
  float *w, v[3];
  int i;

  switch ( dir ) {
case MOVE_UP:      w = stepu;  break;
case MOVE_DOWN:    w = stepd;  break;
case MOVE_LEFT:    w = stepl;  break;
case MOVE_RIGHT:   w = stepr;  break;
case MOVE_FORWARD: w = stepf;  break;
case MOVE_BACK:    w = stepb;  break;
default: return false;
  }
  M4x4MultMTV3f ( v, ad->trans.vm, w );
  for ( i = 0; i < 3; i++ )
    ad->trans.eyepos[i] += v[i];
  LoadViewMatrix ( ad );
  return true;
} /*_MoveViewer*/

char MoveViewer ( char dir )
{
  return _MoveViewer ( &appdata, dir );
} /*MoveViewer*/
                  
void ToggleLampCandle ( AppData *ad )
{
  ad->mainlamp_on = !ad->mainlamp_on;
  SetLightSurfBit ( &ad->light, 1, ad->mainlamp_on );
  SetMainLampOnOff ( &ad->obj[ad->nobjects-2], &ad->trans, &ad->mat,
                     &ad->light, 1, ad->mainlamp_on );
  SetCandleFlameOnOff ( &ad->obj[ad->nobjects-1], &ad->trans, &ad->mat,
                        &ad->light, 1, !ad->mainlamp_on );
  ad->shadows_ok = ad->radiance_ok = false;
} /*ToggleLampCandle*/

static void InitLights ( AppData *ad )
{
  GLfloat amb0[4] = { 0.02, 0.02, 0.02, 0.0 };
  GLfloat dir0[4] = { 0.15, 0.15, 0.15, 0.0 };
  GLfloat pos0[4] = { 0.3, 1.0, 0.55, 0.0 };
  GLfloat atn0[3] = { 1.0, 0.0, 0.0 };

  GLfloat amb2[4] = { 0.024, 0.018, 0.014, 0.0 };
/*  GLfloat dir2[4] = { 0.36, 0.30, 0.24, 0.0 };*/
  GLfloat dir2[4] = { 0.30, 0.30, 0.30, 0.0 };
  GLfloat pos2[4] = { -2.95, -2.8, 1.0, 1.0 };
  GLfloat atn2[3] = { 0.3, 0.1, 0.4 };

  GLfloat csc[3] = { 0.0, 0.0, 0.0 };
#define RS 6.5
  LightBl *light;

  light = &ad->light;

  ConstructShadowTxtFBO ( light, 0, false );
  SetLightSurfBit ( light, 0, false );
  SetLightAmbient ( light, 0, amb0 );
  SetLightDiffuse ( light, 0, dir0 );
  SetLightPosition ( light, 0, pos0 );
  SetLightAttenuation ( light, 0, atn0 );
  SetLightOnOff ( light, 0, true );
  SetupShadowTxtTransformations ( light, 0, csc, RS );
  UpdateShadowMatrix ( light, 0 );

  ConstructShadowTxtFBO ( light, 1, true );
  ad->mainlamp_on = false;
  ToggleLampCandle ( ad );
  SetLightOnOff ( light, 1, true );

  ConstructShadowTxtFBO ( light, 2, true );
  SetLightSurfBit ( light, 2, false );
  SetLightAmbient ( light, 2, amb2 );
  SetLightDiffuse ( light, 2, dir2 );
  SetLightPosition ( light, 2, pos2 );
  SetLightAttenuation ( light, 2, atn2 );
  SetLightOnOff ( light, 2, true );
  SetupShadowTxtTransformations ( light, 2, csc, RS );

  SetLightMaxDepth ( light, RS );
} /*InitLights*/

static void _ResizeMyWorld ( AppData *ad, int width, int height )
{
  float  lr, z;
  Camera *cam;

  cam = &ad->camera;
  LoadViewport ( &ad->trans, 0, 0,
                 cam->win_width = width, cam->win_height = height );
  z = cam->zoom * 0.05/*0.05533*/;
  lr = z*(float)width/(float)height;  /* przyjmujemy aspekt rowny 1 */
  M4x4Frustumf ( ad->trans.pm, NULL, cam->left = -lr, cam->right = lr,
      cam->bottom = -z, cam->top = z, cam->near = 0.25, cam->far = 16.0 );
  LoadVPMatrix ( &ad->trans );
  ResizeDFBO ( &ad->dfb, width, height );
  ExitIfGLError ( "_ResizeMyWorld" );
} /*_ResizeMyWorld*/

void PrintGLInfo ( void )
{
/*
  GLint i, n;

  glGetIntegerv ( GL_MAX_COLOR_ATTACHMENTS, &n );
  printf ( "max colour attachments = %d\n", n );
  glGetIntegerv ( GL_MAX_VIEWPORTS, &n );
  printf ( "max viewports = %d\n", n );
  glGetIntegerv ( GL_MAX_UNIFORM_BUFFER_BINDINGS, &n );
  printf ( "max uniform buffer bindings = %d\n", n );
  glGetIntegerv ( GL_MAX_SHADER_STORAGE_BUFFER_BINDINGS, &n );
  printf ( "max storage buffer bindings = %d\n", n );
  glGetIntegerv ( GL_MAX_COMBINED_SHADER_STORAGE_BLOCKS, &n );
  printf ( "max combined storage blocks = %d\n", n );
  glGetIntegerv ( GL_MAX_FRAGMENT_SHADER_STORAGE_BLOCKS, &n );
  printf ( "max fragment shader storage blocks = %d\n", n );
  glGetIntegerv ( GL_MAX_SHADER_STORAGE_BUFFER_BINDINGS, &n );
  printf ( "max shader storage buffer bindings = %d\n", n );
  glGetIntegerv ( GL_MAX_VERTEX_SHADER_STORAGE_BLOCKS, &n );
  printf ( "max vertex shader storage blocks = %d\n", n );
  glGetIntegerv ( GL_MAX_GEOMETRY_SHADER_STORAGE_BLOCKS, &n );
  printf ( "max geometry shader storage blocks = %d\n", n );
  glGetIntegerv ( GL_MAX_COMPUTE_SHADER_STORAGE_BLOCKS, &n );
  printf ( "max compute shader storage blocks = %d\n", n );
  printf ( "max compute work group count = " );
  for ( i = 0; i < 3; i++ ) {
    glGetIntegeri_v ( GL_MAX_COMPUTE_WORK_GROUP_COUNT, i, &n );
    printf ( "%d, ", n );
  }
  printf ( "\n" );
  glGetIntegerv ( GL_MAX_COMPUTE_WORK_GROUP_INVOCATIONS, &n );
  printf ( "max compute work group invocations = %d\n", n );
  glGetIntegerv ( GL_MAX_COMPUTE_SHARED_MEMORY_SIZE, &n );
  printf ( "max compute shared memory size = %d\n", n );
  glGetIntegerv ( GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS, &n );
  printf ( "max combined texture image units = %d\n", n );
*/
} /*PrintGLInfo*/

void InitMyWorld ( int argc, char *argv[], int width, int height )
{
  int nobj;

  memset ( &appdata, 0, sizeof(AppData) );
  PrintGLVersion ();
  PrintGLInfo ();
  TimerInit ();
  ConstructEmptyVAO ();
  LoadMyShaders ( &appdata );
  LoadGPUSMultMVfShader ();
  LoadBalanceShaders ();
  SetupDFBO ( &appdata.dfb, width, height );
  appdata.trans.trbuf = NewUniformTransBlock ();
  appdata.light.lsbuf = NewUniformLightBlock ();
  appdata.mat.matbuf = NewUniformMatBlock ();
  SetupModelMatrix ( &appdata );
  InitViewMatrix ( &appdata );
  nobj = 0;
  BeginEnteringBalanceElements ( &appdata.belem, FFNEAR, FFFAR );
  EnterRoomWalls ( &appdata.belem, &appdata.obj[nobj++], &appdata.mat );
  EnterDoor ( &appdata.belem, &appdata.obj[nobj], &appdata.mat );
  SetDoorAperture ( &appdata.obj[nobj++], 0.3*PI );
  EnterCarpet ( &appdata.belem, &appdata.obj[nobj++], &appdata.mat );
  EnterChair ( &appdata.belem, &appdata.obj[nobj], &appdata.mat );
  EnterObjectCopy ( &appdata.belem, &appdata.obj[nobj], &appdata.obj[nobj+1] );
  EnterObjectCopy ( &appdata.belem, &appdata.obj[nobj], &appdata.obj[nobj+2] );
  SetChairPosition ( &appdata.obj[nobj++], -1.7, 0.23, -0.15*PI );
  SetChairPosition ( &appdata.obj[nobj++], 0.3, -1.2, 0.55*PI );
  SetChairPosition ( &appdata.obj[nobj++], 0.1, 0.95, -0.6*PI );
  EnterTable ( &appdata.belem, &appdata.obj[nobj++], &appdata.mat );
  EnterBookcase ( &appdata.belem, &appdata.obj[nobj], &appdata.mat );
  SetBookcasePosition ( &appdata.obj[nobj++], 2.98, 1.0, PI );
  EnterGalleon ( &appdata.belem, &appdata.obj[nobj], &appdata.mat );
  SetGalleonPosition ( &appdata.obj[nobj++], -0.4, 0.0,
                       FLOOR_Z+TABLE_HEIGHT+0.28 );
  EnterCandle ( &appdata.belem, &appdata.obj[nobj], &appdata.mat );
  SetCandlePosition ( &appdata.obj[nobj++],
                      0.3, 0.25, FLOOR_Z+TABLE_HEIGHT+0.001 );
  EnterMainLamp ( &appdata.belem, &appdata.obj[nobj++], &appdata.mat );
  EnterCandleFlame ( &appdata.belem, &appdata.obj[nobj], &appdata.mat );
  SetCandlePosition ( &appdata.obj[nobj++],
                      0.3, 0.25, FLOOR_Z+TABLE_HEIGHT+0.001 );
  BindMaterialTextures ( &appdata.mat );
  EndEnteringBalanceElements ( &appdata.belem );
  appdata.niter = 10;
/*appdata.debug_t = appdata.debug_t1 = -1;*/
  appdata.debug_ff = 0;

  appdata.nobjects = nobj;
  InitLights ( &appdata );
  appdata.camera.zoom = 1.0;
  appdata.camera.zoomexp = 0;
  appdata.glowfct = 1.0;
  appdata.ngf = 0;
  appdata.ssao_radius = 0.15;
  _ResizeMyWorld ( &appdata, width, height );
  ExitIfGLError ( "InitMyWorld" );
} /*InitMyObject*/

void ResizeMyWorld ( int width, int height )
{
  _ResizeMyWorld ( &appdata, width, height );
} /*ResizeMyWorld*/

void RedrawMyWorld ( void )
{
  _RedrawMyWorld ( &appdata );
} /*Redraw*/

static char title0[] = "Cienie w scenie";
static char title1[] = "deferred";
static char title2[] = "glow";
static char title3[] = "SSAO";
static char title4[] = "triangle class";
static char title5[] = "triangle patches";
static char title6[] = "elements";
static char title7[] = "central points";
static char title8[] = "macroelements";
static char title9[] = "Lambertian radiance";

char ChangeRenderingMethod ( AppData *ad, char option, char *title )
{
  if ( option == ad->render_option )
    return false;
  if ( (appdata.render_option = option) == ROPT_RADIANCE ) {
  }
  else {
  }
  ProcessWorldRequest ( WMSG_SET_TITLE, title, NULL );
  return true;
} /*ChangeRenderingMethod*/

char ProcessCharCommand ( char charcode )
{
  switch ( toupper ( charcode ) ) {
case 'L':
    ToggleLampCandle ( &appdata );
    return true;
case '0':
    return ChangeRenderingMethod ( &appdata, ROPT_DIRECT, title0 );
case '1':
    return ChangeRenderingMethod ( &appdata, ROPT_DEFERRED1, title1 );
case '2':
    return ChangeRenderingMethod ( &appdata, ROPT_DEFERRED2, title2 );
case '3':
    return ChangeRenderingMethod ( &appdata, ROPT_DEFERRED3, title3 );
case '4':
    return ChangeRenderingMethod ( &appdata, ROPT_TRCLASS1, title4 );
case '5':
    return ChangeRenderingMethod ( &appdata, ROPT_TRCLASS2, title5 );
case '6':
    return ChangeRenderingMethod ( &appdata, ROPT_TRCLASS3, title6 );
case '7':
    return ChangeRenderingMethod ( &appdata, ROPT_ELEMCP, title7 );
case '8':
    return ChangeRenderingMethod ( &appdata, ROPT_MACROEL, title8 );
case '9':
    return ChangeRenderingMethod ( &appdata, ROPT_RADIANCE, title9 );
case 'R':
    InitViewMatrix ( &appdata );
    return true;
case 'V':
    if ( (appdata.vertical = !appdata.vertical) ) {
      Verticalise ( appdata.camera.viewer_rvec, &appdata.camera.viewer_rangle );
      LoadViewMatrix ( &appdata );
      return true;
    }
    else
      return false;
case '+':
    switch ( appdata.render_option ) {
/*  case ROPT_DEFERRED1:
      appdata.debug_t1 ++;
      printf ( "debug_t1 = %d\n", appdata.debug_t1 );
      return true;*/
  case ROPT_DEFERRED2:
      if ( appdata.ngf < 12 ) {
        appdata.ngf ++;
        return true;
      }
      else
        return false;
  case ROPT_DEFERRED3:
      if ( appdata.ssao_radius < 1.0 ) {
        appdata.ssao_radius += 0.01;
        printf ( "R = %4.2f\n", appdata.ssao_radius );
        return true;
      }
      else
        return false;
/*  case ROPT_TRCLASS1:
      appdata.debug_t ++;
      printf ( "debug_t = %d\n", appdata.debug_t );
      return true;*/
  case ROPT_RADIANCE:
      if ( appdata.niter < 30 ) {
        appdata.niter ++;
        appdata.radiance_ok = false;
        printf ( "niter = %d\n", appdata.niter );
        return true;
      }
      else
        return false;
  default:
      return false;
    }
case '-':
    switch ( appdata.render_option ) {
/*  case ROPT_DEFERRED1:
      if ( appdata.debug_t1 >= 0 ) {
        appdata.debug_t1 --;
        printf ( "debug_t1 = %d\n", appdata.debug_t1 );
        return true;
      }
      else
        return false;*/
  case ROPT_DEFERRED2:
      if ( appdata.ngf >= 0 ) {
        appdata.ngf --;
        return true;
      }
      else
        return false;
  case ROPT_DEFERRED3:
      if ( appdata.ssao_radius >= 0.02 ) {
        appdata.ssao_radius -= 0.01;
        printf ( "R = %4.2f\n", appdata.ssao_radius );
        return true;
      }
      else
        return false;
/*  case ROPT_TRCLASS1:
      if ( appdata.debug_t >= 0 ) {
        appdata.debug_t --;
        printf ( "debug_t = %d\n", appdata.debug_t );
      }
      return true;*/
  case ROPT_RADIANCE:
      if ( appdata.niter > 0 ) {
        appdata.niter --;
        appdata.radiance_ok = false;
        printf ( "niter = %d\n", appdata.niter );
        return true;
      }
      else
        return false;
  default:
      return false;
    }
case '<':
    switch ( appdata.render_option ) {
  case ROPT_DEFERRED2:
  case ROPT_DEFERRED3:
      appdata.glowfct *= sqrt ( sqrt ( 0.5 ) );
      return true;
  default:
      return false;
    }
case '>':
    switch ( appdata.render_option ) {
  case ROPT_DEFERRED2:
  case ROPT_DEFERRED3:
      appdata.glowfct *= sqrt ( sqrt ( 2.0 ) );
      return true;
  default:
      return false;
    }
case 'H':
    appdata.obscgl = !appdata.obscgl;
    return true;
case 'C':
    appdata.debug = !appdata.debug;
    return true;
case 'T':
    appdata.debug_ff = !appdata.debug_ff;
    return true;
default:
    return false;
  }
} /*ProcessCharCommand*/

char ChangeZoom ( char inczoom )
{
  int    e;
  double z;

  e = appdata.camera.zoomexp;
  if ( inczoom ) {
    if ( (z = pow ( ZOOM_FCT, (double)(++e) )) > MAX_ZOOM )
      return false;
  }
  else {
    if ( (z = pow ( ZOOM_FCT, (double)(--e) )) < MIN_ZOOM )
      return false;
  }
  appdata.camera.zoomexp = e;  appdata.camera.zoom = z;
  _ResizeMyWorld ( &appdata,
                   appdata.camera.win_width, appdata.camera.win_height );
  return true;
} /*ChangeZoom*/

void DeleteMyWorld ( void )
{
  int i;

  for ( i = 0; i < NPROGRAMS; i++ )
    glDeleteProgram ( appdata.program_id[i] );
  for ( i = 0; i < appdata.nobjects; i++ )
    appdata.obj[i].destroy ( &appdata.obj[i] );
  DeleteGPUSMultMVfProgram ();
  glDeleteBuffers ( 1, &appdata.trans.trbuf );
  glDeleteBuffers ( 1, &appdata.light.lsbuf );
  DestroyShadowFBO ( &appdata.light );
  DeleteMaterials ( &appdata.mat );
  DeleteDFBO ( &appdata.dfb );
  DeleteBalanceElements ( &appdata.belem );
  DeleteBalancePrograms ();
  DeleteEmptyVAO ();
#ifdef DEBUG_BUFFERS_ALLOCATION
  DumpBufferIdentifiers ();
#endif
#ifdef DEBUG_TEXTURES_ALLOCATION
  DumpTextureIdentifiers ();
#endif
  ExitIfGLError ( "DeleteMyWorld" );
} /*DeleteMyWorld*/

