
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>

#include "openglheader.h"

#include "utilities.h"
#include "meshes.h"
#include "meshespriv.h"

#define NMBOFFS 9

/* ////////////////////////////////////////////////////////////////////////// */
static GLuint bpoint = GL_INVALID_INDEX;
static GLint  mbsize, mbofs[NMBOFFS];

GLuint GetAccessToMeshSurfBlock ( GLuint program_id )
{
  static const GLchar *names[] =
    { "meshsurf", "nv", "nhe", "nfac", "nsattr", "pdim",
      "pofs", "nvofs", "MeshNormals", "Colour" };

  if ( bpoint == GL_INVALID_INDEX )
    GetAccessToStorageBlock ( program_id, NMBOFFS, names,
                              &mbsize, mbofs, &bpoint );
  ExitIfGLError ( "GetAccessToMeshSurfBlock" );
  return bpoint;
} /*GetAccessToMeshSurfBlock*/

void UploadMeshParams ( GPUmesh *gmesh )
{
  glBindBuffer ( SSB, gmesh->MSBUF );
  glBufferSubData ( SSB, mbofs[0], sizeof(GLint), &gmesh->nv );
  glBufferSubData ( SSB, mbofs[1], sizeof(GLint), &gmesh->nhe );
  glBufferSubData ( SSB, mbofs[2], sizeof(GLint), &gmesh->nfac );
  glBufferSubData ( SSB, mbofs[3], sizeof(GLint), &gmesh->nsattr );
  glBufferSubData ( SSB, mbofs[4], sizeof(GLint), &gmesh->pdim );
  glBufferSubData ( SSB, mbofs[5], sizeof(GLint), &gmesh->pofs );
  glBufferSubData ( SSB, mbofs[6], sizeof(GLint), &gmesh->nvofs );
  ExitIfGLError ( "UploadMeshParams" );
} /*UploadMeshParams*/

void SetMeshColour ( GPUmesh *gmesh, GLfloat Colour[3] )
{
  glBindBuffer ( SSB, gmesh->MSBUF );
  glBufferSubData ( SSB, mbofs[8], 3*sizeof(GLfloat), Colour );
  ExitIfGLError ( "SetMeshColour" );
} /*SetMeshColour*/

void SetMeshNVS ( GPUmesh *gmesh, GLint nvs )
{
  glBindBuffer ( SSB, gmesh->MSBUF );
  glBufferSubData ( SSB, mbofs[7], sizeof(GLint), &nvs );
  ExitIfGLError ( "SetMeshNVS" );
} /*SetMeshNVS*/

char ReallocGPUmesh ( GPUmesh *gmesh, int nv, int nhe, int nfac, int nsattr,
                      int pdim, int pofs, int nvofs )
{
  int i;

  for ( i = 0; i < 4; i++ )
    if ( gmesh->mbuf[i] > 0 ) glDeleteBuffers ( 1, &gmesh->mbuf[i] );
  glGenBuffers ( 4, gmesh->mbuf );
  glBindBuffer ( SSB, gmesh->MVFBUF );
  glBufferData ( SSB, (nv+nfac+2*nhe)*sizeof(GLuint), NULL, GL_DYNAMIC_DRAW );
  glBindBuffer ( SSB, gmesh->MHEBUF );
  glBufferData ( SSB, nhe*4*sizeof(GLint), NULL, GL_DYNAMIC_DRAW );
  glBindBuffer ( SSB, gmesh->VCBUF );
  glBufferData ( SSB, nv*nsattr*sizeof(GLfloat), NULL, GL_DYNAMIC_DRAW );
  glBindBuffer ( SSB, gmesh->MSBUF );
  glBufferData ( SSB, mbsize, NULL, GL_DYNAMIC_DRAW );
  gmesh->nsattr = nsattr;  gmesh->pdim = pdim;  gmesh->pofs = pofs;
  gmesh->nvofs = nvofs;
  gmesh->nv = nv;
  gmesh->nhe = nhe;
  gmesh->nfac = nfac;
  UploadMeshParams ( gmesh );
  ExitIfGLError ( "ReallocGPUmesh" );
  return true;
} /*ReallocGPUmesh*/

void FreeCPUmeshTab ( CPUmesh *cmesh )
{
  if ( cmesh->mv ) free ( cmesh->mv );
  if ( cmesh->vc ) free ( cmesh->vc );
  if ( cmesh->mhe ) free ( cmesh->mhe );
  if ( cmesh->mfac ) free ( cmesh->mfac );
  if ( cmesh->mvhei ) free ( cmesh->mvhei );
  if ( cmesh->mfhei ) free ( cmesh->mfhei );
  memset ( cmesh, 0, sizeof(CPUmesh) );
} /*FreeCPUmeshTab*/

char ReallocCPUmesh ( CPUmesh *cmesh, int nv, int nhe, int nfac, int nsattr,
                      int pdim, int pofs, int nvofs )
{
  FreeCPUmeshTab ( cmesh );
  cmesh->mv = malloc ( nv*sizeof(BSMvertex) );
  cmesh->vc = malloc ( nv*nsattr*sizeof(double) );
  cmesh->mhe = malloc ( nhe*sizeof(BSMhalfedge) );
  cmesh->mfac = malloc ( nfac*sizeof(BSMfacet) );
  cmesh->mvhei = malloc ( nhe*sizeof(int) );
  cmesh->mfhei = malloc ( nhe*sizeof(int) );
  if ( !cmesh->mv | !cmesh->vc | !cmesh->mhe | !cmesh->mfac |
       !cmesh->mvhei | !cmesh->mfhei ) {
    FreeCPUmeshTab ( cmesh );
    return false;
  }
  else {
    cmesh->nsattr = nsattr;  cmesh->pdim = pdim;  cmesh->pofs = pofs;
    cmesh->nvofs = nvofs;
    cmesh->nv = nv;
    cmesh->nhe = nhe;
    cmesh->nfac = nfac;

memset ( cmesh->mv, 0, nv*sizeof(BSMvertex) );
memset ( cmesh->vc, 0, nv*nsattr*sizeof(double) );
memset ( cmesh->mhe, 0, nhe*sizeof(BSMhalfedge) );
memset ( cmesh->mfac, 0, nfac*sizeof(BSMfacet) );
memset ( cmesh->mvhei, 0, nhe*sizeof(int) );
memset ( cmesh->mfhei, 0, nhe*sizeof(int) );

    return true;
  }
} /*ReallocCPUmesh*/

char CPUmeshToGPU ( CPUmesh *cmesh, GPUmesh *gmesh )
{
  int       dim, nv, nhe, nfac;
  BSMvertex *mv;
  BSMfacet  *mfac;
  GLuint    *vf;
  GLfloat   *vc;
  int       i, size;

  if ( ReallocGPUmesh ( gmesh, nv = cmesh->nv, nhe = cmesh->nhe,
                        nfac = cmesh->nfac, dim = cmesh->nsattr, cmesh->pdim,
                        cmesh->pofs, cmesh->nvofs ) ) {
    size = (nv > nfac ? nv : nfac)*sizeof(GLuint);
    i = nv*dim*sizeof(GLfloat);
    if ( size < i ) size = i;
    if ( !(vf = (GLuint*)malloc ( size )) )
      goto failure;
    glBindBuffer ( SSB, gmesh->MVFBUF );
    mv = cmesh->mv;  mfac = cmesh->mfac;
    for ( i = 0; i < nv; i++ )
      vf[i] = mv[i].firsthalfedge + (mv[i].degree << DEGSHIFT);
    glBufferSubData ( SSB, 0, nv*sizeof(GLuint), vf );
    for ( i = 0; i < nfac; i++ )
      vf[i] = mfac[i].firsthalfedge + (mfac[i].degree << DEGSHIFT);
    glBufferSubData ( SSB, nv*sizeof(GLuint), nfac*sizeof(GLuint), vf );
    glBufferSubData ( SSB, (nv+nfac)*sizeof(GLuint), nhe*sizeof(GLint),
                      cmesh->mvhei );
    glBufferSubData ( SSB, (nv+nfac+nhe)*sizeof(GLuint), nhe*sizeof(GLint),
                      cmesh->mfhei );
    glBindBuffer ( SSB, gmesh->MHEBUF );
    glBufferSubData ( SSB, 0, nhe*4*sizeof(GLint), cmesh->mhe );
    vc = (GLfloat*)vf;
    for ( i = 0; i < nv*dim; i++ )
      vc[i] = (GLfloat)cmesh->vc[i];
    glBindBuffer ( SSB, gmesh->VCBUF );
    glBufferSubData ( SSB, 0, nv*dim*sizeof(GLfloat), vc );
    free ( vf );
    ExitIfGLError ( "CPUmeshToGPU" );
    return true;
  }
failure:
  return false;
} /*CPUmeshToGPU*/

char GPUmeshToCPU ( GPUmesh *gmesh, CPUmesh *cmesh )
{
  int       dim, nv, nhe, nfac;
  BSMvertex *mv;
  BSMfacet  *mfac;
  GLuint    *vf;
  GLfloat   *vc;
  int       i, size;
  
  if ( ReallocCPUmesh ( cmesh, nv = gmesh->nv, nhe = gmesh->nhe,
                        nfac = gmesh->nfac, dim = gmesh->nsattr, gmesh->pdim,
                        gmesh->pofs, gmesh->nvofs ) ) {
    size = (nv > nfac ? nv : nfac)*sizeof(GLuint);
    i = nv*dim*sizeof(GLfloat);
    if ( size < i ) size = i;
    if ( !(vf = (GLuint*)malloc ( size )) )
      goto failure;
    mv = cmesh->mv;  mfac = cmesh->mfac;
    glBindBuffer ( SSB, gmesh->MVFBUF );
    glGetBufferSubData ( SSB, 0, nv*sizeof(GLuint), vf );
    for ( i = 0; i < nv; i++ ) {
      mv[i].firsthalfedge = vf[i] & FHEMASK;
      mv[i].degree = vf[i] >> DEGSHIFT;
    }
    glGetBufferSubData ( SSB, nv*sizeof(GLint), nfac*sizeof(GLuint), vf );
    for ( i = 0; i < nfac; i++ ) {
      mfac[i].firsthalfedge = vf[i] & FHEMASK;
      mfac[i].degree = vf[i] >> DEGSHIFT;
    }
    glGetBufferSubData ( SSB, (nv+nfac)*sizeof(GLint), nhe*sizeof(GLint),
                         cmesh->mvhei );
    glGetBufferSubData ( SSB, (nv+nfac+nhe)*sizeof(GLint), nhe*sizeof(GLint),
                         cmesh->mfhei );
    glBindBuffer ( SSB, gmesh->MHEBUF );
    glGetBufferSubData ( SSB, 0, nhe*4*sizeof(GLint), cmesh->mhe );
    vc = (GLfloat*)vf;
    glBindBuffer ( SSB, gmesh->VCBUF );
    glGetBufferSubData ( SSB, 0, nv*dim*sizeof(GLfloat), vc );
    for ( i = 0; i < nv*dim; i++ )
      cmesh->vc[i] = (double)vc[i];
    free ( vf );
    ExitIfGLError ( "GPUmeshToCPU" );
    return true;
  }
failure:
  return false;
} /*GPUmeshToCPU*/

void DeleteCPUmesh ( CPUmesh *cmesh )
{
  if ( cmesh->mv ) free ( cmesh->mv );
  if ( cmesh->mhe ) free ( cmesh->mhe );
  if ( cmesh->mfac ) free ( cmesh->mfac );
  if ( cmesh->mvhei ) free ( cmesh->mvhei );
  if ( cmesh->mfhei ) free ( cmesh->mfhei );
  if ( cmesh->vc ) free ( cmesh->vc );
  free ( cmesh );
} /*DeleteCPUmesh*/

void DeleteGPUmesh ( GPUmesh *gmesh )
{
  glDeleteBuffers ( 4, gmesh->mbuf );
  free ( gmesh );
} /*DeleteGPUmesh*/

