#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "app2k.h"
#include "../utilities/utilities.h"
#include "../utilities/linkage.h"
#include "../utilities/bezpatches.h"
#include "../utilities/teapot.h"
#include "accumbuf.h"
#include "texture.h"
#include "trans.h"
#include "lights.h"
#include "app2kstruct.h"
#include "app2kproc.h"

static AppData appdata;

void LoadBPShaders ( BPRenderPrograms *brprog )
{
  static const char *filename[] =
    { "app2g0.vert.glsl", "app2g0.tesc.glsl", "app2j0.tese.glsl",
      "app2j0.geom.glsl", "app2k1.frag.glsl", "app2g1.vert.glsl",
      "app2g1.geom.glsl", "app2g1.frag.glsl", "app2g3.tese.glsl",
      "app2g3.frag.glsl" };
  static const GLuint shtype[] =
    { GL_VERTEX_SHADER, GL_TESS_CONTROL_SHADER, GL_TESS_EVALUATION_SHADER,
      GL_GEOMETRY_SHADER, GL_FRAGMENT_SHADER, GL_VERTEX_SHADER,
      GL_GEOMETRY_SHADER, GL_FRAGMENT_SHADER, GL_TESS_EVALUATION_SHADER,
      GL_FRAGMENT_SHADER };
  static const GLchar *UVarNames[] =
    { "ColourSource", "LightingModel", "NormalSource", "ModifyDepth" };
  GLuint shader_id[10], shid[4];
  int    i;

  for ( i = 0; i < 10; i++ )
    shader_id[i] = CompileShaderFiles ( shtype[i], 1, &filename[i] );
  shid[0] = shader_id[0];  shid[1] = shader_id[1];
  shid[2] = shader_id[8];  shid[3] = shader_id[9];
  brprog->program_id[0] = LinkShaderProgram ( 4, shid, "0" );
  brprog->program_id[1] = LinkShaderProgram ( 5, shader_id, "1" );
  brprog->program_id[2] = LinkShaderProgram ( 3, &shader_id[5], "2" );
  GetAccessToTransBlockUniform ( brprog->program_id[1] );
  GetAccessToLightMatUniformBlocks ( brprog->program_id[1] );
  GetAccessToBezPatchStorageBlocks ( brprog->program_id[1], true, false );
  brprog->ColourSourceLoc =
      glGetUniformLocation ( brprog->program_id[1], UVarNames[0] );
  brprog->LightingModelLoc =
      glGetUniformLocation ( brprog->program_id[1], UVarNames[1] );
  brprog->NormalSourceLoc =
      glGetUniformLocation ( brprog->program_id[1], UVarNames[2] );
  brprog->ModifyDepthLoc =
      glGetUniformLocation ( brprog->program_id[1], UVarNames[3] );
  AttachUniformTransBlockToBP ( brprog->program_id[0] );
  AttachUniformTransBlockToBP ( brprog->program_id[2] );
  for ( i = 0; i < 10; i++ )
    glDeleteShader ( shader_id[i] );
  ExitIfGLError ( "LoadBPShaders" );
} /*LoadBPShaders*/

static void ComputeEyePosition ( Camera *camera, TransBl *trans )
{
  M4x4RotateVfv ( trans->wvm0, camera->viewer_rvec,
                  -camera->viewer_rangle );
  M4x4MultMTVf ( trans->eyepos0, trans->wvm0, camera->viewer_pos0 );
  M4x4InvTranslateMfv ( trans->wvm0, camera->viewer_pos0 );
} /*ComputeEyePosition*/
 
void _RotateViewer ( AppData *ad, double delta_xi, double delta_eta )
{
  float   vi[3], lgt, vk[3];
  double  angi, angk;

  vi[0] = (float)delta_eta*ad->camera.tb/(float)ad->camera.win_height;
  vi[1] = (float)delta_xi*ad->camera.rl/(float)ad->camera.win_width;
  vi[2] = 0.0;
  lgt = sqrt ( V3DotProductf ( vi, vi ) );
  vi[0] /= lgt;  vi[1] /= lgt;
  angi = -0.052359878;  /* -3 stopnie */
  V3CompRotationsf ( vk, &angk, ad->camera.viewer_rvec,
                     ad->camera.viewer_rangle, vi, angi );
  memcpy ( ad->camera.viewer_rvec, vk, 3*sizeof(float) );
  ad->camera.viewer_rangle = angk;
  ComputeEyePosition ( &ad->camera, &ad->trans );
} /*_RotateViewer*/

void RotateViewer ( double delta_xi, double delta_eta )
{
  if ( delta_xi == 0.0 && delta_eta == 0.0 )
    return;  /* natychmiast uciekamy - nie chcemy dzielic przez 0 */
  _RotateViewer ( &appdata, (float)delta_xi, (float)delta_eta );
} /*RotateViewer*/

void InitLights ( AppData *ad )
{
  GLfloat amb0[3] = { 0.2, 0.2, 0.3 };
  GLfloat dir0[3] = { 0.8, 0.8, 0.8 };
  GLfloat pos0[4] = { -0.2, 1.0, 1.0, 0.0 };
  GLfloat atn0[3] = { 1.0, 0.0, 0.0 };
  GLfloat csc[3]  = { 0.0, 0.0, 0.0 };

  SetLightAmbient ( &ad->light, 0, amb0 );
  SetLightDirect ( &ad->light, 0, dir0 );
  SetLightPosition ( &ad->light, 0, pos0 );
  SetLightAttenuation ( &ad->light, 0, atn0 );
  SetLightOnOff ( &ad->light, 0, 1 );
  ConstructShadowTxtFBO ( &ad->light, 0 );
  SetupShadowTxtTransformations ( &ad->light, 0, csc, 2.2 );
  UpdateShadowMatrix ( &ad->light, 0 );
} /*InitLights*/

static void _ResizeMyWorld ( AppData *ad, int width, int height )
{
  SetAccumBufSize ( ad->acb, ad->camera.win_width = width,
                    ad->camera.win_height = height );
  if ( ad->particles )
    ResetParticles ( &ad->psprog, &ad->ps, app_time );
} /*_ResizeMyWorld*/

void ResizeMyWorld ( int width, int height )
{
  _ResizeMyWorld ( &appdata, width, height );
} /*ResizeMyWorld*/

void InitCamera ( AppData *ad, int width, int height )
{
  static const float viewer_rvec[3] = {1.0,0.0,0.0};
  static const float viewer_pos0[4] = {0.0,0.0,10.0,1.0};

  memcpy ( ad->camera.viewer_rvec, viewer_rvec, 3*sizeof(float) );
  memcpy ( ad->camera.viewer_pos0, viewer_pos0, 4*sizeof(float) );
  ad->camera.viewer_rangle = 0.0;
  ad->camera.distcnt = ad->camera.heightcnt = ad->camera.Ncnt = 0;
  ad->camera.frameheight = INITIAL_FRAMEHEIGHT;
  ad->camera.unitmm = (float)height/(SCREEN_DPMM*ad->camera.frameheight);
  ad->camera.dist = SCREEN_DIST;
  ad->camera.viewer_pos0[2] = ad->camera.dist / ad->camera.unitmm;
  ad->camera.N = INITIAL_APERTURE;
  memcpy ( ad->trans.eyepos0, ad->camera.viewer_pos0, 4*sizeof(GLfloat) );
  M4x4InvTranslatefv ( ad->trans.wvm0, ad->camera.viewer_pos0 );
  _ResizeMyWorld ( ad, width, height );
} /*InitCamera*/

float RadicalInverse ( unsigned int i, unsigned int base )
{
  float x, b, c;

  for ( x = 0.0, c = b = 1.0/(float)base;  i;  i /= base, c *= b )
    x += (float)(i % base) * c;
  return x;
} /*RadicalInverse*/

void InitMyWorld ( int argc, char *argv[], int width, int height )
{
  GLfloat ident_matrix[16];

  memset ( &appdata, 0, sizeof(AppData) );
  LoadAccumBufShaders ();
  appdata.acb = NewAccumBuf ( width, height );
  LoadBPShaders ( &appdata.brprog );
  LoadMirrorShaders ( appdata.miprog );
  LoadLinkageArticulationProgram ( &appdata.artprog );
  LoadParticleShaders ( &appdata.psprog );
  appdata.trans.trbuf = NewUniformTransBlock ();
  appdata.light.lsbuf = NewUniformLightBlock ();
  appdata.mat.matbuf = NewUniformMatBlock ();
  TimerInit ();
  M4x4Identf ( ident_matrix );
  LoadMMatrix ( &appdata.trans, ident_matrix );
  InitCamera ( &appdata, width, height );
  appdata.TessLevel = 10;
  appdata.BezNormals = GL_TRUE;
  appdata.hold_time = 0.0;
  appdata.cnet = appdata.skeleton = appdata.particles =
    appdata.animate = appdata.hold = appdata.stereo = false;
  appdata.shadows = true;
  ConstructMirror ( &appdata.mirror );
  InitLights ( &appdata );
  if ( !ConstructMyLinkage ( &appdata ) )
    ExitOnError ( "InitMyObject" );
  ArticulateMyLinkage ( appdata.linkage );
} /*InitMyWorld*/

void DrawScene ( AppData *ad, char final )
{
  DrawMyLinkage ( ad, final );
} /*DrawScene*/

void DrawSceneToMirror ( AppData *ad )
{
  glBindFramebuffer ( GL_DRAW_FRAMEBUFFER, ad->mirror.mirror_fbo );
  glViewport ( 0, 0, MIRRORTXT_W, MIRRORTXT_H );
  LoadVPMatrices ( &ad->trans, true );
  glClearColor ( 0.95, 0.95, 0.95, 1.0 );
  glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
  DrawScene ( ad, true );
  glFlush ();
} /*DrawSceneToMirror*/

void DrawSceneToShadows ( AppData *ad )
{
  int    l;
  GLuint mask;

  glClearColor ( 1.0, 1.0, 1.0, 1.0 );
  glViewport ( 0, 0, SHADOW_MAP_SIZE, SHADOW_MAP_SIZE );
  glEnable ( GL_POLYGON_OFFSET_FILL );
  glPolygonOffset ( 2.0f, 4.0f );
  for ( l = 0, mask = 0x00000001;  l < ad->light.nls;  l++, mask <<= 1 )
    if ( ad->light.shmask & mask ) {
      BindShadowTxtFBO ( &ad->trans, &ad->light, l );
      glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
      glColorMask ( GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE );
      LoadMMatrix ( &ad->trans, ad->mirror.mirror_matrix );
      DrawMirror ( &ad->mirror, ad->miprog[0], false );
      DrawScene ( ad, false );
    }
  glColorMask ( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );
  glBindFramebuffer ( GL_FRAMEBUFFER, 0 );
  glDisable ( GL_POLYGON_OFFSET_FILL );
  for ( l = 0, mask = 0x00000001;  l < ad->light.nls;  l++, mask <<= 1 )
    if ( ad->light.shmask & mask ) {
      glActiveTexture ( GL_TEXTURE2+l );
      glBindTexture ( GL_TEXTURE_2D, ad->light.ls[l].shadow_txt[0] );
      glActiveTexture ( GL_TEXTURE2+MAX_NLIGHTS+l );
      glBindTexture ( GL_TEXTURE_2D, ad->light.ls[l].shadow_txt[1] );
    }
} /*DrawSceneToShadows*/

static void SetTransformations ( Camera *camera, TransBl *trans,
                                 char stereo, int i )
{
  float   F, dist, rvmax, ri, tau, phii, xvi, yvi;
  float   l, r, b, t, es;
  GLfloat wvm[16], wpm[16];
  int     j, k;

  if ( stereo ) {
    k = NLAYERS/2;
    if ( i < k )
      { es = -camera->eyeshift;  j = i; }
    else
      { es = camera->eyeshift;   j = i-k; }
  }
  else {
    k = NLAYERS;
    j = i;
    es = 0.0;
  }
  F = camera->F/camera->unitmm;
  dist = camera->dist/camera->unitmm;
  rvmax = F*(dist-F)/(2.0*camera->N*dist);
  ri = rvmax * sqrt ( (float)(j+j+1)/(float)(k+k) );
  tau = 0.5*(sqrt ( 5.0 ) - 1.0);
  phii = (float)(j+j+1)*PI*tau*tau;
  xvi = ri*cos ( phii );
  yvi = ri*sin ( phii );
  M4x4SkewFrustumf ( camera->win_width, camera->win_height, 1.0,
        camera->F, dist, xvi + es, yvi, ((float)j+0.5)/k-0.5,
        RadicalInverse ( j, 2 ) + 0.5*(1.0-1.0/(float)k),
        camera->near, camera->far, &l, &r, &b, &t, trans->wvm0,
        wvm, trans->eyepos1[i], wpm, NULL );
  M4x4Multf ( trans->wvpm1[i], wpm, wvm );
} /*SetTransformations*/

static void AdjustDimensions ( Camera *camera, TransBl *trans )
{
  float   w, h, windiag;
  GLfloat wvm[16], wpm[16];

  w = (float)camera->win_width;  h = (float)camera->win_height;
  camera->unitmm = h/(SCREEN_DPMM*camera->frameheight);
  windiag = sqrt ( (float)(w*w + h*h) ) / SCREEN_DPMM;
  camera->F = SCREEN_DIST/windiag*sqrt ( 36.0*36.0+24.0*24.0 );
  camera->near = MIN_FOV_DIST / camera->unitmm;
  camera->far = MAX_FOV_DIST / camera->unitmm;
  if ( camera->far < 10.0 )
    camera->far = 10.0;
  M4x4SkewFrustumf ( camera->win_width, camera->win_height, 1.0, camera->F,
                     camera->dist / camera->unitmm,
                     0.0, 0.0, 0.0, 0.0, camera->near, camera->far,
                     &camera->left, &camera->right, &camera->bottom, &camera->top,
                     trans->wvm0, wvm, NULL, wpm, NULL );
  camera->rl = camera->right - camera->left;
  camera->tb = camera->top - camera->bottom;
  camera->eyeshift = 0.5*EYE_DIST/camera->unitmm;
  camera->viewer_pos0[2] = SCREEN_DIST / camera->unitmm;
  ComputeEyePosition ( camera, trans );
} /*AdjustDimensions*/

void DrawSceneToWindow ( AppData *ad )
{
  glViewport ( 0, 0, ad->camera.win_width, ad->camera.win_height );
  glBindFramebuffer ( GL_DRAW_FRAMEBUFFER, ad->acb->fbo[1] );
  LoadVPMatrices ( &ad->trans, false );
  glClearColor ( 1.0, 1.0, 1.0, 1.0 );
  glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
  LoadMMatrix ( &ad->trans, ad->mirror.mirror_matrix );
  DrawMirror ( &ad->mirror, ad->miprog[1], true );
  DrawScene ( ad, true );
  AccumBufAverage ( ad->acb, ad->stereo );
  glBindFramebuffer ( GL_READ_FRAMEBUFFER, ad->acb->fbo[0] );
  glBindFramebuffer ( GL_DRAW_FRAMEBUFFER, 0 );
  glBlitFramebuffer ( 0, 0, ad->camera.win_width, ad->camera.win_height,
                      0, 0, ad->camera.win_width, ad->camera.win_height,
                      GL_COLOR_BUFFER_BIT, GL_NEAREST );
  glFlush ();
  ExitIfGLError ( "DrawSceneToWindow" );
} /*DrawSceneToWindow*/

void RedrawMyWorld ( void )
{
  int i;

  if ( !appdata.hold )
    ArticulateMyLinkage ( appdata.linkage );
  glEnable ( GL_DEPTH_TEST );
  DrawSceneToShadows ( &appdata );
  AdjustDimensions ( &appdata.camera, &appdata.trans );
  for ( i = 0; i < NLAYERS; i++ ) {
    SetTransformations ( &appdata.camera, &appdata.trans, appdata.stereo, i );
    SetupMirrorVPMatrices ( appdata.trans.eyepos1[i], appdata.trans.reyepos[i],
                            appdata.trans.mvpm[i], appdata.camera.far );
  }
  DrawSceneToMirror ( &appdata );
  DrawSceneToWindow ( &appdata );
} /*RedrawMyWorld*/

void DeleteMyWorld ( void )
{
  int i;

  glUseProgram ( 0 );
  DeleteLinkageArticulationProgram ( &appdata.artprog );
  for ( i = 0; i < 3; i++ )
    glDeleteProgram ( appdata.brprog.program_id[i] );
  for ( i = 0; i < 2; i++ )
    glDeleteProgram ( appdata.miprog[i] );
  DeleteParticleShaders ( &appdata.psprog );
  DeleteAccumBufShaders ();
  glDeleteBuffers ( 1, &appdata.trans.trbuf );
  glDeleteBuffers ( 1, &appdata.light.lsbuf );
  glDeleteBuffers ( 1, &appdata.mat.matbuf );
  glDeleteBuffers ( 1, &appdata.lktrbuf );
  kl_DestroyLinkage ( appdata.linkage );
  DeleteMirror ( &appdata.mirror );
  DeleteShadowFBO ( &appdata.light );
  DeleteAccumBuf ( appdata.acb );
  DeleteEmptyVAO ();
#ifdef DEBUG_BUFFERS_ALLOCATION
  DumpBufferIdentifiers ();
#endif
} /*DeleteMyWorld*/

char ChangeFocus ( char incfd )
{
  int    e;
  double f;

  e = appdata.camera.distcnt;
  if ( incfd ) {
    if ( (f = SCREEN_DIST*pow ( FOCUS_FCT, (double)(--e) )) < MIN_FOV_DIST )
      return false;
  }
  else {
    if ( (f = SCREEN_DIST*pow ( FOCUS_FCT, (double)(++e) )) > MAX_FOV_DIST )
      return false;
  }
  appdata.camera.distcnt = e;  appdata.camera.dist = f;
  return true;
} /*ChangeFocus*/

char ChangeZoom ( char inczoom )
{
  int    e;
  double z;

  e = appdata.camera.heightcnt;
  if ( inczoom ) {
    if ( (z = INITIAL_FRAMEHEIGHT*pow ( ZOOM_FCT, (double)(--e) )) < MIN_FRAME_HEIGHT )
      return false;
  }
  else {
    if ( (z = INITIAL_FRAMEHEIGHT*pow ( ZOOM_FCT, (double)(++e) )) > MAX_FRAME_HEIGHT )
      return false;
  }
  appdata.camera.heightcnt = e;  appdata.camera.frameheight = z;
  return true;
} /*ChangeZoom*/

char ChangeAperture ( char incapr )
{
  int    e;
  double a;

  e = appdata.camera.Ncnt;
  if ( incapr ) {
    if ( (a = INITIAL_APERTURE*pow ( 2.0, 0.5*(double)(++e) )) > MAX_APERTURE )
      return false;
  }
  else {
    if ( (a = INITIAL_APERTURE*pow ( 2.0, 0.5*(double)(--e) )) < MIN_APERTURE )
      return false;
  }
  appdata.camera.Ncnt = e;  appdata.camera.N = a;
  return true;
} /*ChangeAperture*/

char HoldOnOff ( AppData *ad )
{
  if ( (ad->hold = !ad->hold) ) {
    TimerToc ();
    ad->hold_time0 = app_time;
    return false;
  }
  else {
    TimerTic ();
    ad->hold_time += app_time - ad->hold_time0;
    return true;
  }
} /*HoldOnOff*/

char ProcessCharCommand ( char charcode )
{
  switch ( toupper ( charcode ) ) {
case '+':
    if ( appdata.TessLevel < MAX_TESS_LEVEL ) {
      SetBezierPatchTessLevel ( appdata.bezp[0].bpatches[0], ++appdata.TessLevel );
      SetBezierPatchTessLevel ( appdata.bezp[1].bpatches[0], appdata.TessLevel );
      return true;
    }
    else return false;
case '-':
    if ( appdata.TessLevel > MIN_TESS_LEVEL ) {
      SetBezierPatchTessLevel ( appdata.bezp[0].bpatches[0], --appdata.TessLevel );
      SetBezierPatchTessLevel ( appdata.bezp[1].bpatches[0], appdata.TessLevel );
      return true;
    }
    else return false;
case 'N':
    appdata.BezNormals = appdata.BezNormals == 0;
    SetBezierPatchNVS ( appdata.bezp[0].bpatches[0], appdata.BezNormals );
    SetBezierPatchNVS ( appdata.bezp[1].bpatches[0], appdata.BezNormals );
    return true;
case 'C':
    appdata.cnet = !appdata.cnet;
    return true;
case 'S':
    appdata.skeleton = !appdata.skeleton;
    return true;
case 'B':
    glUseProgram ( appdata.brprog.program_id[1] );
    glUniform1i ( appdata.brprog.LightingModelLoc,
          appdata.brprog.lighting_model = !appdata.brprog.lighting_model );
    return true;
case ' ':
    if ( (appdata.animate = !appdata.animate) )
      TimerTic ();
    return appdata.animate;
case '0':
    appdata.brprog.colour_source = 0;
    return true;
case '1':
    appdata.brprog.colour_source = 1;
    return true;
case '2':
    appdata.brprog.colour_source = 2;
    return true;
case 'D':
    appdata.brprog.normal_source = (appdata.brprog.normal_source+1) % 3;
    return true;
case 'M':
    appdata.brprog.modify_depth = !appdata.brprog.modify_depth;
    return true;
case 'U':
    SetShadowsOnOff ( &appdata.light, appdata.shadows = !appdata.shadows );
    return true;
case 'P':
    if ( (appdata.particles = !appdata.particles) )
      ResetParticles ( &appdata.psprog, &appdata.ps, app_time );
    return true;
case 'R':
    if ( appdata.particles )
      ResetParticles ( &appdata.psprog, &appdata.ps, app_time );
    return appdata.particles;
case 'X':
    return HoldOnOff ( &appdata );
case '>':
    return ChangeFocus ( true );
case '<':
    return ChangeFocus ( false );
default:    /* ignorujemy wszystkie inne klawisze */
    return false;
  }
} /*ProcessCharCommand*/

char ToggleStereo ( void )
{
  appdata.stereo = !appdata.stereo;
  return true;
} /*ToggleStereo*/

char MoveOn ( void )
{
  return !appdata.hold;
} /*MoveOn*/

