#version 450 core

#define MAX_NLIGHTS    8
#define MAX_MATERIALS 10

in FVertex {
    flat int instance;
    vec3     Colour;
    vec3     Position;
    vec3     pu, pv, Normal, TNormal;
    vec2     PatchCoord, TexCoord;
    vec4     ShadowPos[MAX_NLIGHTS];
  } In;

out vec4 out_Colour;

uniform TransBlock {
    mat4 mm, mmti, vm, pm, vpm;
    vec4 eyepos;
  } trb;

struct LSPar {
    vec4 position;
    vec3 ambient;
    vec3 direct;
    vec3 attenuation;
    mat4 shadow_vpm;
  };

uniform LSBlock {
    uint  nls;              /* liczba zrodel swiatla */
    uint  mask;             /* maska wlaczonych zrodel */
    uint  shmask;           /* maska tekstur cienia */
    LSPar ls[MAX_NLIGHTS];  /* poszczegolne zrodla swiatla */
  } light;

struct Material {
    vec4  emission0, emission1;
    vec3  diffref;    /* odbijanie rozproszone swiatla */
    vec3  specref;    /* odbijanie zwierciadlane swiatla */
    float shininess, wa, we;  /* polysk */
  };

uniform MatBlock {
    int mtn;
    Material mat[MAX_MATERIALS];
  } mat;

uniform int ColourSource = 0, LightingModel = 0, NormalSource = 0;
uniform bool ModifyDepth = false;

layout (binding = 0) uniform sampler2D tex;
layout (binding = 2) uniform sampler2DShadow shtex[MAX_NLIGHTS];

Material mm;
vec3     normal, tnormal, pnormal;

void GetColours ( vec2 tc )
{
  switch ( ColourSource ) {
default:
    mm.diffref = In.Colour;
    mm.specref = vec3 ( 0.25 );
    mm.shininess = 20.0;  mm.wa = mm.we = 1.0;
    break;
case 1:
    mm = mat.mat[mat.mtn];
    break;
case 2:
    mm = mat.mat[mat.mtn];
    if ( tc.x >= 0.0 && tc.x <= 1.0 && tc.y >= 0.0 && tc.y <= 1.0 )
      mm.diffref = texture ( tex, tc ).rgb;
    break;
  }
} /*GetColours*/

#define SQRT3 1.7320508
#define C     (-0.005)

const mat3x2 dtxm1 = mat3x2 ( 10.0, 0.0,  0.0, 10.0,  0.0, 0.0 );
const mat3x2 dtxm2 = mat3x2 ( 6.0*SQRT3, 0.0,  0.0, 10.0,  0.0, 0.0 );

vec3 NTex12 ( float dx, float dy )
{
  float r;

  r = 2.0*sqrt ( dx*dx + dy*dy );
  if ( r >= 1.0 )
    return vec3 ( 0.0, 0.0, 0.0 );
  else if ( r > 0.0 )
    return C*vec3 ( (r+r-3.0)*r*r+1.0, (24.0*r-24.0)*vec2 ( dx, dy ) );
  else
    return vec3 ( C, 0.0, 0.0 );
} /*NTex12*/

vec3 NormalTex12 ( float dx, float dy, mat2 L, out float dz )
{
  vec2 Ddq;
  vec3 d, dpu, dpv;

  d = NTex12 ( dx, dy );
  dz = d.x;
  Ddq = L * d.yz;
  dpu = In.pu + Ddq.x*pnormal;
  dpv = In.pv + Ddq.y*pnormal;
  return normalize ( cross ( dpu, dpv ) );
} /*NormalTex12*/

vec3 NormalTexture1 ( vec2 pc, out float dz )
{
  pc = fract ( dtxm1*vec3 ( pc, 1.0 ) );
  return NormalTex12 ( pc.x-0.5, pc.y-0.5, mat2(dtxm1), dz );
} /*NormalTexture1*/

vec3 NormalTexture2 ( vec2 pc, out float dz )
{
  int  c;
  float dx, dy;

  pc = dtxm2*vec3 ( pc, 1.0 );
  pc = fract ( vec2 ( pc.x/SQRT3, pc.y ) );
  switch ( (pc.x > 0.5 ? 1 : 0) + (pc.y > 0.5 ? 2 : 0) ) {
case 0: c = 3.0*pc.x-pc.y > 0.5 ? 1 : 0;  break;
case 1: c = 3.0*pc.x+pc.y > 2.5 ? 3 : 1;  break;
case 2: c = 3.0*pc.x+pc.y > 1.5 ? 2 : 0;  break;
case 3: c = 3.0*pc.x-pc.y > 1.5 ? 3 : 2;  break;
  }
  switch ( c ) {
case 0: dx = pc.x;      dy = pc.y-0.5;  break;  /* a */
case 1: dx = pc.x-0.5;  dy = pc.y;      break;  /* c */
case 2: dx = pc.x-0.5;  dy = pc.y-1.0;  break;  /* b */
case 3: dx = pc.x-1.0;  dy = pc.y-0.5;  break;  /* d */
  }
  return NormalTex12 ( dx*SQRT3, dy, mat2(dtxm2), dz );
} /*NormalTexture2*/

#define M3ROW(M,I) vec3 ( M[0][I], M[1][I], M[2][I] )

vec3 GetNormalVector ( vec2 pc, out float dz )
{
  vec3  nv;

  switch ( NormalSource ) {
default: dz = 0.0; return pnormal;
 case 1: nv = NormalTexture1 ( pc, dz );  break;
 case 2: nv = NormalTexture2 ( pc, dz );  break;
  }
  dz *= 0.5*dot ( M3ROW ( trb.vpm, 2 ), pnormal ) * gl_FragCoord.w;
  return nv;
} /*GetNormalVector*/

vec3 posDifference ( vec4 p, vec3 pos, out float dist )
{
  vec3 v;

  if ( p.w != 0.0 ) {
    v = p.xyz/p.w-pos.xyz;
    dist = sqrt ( dot ( v, v ) );
  }
  else
    v = p.xyz;
  return normalize ( v );
} /*posDifference*/

float attFactor ( vec3 att, float dist )
{
  return 1.0/(((att.z*dist)+att.y)*dist+att.x);
} /*attFactor*/

vec3 MatEmission ( float cosnv )
{
  if ( cosnv > 0.0 )
    return mm.emission1.rgb * pow ( cosnv, mm.emission1.a ) + mm.emission0.rgb;
  else
    return mm.emission0.rgb;
} /*MatEmission*/

float IsEnlighted ( uint l )
{
  return textureProj ( shtex[l], In.ShadowPos[l] );
} /*IsEnlighted*/

vec3 LambertLighting ( void )
{
  vec3  lv, vv, Colour;
  float d, e, s, dist;
  uint  i, mask;

  vv = posDifference ( trb.eyepos, In.Position, dist );
  e = dot ( vv, tnormal );
  Colour = MatEmission ( dot ( normal, vv ) );
  for ( i = 0, mask = 0x00000001;  i < light.nls;  i++, mask <<= 1 )
    if ( (light.mask & mask) != 0 ) {
      Colour += light.ls[i].ambient * mm.diffref;
      s = ((light.shmask & mask) != 0) ? IsEnlighted ( i ) : 1.0;
      if ( s > 0.0 ) {
        lv = posDifference ( light.ls[i].position, In.Position, dist );
        d = dot ( lv, normal );
        if ( e > 0.0 ) {
          if ( d > 0.0 ) {
            if ( light.ls[i].position.w != 0.0 )
              d *= attFactor ( light.ls[i].attenuation, dist );
            Colour += (s*d * light.ls[i].direct) * mm.diffref;
          }
        }
        else {
          if ( d < 0.0 ) {
            if ( light.ls[i].position.w != 0.0 )
              d *= attFactor ( light.ls[i].attenuation, dist );
            Colour -= (s*d * light.ls[i].direct) * mm.diffref;
          }
        }
      }
    }
  return clamp ( Colour, 0.0, 1.0 );
} /*LambertLighting*/

float wFactor ( float lvn, float wa, float we )
{
  return 1.0+(wa-1.0)*pow ( 1.0-lvn, we );
} /*wFactor*/

vec3 BlinnPhongLighting ( void )
{
  vec3  lv, vv, hv, Colour;
  float a, d, e, f, s, dist;
  uint  i, mask;

  vv = posDifference ( trb.eyepos, In.Position, dist );
  e = dot ( vv, tnormal );
  Colour = MatEmission ( dot ( normal, vv ) );
  for ( i = 0, mask = 0x00000001;  i < light.nls;  i++, mask <<= 1 )
    if ( (light.mask & mask) != 0 ) {
      Colour += light.ls[i].ambient * mm.diffref;
      s = ((light.shmask & mask) != 0) ? IsEnlighted ( i ) : 1.0;
      if ( s > 0.0 ) {
        lv = posDifference ( light.ls[i].position, In.Position, dist );
        d = dot ( lv, normal );
        if ( e > 0.0 ) {
          if ( d > 0.0 ) {
            a = light.ls[i].position.w != 0.0 ?
                  s*attFactor ( light.ls[i].attenuation, dist ) : s;
            Colour += (a*d*light.ls[i].direct) * mm.diffref;
            hv = normalize ( lv+vv );
            f = pow ( dot ( hv, normal ), mm.shininess );
            Colour += (a*f*wFactor(d,mm.wa,mm.we))*mm.specref;
          }
        }
        else {
          if ( d < 0.0 ) {
            a = light.ls[i].position.w != 0.0 ?
                  s*attFactor ( light.ls[i].attenuation, dist ) : s;
            Colour -= (a*d*light.ls[i].direct) * mm.diffref;
            hv = normalize ( lv+vv );
            f = pow ( -dot ( hv, normal ), mm.shininess );
            Colour += (a*f*wFactor(-d,mm.wa,mm.we))*mm.specref;
          }
        }
      }
    }
  return  clamp ( Colour, 0.0, 1.0 );
} /*BlinnPhongLighting*/

#define AGamma(colour) pow ( colour, vec3(256.0/563.0) )

void main ( void )
{
  vec3  Colour;
  float dz;

  pnormal = normalize ( In.Normal );
  tnormal = In.TNormal;
  normal = GetNormalVector ( In.PatchCoord, dz );
  gl_FragDepth = ModifyDepth ? gl_FragCoord.z + dz : gl_FragCoord.z;
  GetColours ( In.TexCoord );
  switch ( LightingModel ) {
default: Colour = LambertLighting ();     break;
 case 1: Colour = BlinnPhongLighting ();  break;
  }
  out_Colour = vec4 ( AGamma ( Colour ), 1.0 );
} /*main*/
