#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "../utilities/utilities.h"
#include "lights.h"

#define NLSOFFS   7
#define NMATUOFFS 9

static const GLchar *ULSNames[NLSOFFS+1] =
  { "LSBlock", "LSBlock.nls", "LSBlock.mask",
    "LSBlock.ls[0].ambient", "LSBlock.ls[0].direct", "LSBlock.ls[0].position",
    "LSBlock.ls[0].attenuation", "LSBlock.ls[1].ambient" };
static const GLchar *UMatNames[NMATUOFFS+1] =
  { "MatBlock", "MatBlock.mtn", "MatBlock.mat[0].emission0",
    "MatBlock.mat[0].emission1", "MatBlock.mat[0].diffref",
    "MatBlock.mat[0].specref", "MatBlock.mat[0].shininess",
    "MatBlock.mat[0].wa", "MatBlock.mat[0].we",
    "MatBlock.mat[1].emission0" };

static GLuint lsbbp = GL_INVALID_INDEX, matbbp = GL_INVALID_INDEX;
static GLint  lsbsize, lsbofs[NLSOFFS], matbsize, matbofs[NMATUOFFS];

void GetAccessToLightMatUniformBlocks ( GLuint program_id )
{
  if ( lsbbp == GL_INVALID_INDEX )
    GetAccessToUniformBlock ( program_id, NLSOFFS, &ULSNames[0],
                              &lsbsize, lsbofs, &lsbbp );
  else
    AttachUniformBlockToBP ( program_id, ULSNames[0], lsbbp );
  if ( matbbp == GL_INVALID_INDEX )
    GetAccessToUniformBlock ( program_id, NMATUOFFS, &UMatNames[0],
                              &matbsize, matbofs, &matbbp );
  else
    AttachUniformBlockToBP ( program_id, UMatNames[0], matbbp );
} /*GetAccessToLightMatUniformBlocks*/

void AttachUniformLightMatBlockToBP ( GLuint program_id )
{
  AttachUniformBlockToBP ( program_id, ULSNames[0], lsbbp );
  AttachUniformBlockToBP ( program_id, UMatNames[0], matbbp );
} /*AttachUniformLightMatBlockToBP*/

GLuint NewUniformLightBlock ( void )
{
  return NewUniformBuffer ( lsbsize, lsbbp );
} /*NewUniformLightBlock*/

void SetLightAmbient ( LightBl *light, int l, GLfloat amb[3] )
{
  GLint ofs;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( light->ls[l].ambient, amb, 3*sizeof(GLfloat) );
  ofs = l*(lsbofs[6]-lsbofs[2]) + lsbofs[2];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 3*sizeof(GLfloat), amb );
  ExitIfGLError ( "SetLightAmbient" );
} /*SetLightAmbient*/

void SetLightDirect ( LightBl *light, int l, GLfloat dir[3] )
{
  GLint ofs;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( light->ls[l].direct, dir, 3*sizeof(GLfloat) );
  ofs = l*(lsbofs[6]-lsbofs[2]) + lsbofs[3];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 3*sizeof(GLfloat), dir );
  ExitIfGLError ( "SetLightDirect" );
} /*SetLightDirect*/

void SetLightPosition ( LightBl *light, int l, GLfloat pos[4] )
{
  GLint   ofs;
  GLfloat w, *p;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( p = light->ls[l].position, pos, 4*sizeof(GLfloat) );
  w = p[3];
  if ( w != 0.0 && w != 1.0 )
    p[0] /= w, p[1] /= w, p[2] /= w, p[3] = 1.0;
  ofs = l*(lsbofs[6]-lsbofs[2]) + lsbofs[4];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 4*sizeof(GLfloat), p );
  ExitIfGLError ( "SetLightPosition" );
} /*SetLightPosition*/

void SetLightAttenuation ( LightBl *light, int l, GLfloat atn[3] )
{
  GLint ofs;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( light->ls[l].attenuation, atn, 3*sizeof(GLfloat) );
  ofs = l*(lsbofs[6]-lsbofs[2]) + lsbofs[5];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 3*sizeof(GLfloat), atn );
  ExitIfGLError ( "SetLightAttenuation" );
} /*SetLightAttenuation*/

void SetLightOnOff ( LightBl *light, int l, char on )
{
  GLuint mask;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  mask = 0x01 << l;
  if ( on ) {
    light->mask |= mask;
    if ( l >= light->nls )
      light->nls = l+1;
  }
  else {
    light->mask &= ~mask;
    for ( mask = 0x01 << (light->nls-1); mask; mask >>= 1 ) {
      if ( light->mask & mask )
        break;
      else
        light->nls --;
    }
  }
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[0], sizeof(GLuint), &light->nls );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[1], sizeof(GLuint), &light->mask );
  ExitIfGLError ( "SetLightOnOff" );
} /*SetLightOnOff*/

/* ////////////////////////////////////////////////////////////////////////// */
GLuint NewUniformMatBlock ( void ) 
{
  return NewUniformBuffer ( matbsize, matbbp );
} /*NewUniformMatBlock*/

int AllocMaterialID ( MatBl *mat )
{
  if ( mat->nmat >= MAX_MATERIALS )
    ExitOnError ( "AllocMaterialID" );
  return mat->nmat ++;
} /*AllocMaterialID*/

int SetupMaterial ( MatBl *matbl, int m, const GLfloat diffr[3],
                    const GLfloat specr[3], GLfloat shn, GLfloat wa, GLfloat we )
{
  GLint ofs;
  GLfloat em[4] = { 0.0, 0.0, 0.0, 1.0 };

  if ( m < 0 )
    m = AllocMaterialID ( matbl );
  if ( m < MAX_MATERIALS ) {
    memcpy ( matbl->mat[m].diffref, diffr, 4*sizeof(GLfloat) );
    memcpy ( matbl->mat[m].specref, specr, 4*sizeof(GLfloat) );
    matbl->mat[m].shininess = shn;
    matbl->mat[m].wa = wa;
    matbl->mat[m].we = we;
    ofs = m*(matbofs[NMATUOFFS-1]-matbofs[1]);
    glBindBuffer ( GL_UNIFORM_BUFFER, matbl->matbuf );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[1], 4*sizeof(GLfloat), em );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[2], 4*sizeof(GLfloat), em );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[3], 3*sizeof(GLfloat), diffr );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[4], 3*sizeof(GLfloat), specr );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[5], sizeof(GLfloat), &shn );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[6], sizeof(GLfloat), &wa );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[7], sizeof(GLfloat), &we );
    ExitIfGLError ( "SetupMaterial" );
  }
  return m;
} /*SetupMaterial*/

void SetMaterialEmission ( MatBl *matbl, int m,
                           const GLfloat em0[4], const GLfloat em1[4] )
{
  GLint ofs;

  if ( m >= 0 && m < matbl->nmat ) {
    ofs = m*(matbofs[NMATUOFFS-1]-matbofs[1]);
    glBindBuffer ( GL_UNIFORM_BUFFER, matbl->matbuf );
    memcpy ( matbl->mat[m].emission0, em0, 4*sizeof(GLfloat) );
    memcpy ( matbl->mat[m].emission1, em1, 4*sizeof(GLfloat) );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[1], 4*sizeof(GLfloat), em0 );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[2], 4*sizeof(GLfloat), em1 );
    ExitIfGLError ( "SetMaterialEmission" );
  }
} /*SetMaterialEmission*/

void ChooseMaterial ( MatBl *matbl, GLint m )
{
  if ( m < 0 || m >= matbl->nmat )
    m = 0;
  glBindBuffer ( GL_UNIFORM_BUFFER, matbl->matbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, matbofs[0], sizeof(GLint), &m );
  ExitIfGLError ( "ChooseMaterial" );
} /*ChooseMaterial*/

