#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "../utilities/utilities.h"
#include "lights.h"
#include "trans.h"
#include "../utilities/bezpatches.h"
#include "../utilities/teapot.h"
#include "app2a.h"
#include "app2astruct.h"
#include "app2aproc.h"

static AppData appdata;

void LoadBPShaders ( GLuint program_id[2] )
{
  static const char *filename[] =
    { "app2.vert.glsl", "app2.tesc.glsl", "app2.tese.glsl",
      "app2.geom.glsl", "app2.frag.glsl",
      "app2a1.vert.glsl", "app2a1.frag.glsl" };
  static const GLuint shtype[7] =
    { GL_VERTEX_SHADER, GL_TESS_CONTROL_SHADER, GL_TESS_EVALUATION_SHADER,
      GL_GEOMETRY_SHADER, GL_FRAGMENT_SHADER,
      GL_VERTEX_SHADER, GL_FRAGMENT_SHADER };
  GLuint shader_id[7];
  int    i;

  for ( i = 0; i < 7; i++ )
    shader_id[i] = CompileShaderFiles ( shtype[i], 1, &filename[i] );
  program_id[0] = LinkShaderProgram ( 5, shader_id, "0" );
  program_id[1] = LinkShaderProgram ( 2, &shader_id[5], "1" );
  GetAccessToTransBlockUniform ( program_id[0] );
  GetAccessToLightBlockUniform ( program_id[0] );
  GetAccessToBezPatchStorageBlocks ( program_id[0], false, false );
  AttachUniformTransBlockToBP ( program_id[1] );
  for ( i = 0; i < 7; i++ )
    glDeleteShader ( shader_id[i] );
  ExitIfGLError ( "LoadBPShaders" );
} /*LoadBPShaders*/

void SetupModelMatrix ( AppData *ad, float axis[3], float angle )
{
#define SCF 0.33
  GLfloat ms[16], mr[16], mt[16], ma[16];

  memcpy ( ad->model_rot_axis, axis, 3*sizeof(float) );
  ad->model_rot_angle = angle;
  M4x4Scalef ( ms, SCF, SCF, SCF*4.0/3.0 );
  M4x4Translatef ( mt, 0.0, 0.0, -0.6 );
  M4x4Multf ( ma, mt, ms );
  M4x4RotateVf ( mr, axis[0], axis[1], axis[2], angle );
  M4x4Multf ( ms, mr, ma );
  LoadMMatrix ( &ad->trans, ms );
} /*SetupModelMatrix*/

void _RotateViewer ( AppData *ad, double delta_xi, double delta_eta )
{
  float   vi[3], lgt, vk[3];
  double  angi, angk;

  vi[0] = (float)delta_eta*ad->camera.tb/(float)ad->camera.win_height;
  vi[1] = (float)delta_xi*ad->camera.rl/(float)ad->camera.win_width;
  vi[2] = 0.0;
  lgt = sqrt ( V3DotProductf ( vi, vi ) );
  vi[0] /= lgt;  vi[1] /= lgt;
  angi = -0.052359878;  /* -3 stopnie */
  V3CompRotationsf ( vk, &angk, ad->camera.viewer_rvec,
                     ad->camera.viewer_rangle, vi, angi );
  memcpy ( ad->camera.viewer_rvec, vk, 3*sizeof(float) );
  ad->camera.viewer_rangle = angk;
  M4x4RotateVfv ( ad->trans.vm, ad->camera.viewer_rvec,
                  -ad->camera.viewer_rangle );
  M4x4MultMTVf ( ad->trans.eyepos, ad->trans.vm, ad->camera.viewer_pos0 );
  M4x4InvTranslateMfv ( ad->trans.vm, ad->camera.viewer_pos0 );
  LoadVPMatrix ( &ad->trans );
} /*_RotateViewer*/

void RotateViewer ( double delta_xi, double delta_eta )
{
  if ( delta_xi == 0.0 && delta_eta == 0.0 )
    return;  /* natychmiast uciekamy - nie chcemy dzielic przez 0 */
  _RotateViewer ( &appdata, delta_xi, delta_eta );
} /*RotateViewer*/

void InitLights ( AppData *ad )
{
  GLfloat amb0[3] = { 0.2, 0.2, 0.3 };
  GLfloat dir0[3] = { 0.8, 0.8, 0.8 };
  GLfloat pos0[4] = { 0.0, 1.0, 1.0, 0.0 };
  GLfloat atn0[3] = { 1.0, 0.0, 0.0 };

  SetLightAmbient ( &ad->light, 0, amb0 );
  SetLightDirect ( &ad->light, 0, dir0 );
  SetLightPosition ( &ad->light, 0, pos0 );
  SetLightAttenuation ( &ad->light, 0, atn0 );
  SetLightOnOff ( &ad->light, 0, 1 );
} /*InitLights*/

void ConstructMyTeapot ( AppData *ad )
{
  const GLfloat MyColour[4] = { 1.0, 1.0, 1.0, 1.0 };

  ad->myteapot = ConstructTheTeapot ( MyColour );
  SetBezierPatchTessLevel ( ad->myteapot, ad->TessLevel );
  SetBezierPatchNVS ( ad->myteapot, ad->BezNormals );
} /*ConstructMyTeapot*/

static void _ResizeMyWorld ( AppData *ad, int width, int height )
{
  float lr;

  glViewport ( 0, 0, ad->camera.win_width = width,
               ad->camera.win_height = height );
  lr = 0.5533*(float)width/(float)height;
  M4x4Frustumf ( ad->trans.pm, NULL, ad->camera.left = -lr, ad->camera.right = lr,
                 ad->camera.bottom = -0.5533, ad->camera.top = 0.5533,
                 ad->camera.near = 5.0, ad->camera.far = 15.0 );
  ad->camera.rl = 2.0*lr;  ad->camera.tb = 2.0*0.5533;
  LoadVPMatrix ( &ad->trans );
} /*_ResizeMyWorld*/

void ResizeMyWorld ( int width, int height )
{
  _ResizeMyWorld ( &appdata, width, height );
} /*ResizeMyWorld*/

void InitCamera ( AppData *ad, int width, int height )
{
  static const float viewer_rvec[3] = {1.0,0.0,0.0};
  static const float viewer_pos0[4] = {0.0,0.0,10.0,1.0};

  memcpy ( ad->camera.viewer_rvec, viewer_rvec, 3*sizeof(float) );
  memcpy ( ad->camera.viewer_pos0, viewer_pos0, 4*sizeof(float) );
  memcpy ( ad->trans.eyepos, viewer_pos0, 4*sizeof(GLfloat) );
  ad->camera.viewer_rangle = 0.0;
  M4x4Translatef ( ad->trans.vm, -viewer_pos0[0],
                   -viewer_pos0[1], -viewer_pos0[2] );
  _ResizeMyWorld ( ad, width, height );
} /*InitCamera*/

void InitMyWorld ( int argc, char *argv[], int width, int height )
{
  float axis[4] = {0.0,0.0,1.0};

  memset ( &appdata, 0, sizeof(AppData) );
  LoadBPShaders ( appdata.program_id );
  appdata.trans.trbuf = NewUniformTransBlock ();
  appdata.light.lsbuf = NewUniformLightBlock ();
  TimerInit ();
  SetupModelMatrix ( &appdata, axis, 0.0 );
  InitCamera ( &appdata, width, height );
  appdata.TessLevel = 10;
  appdata.BezNormals = GL_TRUE;
  appdata.cnet = appdata.skeleton = false;
  ConstructMyTeapot ( &appdata );
  InitLights ( &appdata );
} /*InitMyWorld*/

void DrawMyTeapot ( AppData *ad )
{
  glUseProgram ( ad->program_id[0] );
  if ( ad->skeleton )
    glPolygonMode ( GL_FRONT_AND_BACK, GL_LINE );
  else
    glPolygonMode ( GL_FRONT_AND_BACK, GL_FILL );
  DrawBezierPatches ( ad->myteapot );
} /*DrawMyTeapot*/

void DrawMyCNet ( BezierPatchObjf *bp, GLuint prog_id )
{
  glUseProgram ( prog_id );
  DrawBezierNets ( bp );
} /*DrawMyCNet*/

void RedrawMyWorld ( void )
{
  glClearColor ( 1.0, 1.0, 1.0, 1.0 );
  glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
  glEnable ( GL_DEPTH_TEST );
  DrawMyTeapot ( &appdata );
  if ( appdata.cnet )
    DrawMyCNet ( appdata.myteapot, appdata.program_id[1] );
  glUseProgram ( 0 );
  glFlush ();
  ExitIfGLError ( "RedrawMyWorld" );
} /*RedrawMyWorld*/

void DeleteMyWorld ( void )
{
  glUseProgram ( 0 );
  glDeleteProgram ( appdata.program_id[0] );
  glDeleteProgram ( appdata.program_id[1] );
  glDeleteBuffers ( 1, &appdata.trans.trbuf );
  glDeleteBuffers ( 1, &appdata.light.lsbuf );
  DeleteBezierPatches ( appdata.myteapot );
  DeleteEmptyVAO ();
#ifdef DEBUG_BUFFERS_ALLOCATION
  DumpBufferIdentifiers ();
#endif
} /*DeleteMyWorld*/

char ProcessCharCommand ( char charcode )
{
  switch ( toupper ( charcode ) ) {
case '+':
    if ( appdata.TessLevel < MAX_TESS_LEVEL ) {
      SetBezierPatchTessLevel ( appdata.myteapot, ++appdata.TessLevel );
      return true;
    }
    else return false;
case '-':
    if ( appdata.TessLevel > MIN_TESS_LEVEL ) {
      SetBezierPatchTessLevel ( appdata.myteapot, --appdata.TessLevel );
      return true;
    }
    else return false;
case 'N':
    SetBezierPatchNVS ( appdata.myteapot,
                        appdata.BezNormals = appdata.BezNormals == 0 );
    return true;
case 'C':
    appdata.cnet = !appdata.cnet;
    return true;
case 'S':
    appdata.skeleton = !appdata.skeleton;
    return true;
default:
    return false;
  }
} /*ProcessCharCommand*/

char MoveOn ( void )
{
  if ( (appdata.model_rot_angle += ANGULAR_VELOCITY * TimerTocTic ()) >= PI )
    appdata.model_rot_angle -= 2.0*PI;
  SetupModelMatrix ( &appdata, appdata.model_rot_axis,
                     appdata.model_rot_angle );
  return true;
} /*MoveOn*/

