# author: Linh Anh Nguyen, nguyen@mimuw.edu.pl
# created: 2025-06-15

import sys
from queue import PriorityQueue
import CompFP as cfp

#===================================================================

class InversedRoleName:
    @staticmethod
    def getInversedRoleName0(r):
        assert type(r) == str and len(r) > 0 and r[-1] != '-'
        return r + "-"

    @staticmethod
    def getInversedRoleName(r):
        return r + "-" if r[-1] != '-' else r[:-1]

    @staticmethod
    def isAtomicRoleName(r):
        assert type(r) == str and len(r) > 0
        return r[-1] != '-'
    
class FuzzyGraph(cfp.FuzzyGraph):
    def __init__(self, withI, withO):
        super().__init__()
        self.withI = withI
        self.withO = withO
        self.IndNames = {}

    def addNamedIndividual(self, IName, vID):
        v = self.getVertex(vID)
        self.IndNames[IName] = v
        if self.withO:
            self.SV.add(IName)
            v.addLabel(IName, 1)
        
    def addEdge(self, r, xID, yID, d):
        if type(r) != str:
            r = str(r)
        
        self.SE.add(r)
        x = self.getVertex(xID)
        y = self.getVertex(yID)
        if d > 0:
            self.E.add(cfp.Edge(r, x, y, d))        
            if self.withI:
                r2 = InversedRoleName.getInversedRoleName0(r)
                self.SE.add(r2)
                self.E.add(cfp.Edge(r2, y, x, d))

    def read(self, filename = None):
        if filename is not None:
            f = open(filename, "r")
        else:
            f = sys.stdin

        for line in f:
            l = line.rstrip().split()
            if len(l) == 2:
                self.addNamedIndividual(l[0], l[1])
            elif len(l) == 3:
                self.addVertexLabel(l[0], l[1], float(l[2]))
            elif len(l) == 4:
                self.addEdge(l[2], l[0], l[1], float(l[3]))
            elif len(l) != 0:
                assert False, "Bad input file!"

        if filename is not None:
            f.close()
    
    def __str__(self):
        buf = "{\n"
        for ind in self.IndNames:
            buf += "  individual(" + str(ind) + ":" + \
                   str(self.IndNames[ind].ID) + ")\n"
        for v in self.V:
            buf += "  " + str(v) + "\n"
        for e in self.E:
            buf += "  " + str(e) + "\n"
        buf += "}"
        return buf

#===================================================================

class Vertex:
    def __init__(self, ID):
        self.ID = ID
        self.block = None

    def __str__(self):
        return str(self.ID)

    @staticmethod
    def getVertex(vertices, ID):
        if ID not in vertices:
            vertices[ID] = Vertex(ID)
        return vertices[ID]

    def findBlock(self, d):
        b = self.block
        while b.parent != None and b.parent.degree >= d:
            b = b.parent
        b.flatten()
        return b

class Block:
    def __init__(self, B, vertices):
        self.repr = None
        self.parent = None
        if type(B) == cfp.SCBlock:
            self.flattened = True
            self.degree = 1
            self.subblocks = None
            self.elements = []
            v = B.elements.head
            while v != None:
                v2 = Vertex.getVertex(vertices, v.ID)
                self.elements.append(v2)
                v2.block = self
                v = v.next
        else:
            self.flattened = False
            self.degree = B.degree
            self.elements = None
            self.subblocks = []
            b = B.subblocks.head
            while b != None:
                b2 = Block(b, vertices)
                b2.parent = self
                self.subblocks.append(b2)
                b = b.next

    def flatten(self):
        if self.flattened:
            return
        assert len(self.subblocks) > 1

        self.elements = []
        for b in self.subblocks:
            b.flatten()
            if len(b.elements) > len(self.elements):
                self.elements, b.elements = b.elements, self.elements
            for v in b.elements:
                self.elements.append(v)
                v.block = self
            b.elements = []
        self.flattened = True
        self.subblocks = None

    def __str__(self):
        buf = f"Block({self.repr}, {self.degree}, {{"
        if self.flattened:
            for i, v in enumerate(self.elements):
                buf += str(v)
                if i < len(self.elements) - 1:
                    buf += ", "
        else:
            for i, b in enumerate(self.subblocks):
                buf += str(b)
                if i < len(self.subblocks) - 1:
                    buf += ", "
        buf += "})"
        return buf
    
#===================================================================

def minimize(G, bB = None, gamma = 1, verbose = False,
             returnReducedSize = False, filename = None):
    assert isinstance(G, FuzzyGraph)
    assert bB == None or isinstance(bB, cfp.Block)
    
    if not returnReducedSize and filename != None:
        original_stdout = sys.stdout
        sys.stdout = open(filename, "w")

    if bB == None:
        inst = cfp.CompFP(G)
        inst.verbose = verbose
        bB = inst.compFP()

    vertices = {}    
    mB = Block(bB, vertices)
    if verbose:
        print("--------------------------------------------------------------------------------\n")
        print("mB =", mB)

    D = set([gamma])
    E = {}
    for yID, y2 in vertices.items():
        y = G.getVertex(yID)
        for e in y.comingEdges:
            assert e.degree > 0
            x2 = Vertex.getVertex(vertices, e.origin.ID)
            if x2 not in E:
                E[x2] = {}
            E[x2][(e.label, y2)] = e.degree
            if e.degree < gamma:
                D.add(e.degree)

    V2 = []
    E2 = {}
    IndNames2 = {}
    Q = PriorityQueue()

    t = 0 # time
    for a, v in G.IndNames.items():
        v2 = Vertex.getVertex(vertices, v.ID)
        B = v2.findBlock(gamma)
        if B.repr == None:
            V2.append(v2)
            IndNames2[a] = v2
            while True:
                B.repr = v2
                B = B.parent
                if B == None or B.repr != None:
                    break
            for (R, y2), d in E.get(v2, {}).items():
                t += 1
                Q.put((-d, t, v2, R, y2))
        else:
            IndNames2[a] = B.repr

    for d in sorted(D, reverse = True):
        while not Q.empty() and -Q.queue[0][0] >= d:
            (_, _, x, R, y) = Q.get()
            Bp = B = y.findBlock(d)
            if verbose:
                print("\nd, x, R, y =", d, x, R, y)
                print("  mB =", mB)
            if B.repr == None:
                V2.append(y)
                while True:
                    Bp.repr = y
                    Bp = Bp.parent
                    if Bp == None or Bp.repr != None:
                        break
                for (S, z), dp in E.get(y, {}).items():
                    t += 1
                    Q.put((-dp, t, y, S, z))
            yp = B.repr
            if verbose:
                print("  yp =", yp)
            if (x, R, yp) not in E2:
                E2[(x, R, yp)] = d
                if G.withI:
                    E2[(yp, InversedRoleName.getInversedRoleName(R), x)] = d

    if returnReducedSize:
        return (bB, len(V2), sum(1 for (x, R, y) in E2
                                 if InversedRoleName.isAtomicRoleName(R)))
    
    if verbose:
        print("\n--------------------------------------------------------------------------------\n")
    
    for o, v2 in IndNames2.items():
        print(o, v2.ID)

    for v2 in V2:
        v = G.getVertex(v2.ID)
        for p, d in v.label.items():
            if p not in IndNames2:
                print(v2.ID, p, d)

    for (x, R, y), d in E2.items():
        if InversedRoleName.isAtomicRoleName(R):
            print(x.ID, y.ID, R, d)
        
    if filename is not None:
        sys.stdout.close()
        sys.stdout = original_stdout

#========================================================================

def computeReducedSizeU(G, bB = None): # the case with the universal role
    assert isinstance(G, FuzzyGraph)
    assert bB == None or isinstance(bB, cfp.Block)

    if bB == None:
        inst = cfp.CompFP(G)
        inst.verbose = verbose
        inst.compFP()

    V2 = set()
    for v in G.vertices.values():
        assert v.scBlock != None
        V2.add(v.scBlock)

    E2 = set()
    for e in G.E:
        x = e.origin.scBlock
        y = e.destination.scBlock
        assert e.degree > 0
        E2.add((e.label, x, y))

    return (bB, len(V2), len(E2) if not G.withI else len(E2)//2)

#===================================================================

if __name__ == "__main__":
    prompt = "Call the program in the form:\n\n" + \
            "  python3 fuzzyMinG.py [options] < input_file\n\n" + \
            "where input_file specifies the input and options are among:\n\n" + \
            "  gamma=value  where value is a real number from (0,1], 1 as default,\n\n" + \
            "  withI        with inverse roles,\n\n" + \
            "  withO        with nominals,\n\n" + \
            "  verbose      verbose\n"
    
    if "--help" in sys.argv:
        assert len(sys.argv) == 2
        sys.stderr.write(prompt)
        quit()

    verbose = "verbose" in sys.argv
    withI = "withI" in sys.argv
    withO = "withO" in sys.argv
    gamma = 1
    for arg in sys.argv:
        if len(arg) > 6 and arg[:6] == "gamma=":
            gamma = float(arg[6:])

    G = FuzzyGraph(withI, withO)
    G.read()
    minimize(G, gamma = gamma, verbose = verbose)
