# author: Linh Anh Nguyen, nguyen@mimuw.edu.pl
# created: 2025-06-14

import time
import random
import numpy as np
import fuzzyMinG as fmg
import crispMin as crm
import CompFP as cfp

#============================================================================

def runCrispMin(G):
    assert isinstance(G, crm.FuzzyGraph)
    start = time.time()
    (n, m) = crm.minimize(G, returnReducedSize = True)
    end = time.time()
    return (n, m, end - start)

def runFuzzyMinG(G, bB = None, gamma = 1):
    assert isinstance(G, fmg.FuzzyGraph)
    start = time.time()
    (bB, n, m) = fmg.minimize(G, bB, gamma, returnReducedSize = True)
    end = time.time()
    return (bB, n, m, end - start)

def genRandFuzzyGraphs(k, n, m, o, p, l, sCN, sRN, acyclic, withI, withO):
    assert l >= 1

    cG = crm.FuzzyGraph(withI, withO)
    fG = fmg.FuzzyGraph(withI, withO)

    for i in range(k):
        for j in range(o):
            cG.addNamedIndividual("o" + str(i*o + j), i*n + j)
            fG.addNamedIndividual("o" + str(i*o + j), i*n + j)
  
        vertexLabels = set()
        while len(vertexLabels) < p:
            vertexLabels.add((i*n + random.randrange(n),
                              "p" + str(random.randrange(sCN))))

        for (x, q) in vertexLabels:
            d = (random.randrange(l) + 1) / l
            cG.addVertexLabel(x, q, d)
            fG.addVertexLabel(x, q, d)

        edges = set()
        while len(edges) < m:
            x = i*n + random.randrange(n)
            y = i*n + random.randrange(n)
            if acyclic:
                x, y = min(x,y), max(x,y)
            edges.add((x, y, random.randrange(sRN)))

        for (x, y, r) in edges:
            d = (random.randrange(l) + 1) / l
            cG.addEdge(r, x, y, d)
            fG.addEdge(r, x, y, d)

    assert not cG.withU
    cG.deleteUnreachable()    

    return (cG, fG)

def runTest(k, n, m, o, p, l, sCN, sRN, acyclic, withI, withO):
    (cG, fG) = genRandFuzzyGraphs(k, n, m, o, p, l, sCN, sRN,
                                    acyclic, withI, withO)

    (bB, n1, m1, t1) = runFuzzyMinG(fG)
    print("...", n1, m1, t1)
    (_, n3, m3) = fmg.computeReducedSizeU(fG, bB)
    print("...", n3, m3)
    (n2, m2, t2) = runCrispMin(cG)
    print("---", n2, m2, t2)

    return (n3, m3, n2, m2, n1, m1, round(t1))

def run(filename):
    scenarios = [
        # [ k, n, m, o, p, l, sCN, sRN, acyclic, withI, withO]
        [100, 500, 1000, 10, 20, 3, 3, 3, 0, 0, 0], 
        [100, 500, 1000, 10, 20, 3, 3, 3, 0, 0, 1], 
        [100, 500, 1000, 10, 20, 3, 3, 3, 0, 1, 0], 
        [100, 500, 1000, 10, 20, 3, 3, 3, 0, 1, 1], 
        [100, 500, 1000, 10, 20, 3, 3, 3, 1, 0, 0], 
        [100, 500, 1000, 10, 20, 3, 3, 3, 1, 0, 1], 
        [100, 500, 1000, 10, 20, 3, 3, 3, 1, 1, 0], 
        [100, 500, 1000, 10, 20, 3, 3, 3, 1, 1, 1], 
        [100, 500, 2000, 20, 40, 3, 3, 3, 0, 0, 0], 
        [100, 500, 2000, 20, 40, 3, 3, 3, 0, 0, 1], 
        [100, 500, 2000, 20, 40, 3, 3, 3, 1, 0, 0], 
        [100, 500, 2000, 20, 40, 3, 3, 3, 1, 0, 1], 
        [100, 500, 4000, 40, 80, 3, 3, 3, 1, 0, 0], 
        [100, 500, 4000, 40, 80, 3, 3, 3, 1, 0, 1], 
        [50000, 5,  6, 1, 2, 10, 1, 1, 0, 0, 0], 
        [50000, 5,  6, 1, 2, 10, 1, 1, 0, 0, 1], 
        [50000, 5,  6, 1, 2, 10, 1, 1, 0, 1, 0], 
        [50000, 5,  6, 1, 2, 10, 1, 1, 0, 1, 1], 
        [50000, 5,  6, 1, 2, 10, 1, 1, 1, 0, 0], 
        [50000, 5,  6, 1, 2, 10, 1, 1, 1, 0, 1], 
        [50000, 5,  6, 1, 2, 10, 1, 1, 1, 1, 0], 
        [50000, 5,  6, 1, 2, 10, 1, 1, 1, 1, 1], 
        [50000, 5, 10, 1, 2, 10, 1, 1, 0, 0, 0], 
        [50000, 5, 10, 1, 2, 10, 1, 1, 0, 0, 1], 
        [50000, 5, 10, 1, 2, 10, 1, 1, 1, 0, 0], 
        [50000, 5, 10, 1, 2, 10, 1, 1, 1, 0, 1], 
        [50000, 5, 15, 1, 2, 10, 1, 1, 0, 0, 0],
        [50000, 5, 15, 1, 2, 10, 1, 1, 0, 0, 1],
        [50000, 5, 15, 1, 2, 10, 1, 1, 1, 0, 0],
        [50000, 5, 15, 1, 2, 10, 1, 1, 1, 0, 1],
        [100, 1000, 2000, 10, 20, 10, 10, 10, 0, 0, 0], 
        [100, 1000, 2000, 10, 20, 10, 10, 10, 0, 0, 1], 
        [100, 1000, 2000, 10, 20, 10, 10, 10, 0, 1, 0], 
        [100, 1000, 2000, 10, 20, 10, 10, 10, 0, 1, 1], 
        [100, 1000, 2000, 10, 20, 10, 10, 10, 1, 0, 0], 
        [100, 1000, 2000, 10, 20, 10, 10, 10, 1, 0, 1], 
        [100, 1000, 2000, 10, 20, 10, 10, 10, 1, 1, 0], 
        [100, 1000, 2000, 10, 20, 10, 10, 10, 1, 1, 1], 
        [100, 1000, 4000, 10, 40, 10, 10, 10, 0, 0, 0],
        [100, 1000, 4000, 10, 40, 10, 10, 10, 0, 0, 1],
        [100, 1000, 4000, 10, 40, 10, 10, 10, 1, 0, 0],
        [100, 1000, 4000, 10, 40, 10, 10, 10, 1, 0, 1],
        [100, 1000, 8000, 10, 80, 10, 10, 10, 1, 0, 0], 
        [100, 1000, 8000, 10, 80, 10, 10, 10, 1, 0, 1], 
        [2000, 100, 200, 10, 10, 10, 1, 2, 0, 0, 0], 
        [2000, 100, 200, 10, 10, 10, 1, 2, 0, 0, 1], 
        [2000, 100, 200, 10, 10, 10, 1, 2, 0, 1, 0], 
        [2000, 100, 200, 10, 10, 10, 1, 2, 0, 1, 1], 
        [2000, 100, 200, 10, 10, 10, 1, 2, 1, 0, 0], 
        [2000, 100, 200, 10, 10, 10, 1, 2, 1, 0, 1], 
        [2000, 100, 200, 10, 10, 10, 1, 2, 1, 1, 0], 
        [2000, 100, 200, 10, 10, 10, 1, 2, 1, 1, 1], 
        [2000, 100, 400, 10, 10, 10, 1, 2, 0, 0, 0], 
        [2000, 100, 400, 10, 10, 10, 1, 2, 0, 0, 1], 
        [2000, 100, 400, 10, 10, 10, 1, 2, 1, 0, 0], 
        [2000, 100, 400, 10, 10, 10, 1, 2, 1, 0, 1], 
        [2000, 100, 800, 10, 10, 10, 1, 2, 1, 0, 0], 
        [2000, 100, 800, 10, 10, 10, 1, 2, 1, 0, 1], 
        [10000, 10, 20, 2, 2, 100, 1, 1, 0, 0, 0], 
        [10000, 10, 20, 2, 2, 100, 1, 1, 0, 0, 1], 
        [10000, 10, 20, 2, 2, 100, 1, 1, 0, 1, 0], 
        [10000, 10, 20, 2, 2, 100, 1, 1, 0, 1, 1], 
        [10000, 10, 20, 2, 2, 100, 1, 1, 1, 0, 0], 
        [10000, 10, 20, 2, 2, 100, 1, 1, 1, 0, 1], 
        [10000, 10, 20, 2, 2, 100, 1, 1, 1, 1, 0], 
        [10000, 10, 20, 2, 2, 100, 1, 1, 1, 1, 1], 
        [10000, 10, 30, 3, 3, 100, 1, 1, 0, 0, 0], 
        [10000, 10, 30, 3, 3, 100, 1, 1, 0, 0, 1], 
        [10000, 10, 30, 3, 3, 100, 1, 1, 1, 0, 0], 
        [10000, 10, 30, 3, 3, 100, 1, 1, 1, 0, 1], 
        [60000, 5, 6, 1, 2, 100, 10, 10, 1, 0, 0], 
        [70000, 5, 6, 1, 2, 100, 10, 10, 1, 0, 0], 
        [80000, 5, 6, 1, 2, 100, 10, 10, 1, 0, 0], 
        [90000, 5, 6, 1, 2, 100, 10, 10, 1, 0, 0], 
        [100000, 5, 6, 1, 2, 100, 10, 10, 1, 0, 0], 
        [60000, 5, 10, 1, 2, 100, 10, 10, 1, 0, 0], 
        [70000, 5, 10, 1, 2, 100, 10, 10, 1, 0, 0], 
        [80000, 5, 10, 1, 2, 100, 10, 10, 1, 0, 0], 
        [90000, 5, 10, 1, 2, 100, 10, 10, 1, 0, 0], 
        [100000, 5, 10, 1, 2, 100, 10, 10, 1, 0, 0], 
    ]
    repeats = 3
    rscols = 7

    for scenario in scenarios:
        [k, n, m, o, p, l, sCN, sRN, acyclic, withI, withO] = scenario
        rs = np.zeros(rscols, dtype=float)
        for _ in range(repeats):
            rs += np.array(runTest(k, n, m, o, p, l, sCN, sRN,
                                   acyclic, withI, withO))
        rs /= repeats

        s = ""
        for x in scenario:
            s += str(x) + " "
        for i in range(rscols):
            s += str(round(rs[i]))
            s += " " if i < rscols - 1 else "\n"
        with open(filename, "a") as f:
            f.write(s)
        print("===")

if __name__ == "__main__":
    run("results.txt")
