# author: Linh Anh Nguyen, nguyen@mimuw.edu.pl
# created: 2023-03-08
# last modified: 2025-06-16

import sys
from queue import Queue
import CompCB as crispbis

#===================================================================

def readFuzzyGraph(G, filename = None):
    if filename is not None:
        f = open(filename, "r")
    else:
        f = sys.stdin

    for line in f:
        l = line.rstrip().split()
        if len(l) == 2:
            G.addNamedIndividual(l[0], l[1])
        elif len(l) == 3:
            G.addVertexLabel(l[0], l[1], float(l[2]))
        elif len(l) == 4:
            G.addEdge(l[2], l[0], l[1], float(l[3]))
        elif len(l) != 0:
            assert False, "Bad input file!"

    if filename is not None:
        f.close()
    
class InversedRoleName:
    @staticmethod
    def getInversedRoleName(r):
        assert type(r) == str and len(r) > 0 and r[-1] != '-'
        return r + "-"

    @staticmethod
    def isAtomicRoleName(r):
        assert type(r) == str and len(r) > 0
        return r[-1] != '-'
    
class FuzzyGraph(crispbis.FuzzyGraph):
    def __init__(self, withI = False, withO = False, withU = False):
        super().__init__()
        self.withI = withI
        self.withO = withO
        self.withU = withU
        self.IndNames = {}

    def addNamedIndividual(self, IName, vID):
        v = self.getVertex(vID)
        self.IndNames[IName] = v
        if self.withO:
            self.SV.add(IName)
            v.addLabel(IName, 1)

    def addEdge(self, r, xID, yID, d):
        if type(r) != str:
            r = str(r)
        
        self.SE.add(r)
        x = self.getVertex(xID)
        y = self.getVertex(yID)
        self.E.add(crispbis.Edge(r, x, y, d))
        
        if self.withI:
            r2 = InversedRoleName.getInversedRoleName(r)
            self.SE.add(r2)
            self.E.add(crispbis.Edge(r2, y, x, d))

    def deleteUnreachable(self):
        edges = {}
        for e in self.E:
            if e.origin not in edges:
                edges[e.origin] = set()
            edges[e.origin].add(e.destination)
                
        reachable = set()
        Q = Queue()
        for v in self.IndNames.values():
            if v not in reachable:
                reachable.add(v)
                Q.put(v)

        while not Q.empty():
            x = Q.get()
            for y in edges.get(x, []):
                if y not in reachable:
                    reachable.add(y)
                    Q.put(y)

        for (ID, y) in self.vertices.items():
            for r in y.comingEdges:
                y.comingEdges[r] = [e for e in y.comingEdges[r]
                                    if e.origin in reachable]
            y.comingEdges = {r:value for (r, value) in y.comingEdges.items()
                                       if value != []}
            
        self.vertices = {ID:x for (ID, x) in self.vertices.items() if x in reachable}
        self.E = {e for e in self.E if e.origin in reachable}
            
    def read(self, filename = None):
        readFuzzyGraph(self, filename)
        if not self.withU:
            self.deleteUnreachable()    
    
    def __str__(self):
        buf = "{\n"
        for ind in self.IndNames:
            buf += "  individual(" + str(ind) + ":" + \
                   str(self.IndNames[ind].ID) + ")\n"
        for v in self.V:
            buf += "  " + str(v) + "\n"
        for e in self.E:
            buf += "  " + str(e) + "\n"
        buf += "}"
        return buf

def getBlockName(block):
    assert isinstance(block, crispbis.Block)
    return "_".join([str(e) for e in block.sortedList()])

def minimize(G, with_compound_names = False, verbose = False,
             returnReducedSize = False, filename = None):
    assert isinstance(G, FuzzyGraph)
    if not returnReducedSize and filename != None:
        original_stdout = sys.stdout
        sys.stdout = open(filename, "w")

    inst = crispbis.CompCB(G)
    inst.verbose = verbose
    P = inst.compCB()

    blockNames = {}
    if with_compound_names:
        for b in P.blocks:
            blockNames[b] = getBlockName(b)
    else:
        k = 0
        L = [(getBlockName(b), b) for b in P.blocks]
        L.sort()
        for e in L:
            blockNames[e[1]] = str(k)
            k += 1
    
    edges = {}
    for e in G.E:
        bx = blockNames[e.origin.block]
        by = blockNames[e.destination.block]
        if not G.withI or InversedRoleName.isAtomicRoleName(e.label):
            if (bx, by, e.label) not in edges:
                edges[(bx, by, e.label)] = e.blockEdge.maxKey()

    if returnReducedSize:
        return (len(P.blocks), len(edges))

    for o in sorted(G.IndNames):
        b = G.IndNames[o].block
        print(o, blockNames[b])

    vlm = {}
    for b in P.blocks:
        x = b.vertices.head
        for p in x.label:
            if p not in G.IndNames:
                vlm[(blockNames[b], p)] = x.label[p]
                
    for vl in sorted(vlm): 
        print(vl[0], vl[1], f'{vlm[vl]:g}')

    for e in sorted(edges):
        print(e[0], e[1], e[2], f'{edges[e]:g}')

    if filename != None:
        sys.stdout.close()
        sys.stdout = original_stdout

#===================================================================
# A NAIVE VERSION FOR TESTING THE CORRECTNESS:
#===================================================================

class FuzzyGraph2(crispbis.FuzzyGraph2):
    def __init__(self, withI = False, withO = False, withU = False):
        super().__init__()
        self.withI = withI
        self.withO = withO
        self.withU = withU
        self.IndNames = {}
        
    def addNamedIndividual(self, IName, vID):
        self.addVertex(vID)
        if IName in self.IndNames:
            sys.exit("Named individual " + IName + " declared twice!")
        self.IndNames[IName] = vID
        if self.withO:
            self.addVertexLabel(vID, IName, 1)

    def addEdge(self, r, xID, yID, d):
        if type(r) != str:
            r = str(r)
        
        self.SE.add(r)
        self.addVertex(xID)
        self.addVertex(yID)
        if r not in self.E[xID]:
            self.E[xID][r] = {}
        if yID in self.E[xID][r]:
            sys.exit("Role instance (" + xID + ", " + yID + ", " + r + 
                     ") defined twice!")
        self.E[xID][r][yID] = d

        if self.withI:
            r2 = InversedRoleName.getInversedRoleName(r)
            self.SE.add(r2)
            if r2 not in self.E[yID]:
                self.E[yID][r2] = {}
            self.E[yID][r2][xID] = d
            
    def deleteUnreachable(self):
        reachable = set()
        Q = Queue()
        for vID in self.IndNames.values():
            if vID not in reachable:
                reachable.add(vID)
                Q.put(vID)

        while not Q.empty():
            xID = Q.get()
            for r in self.E.get(xID, []):
                for yID in self.E[xID][r]:
                    if yID not in reachable:
                        reachable.add(yID)
                        Q.put(yID)

        self.V = {vID:value for (vID, value) in self.V.items() if vID in reachable}
        self.E = {xID:value for (xID, value) in self.E.items() if xID in reachable}
            
    def read(self, filename = None):
        readFuzzyGraph(self, filename)
        if not self.withU:
            self.deleteUnreachable()    

    def __str__(self):
        return str(self.IndNames) + "\n" + str(self.V) + "\n" + str(self.E)
        
def minimize2(G, with_compound_names, filename = None):
    assert isinstance(G, FuzzyGraph2)
    if filename is not None:
        original_stdout = sys.stdout
        sys.stdout = open(filename, "w")

    inst = crispbis.CompCB2(G)
    P = inst.compCB()

    blockNames = {}
    if with_compound_names:
        for b in P:
            bn = "_".join(b)
            for v in b:
                blockNames[v] = bn
    else:
        k = 0
        L = [("_".join(b), b) for b in P]
        L.sort()
        for e in L:
            bn = str(k)
            k += 1
            for v in e[1]:
                blockNames[v] = bn
    
    for o in sorted(G.IndNames):
        print(o, blockNames[G.IndNames[o]])

    vlm = {}
    for b in P:
        x = b[0]
        for p in G.V[x]:
            if p not in G.IndNames:
                vlm[(blockNames[x], p)] = G.V[x][p]

    for vl in sorted(vlm): 
        print(vl[0], vl[1], f'{vlm[vl]:g}')

    edges = {}
    for x in G.E:
        bx = blockNames[x]
        for r in G.E[x]:
            if not G.withI or InversedRoleName.isAtomicRoleName(r):
                for y in G.E[x][r]:
                    by = blockNames[y]
                    if (bx, by, r) not in edges:
                        edges[(bx, by, r)] = G.E[x][r][y]
                    else:
                        edges[(bx, by, r)] = max(edges[(bx, by, r)], 
                                                 G.E[x][r][y])
                           
    for e in sorted(edges):
        print(e[0], e[1], e[2], f'{edges[e]:g}')

    if filename is not None:
        sys.stdout.close()
        sys.stdout = original_stdout

#===================================================================

if __name__ == "__main__":
    with_compound_names = "with_compound_names" in sys.argv
    verbose = "verbose" in sys.argv
    naive = "naive" in sys.argv
    withI = "withI" in sys.argv
    withO = "withO" in sys.argv
    withU = "withU" in sys.argv
    
    if not naive:
        G = FuzzyGraph(withI, withO, withU)
        G.read()
        minimize(G, with_compound_names = with_compound_names,
                 verbose = verbose)
    else:
        G = FuzzyGraph2(withI, withO, withU)
        G.read()
        minimize2(G, with_compound_names = with_compound_names)
