# author: Linh Anh Nguyen, nguyen@mimuw.edu.pl
# created: 2021-07-09
# modified: 2025-06-15

import sys
import random

class DList: # doubly linked list
    def __init__(self, kind = 1):
        assert kind in [1,2]
        self.kind = kind
        self.head = None
        self.size = 0

    def add(self, elem):
        if self.kind == 1:
            elem.prev = None
            elem.next = self.head
            if self.head is not None:
                self.head.prev = elem
        else:
            elem.prev2 = None
            elem.next2 = self.head
            if self.head is not None:
                self.head.prev2 = elem
        self.head = elem
        self.size += 1

    def remove(self, elem):
        if self.kind == 1:
            p = elem.prev
            n = elem.next
            if p is not None:
                p.next = n
            if n is not None:
                n.prev = p
            elem.prev = None
            elem.next = None
        else:
            p = elem.prev2
            n = elem.next2
            if p is not None:
                p.next2 = n
            if n is not None:
                n.prev2 = p
            elem.prev2 = None
            elem.next2 = None

        if self.head == elem:
            self.head = n
        self.size -= 1

    def empty(self):
        return self.head is None

    def __str__(self):
        buf = "["
        first = True
        x = self.head
        while x is not None:
            if not first:
                buf += ", "
            first = False
            buf += str(x)
            if self.kind == 1:
                x = x.next
            else:
                x = x.next2
        buf += "]"
        return buf

#-------------------------------------------------------------------------------

class Vertex:
    def __init__(self, ID):
        self.ID = ID
        self.label = {}
        self.scBlock = None
        self.next = None
        self.prev = None
        self.comingEdges = []
        self.processed = False
        self.label2 = set()
        self.label3 = set()
        self.label4 = set()
        self.label5 = set()

    def addLabel(self, p, d):
        assert p not in self.label
        self.label[p] = d

    def __str__(self):
        return "vertex(%s:%s)" % (str(self.ID), str(self.label))

class Edge:
    def __init__(self, r, x, y, d, pce = None):
        self.label = r
        self.origin = x
        self.destination = y
        self.degree = d
        self.pcEdge = pce
        y.comingEdges.append(self)

    def __str__(self):
        return "edge(%s, %s, %s, %.2f)" % (str(self.origin.ID), str(self.destination.ID), str(self.label), self.degree)

class Block:
    def __init__(self, parent):
        self.parent = parent
        self.next = None
        self.prev = None

class FBlock(Block):
    def __init__(self, parent, degree):
        super().__init__(parent)
        self.degree = degree
        self.subblocks = DList()
        if parent is not None:
            parent.subblocks.add(self)

    def toString(self, offset = 0):
        indent = ""
        for _ in range(offset):
            indent += " "

        buf = indent + str(self.degree) + ":{\n"

        b = self.subblocks.head
        while b is not None:
            buf += b.toString(offset + 2) + "\n"
            b = b.next

        buf += indent + "}"
        return buf

    def __str__(self):
        return self.toString()

    def export(self):
        L = []
        b = self.subblocks.head
        while b is not None:
            L.append(b.export())
            b = b.next
        return (L, self.degree)
    
    def anyElement(self):
        return self.subblocks.head.anyElement()
    
class SCBlock(Block):
    def __init__(self, parent, elements, pComponent):
        super().__init__(parent)
        self.elements = elements
        self.pComponent = pComponent

        self.next2 = None
        self.prev2 = None
        self.departingSubblocks1 = {}
        self.departingSubblocks2 = {}

        if parent is not None:
            parent.subblocks.add(self)

        x = self.elements.head
        while x is not None:
            x.scBlock = self
            x = x.next

        if pComponent is not None:
            pComponent.addBlock(self)

    def size(self):
        return self.elements.size

    def toString(self, offset = 0):
        buf = ""
        for _ in range(offset):
            buf += " "
        buf += "{"

        x = self.elements.head
        first = True
        while x is not None:
            if not first:
                buf += ", "
            first = False
            buf += str(x.ID)
            x = x.next

        buf += "}"
        return buf

    def __str__(self):
        return self.toString()

    def export(self):
        rs = []
        x = self.elements.head
        while x is not None:
            rs.append(x.ID)
            x = x.next
        return rs

    def anyElement(self):
        return self.elements.head.ID

class PComponent:
    def __init__(self, pPartition):
        self.pPartition = pPartition
        self.scBlocks = DList(2)
        self.next = None
        self.prev = None
        pPartition.addPComponent(self)

    def size(self):
        return self.scBlocks.size

    def compound(self):
        return self.size() > 1

    def smallerBlock(self):
        assert self.compound()
        b1 = self.scBlocks.head
        b2 = b1.next2
        if b1.size() < b2.size():
            return b1
        return b2

    def addBlock(self, b):
        self.scBlocks.add(b)
        if self.size() == 2:
            self.pPartition.simpleComponents.remove(self)
            self.pPartition.compoundComponents.add(self)

    def removeBlock(self, b):
        self.scBlocks.remove(b)
        if self.size() == 1:
            self.pPartition.compoundComponents.remove(self)
            self.pPartition.simpleComponents.add(self)

    @staticmethod
    def createPComponent(pPartition, b):
        pc = PComponent(pPartition)
        pc.addBlock(b)
        b.pComponent = pc

    def __str__(self):
        return str(self.scBlocks)

class PPartition:
    def __init__(self):
        self.compoundComponents = DList()
        self.simpleComponents = DList()

    def addPComponent(self, pc):
        if pc.compound():
            self.compoundComponents.add(pc)
        else:
            self.simpleComponents.add(pc)

    def __str__(self):
        return str(self.compoundComponents) + "+" + str(self.simpleComponents)

class PComponentEdge:
    def __init__(self, spce = None):
        self.counter = {}
        self.departingPCEdge = None
        self.sourcePCEdge = spce

    def pushKey(self, r, d):
        if r not in self.counter:
            self.counter[r] = {}
        if d not in self.counter[r]:
            self.counter[r][d] = 1
        else:
            self.counter[r][d] += 1

    def popKey(self, r, d):
        if r in self.counter and d in self.counter[r]:
            self.counter[r][d] -= 1
            if self.counter[r][d] == 0:
                del self.counter[r][d]
                if len(self.counter[r]) == 0:
                    del self.counter[r]

    def maxKey(self, r):
        if r not in self.counter:
            return 0
        return max(self.counter[r])

    def __str__(self):
        return str(self.counter)

class LabelDegree:
    def __init__(self, *args):
        if len(args) == 2:
            v, p = args[0], args[1]
            self.aboutVertex = True
            self.vertex = v
            self.vertexLabel = p
            self.degree = v.label[p]
            self.edge = None
        else:
            e = args[0]
            self.aboutVertex = False
            self.vertex = None
            self.vertexLabel = None
            self.degree = e.degree
            self.edge = e

    def __str__(self):
        if self.aboutVertex:
            return str(self.vertex) + "(" + str(self.vertexLabel) + ")"
        else:
            return str(self.edge)

#-------------------------------------------------------------------------------

class FuzzyGraph:
    def __init__(self):
        self.E = set()
        self.SV = set()
        self.SE = set()
        self.vertices = {} # {ID: vertex, ...}

    def V(self):
        return self.vertices.values()

    def getVertex(self, ID):
        if ID not in self.vertices:
            self.vertices[ID] = Vertex(ID)
        return self.vertices[ID]

    def addVertexLabel(self, ID, p, d):
        self.SV.add(p)
        v = self.getVertex(ID)
        if d > 0:
            v.addLabel(p, d)

    def addEdge(self, r, xID, yID, d):
        self.SE.add(r)
        x = self.getVertex(xID)
        y = self.getVertex(yID)
        if d > 0:
            self.E.add(Edge(r, x, y, d))

    def read(self, filename = None):
        if filename is not None:
            f = open(filename, "r")
        else:
            f = sys.stdin

        for line in f:
            l = line.rstrip().split()
            if len(l) == 3:
                self.addVertexLabel(l[0], l[1], float(l[2]))
            elif len(l) == 4:
                self.addEdge(l[2], l[0], l[1], float(l[3]))
            elif len(l) == 0:
                pass
            else:
                assert False, "Bad input file!"

        if filename is not None:
            f.close()

    def __str__(self):
        rs = ""
        for e in self.V():
            rs += str(e) + "\n"
        for e in self.E:
            rs += str(e) + "\n"
        return rs

#-------------------------------------------------------------------------------

class CompFP:
    def __init__(self, G):
        self.G = G
        self.verbose = False

    def initialize(self):
        self.vertices = self.G.V()
        self.edges = self.G.E

        vertices2 = DList()
        for x in self.vertices:
            vertices2.add(x)

        self.P = PPartition()
        pc = PComponent(self.P)
        self.B = SCBlock(None, vertices2, pc)

        pcEdges = {}
        for x in self.vertices:
            pcEdges[x] = PComponentEdge()

        for e in self.edges:
            e.pcEdge = pcEdges[e.origin]
            e.pcEdge.pushKey(e.label, e.degree)

        self.labelDegrees = []
        for x in self.vertices:
            for p in x.label:
                self.labelDegrees.append(LabelDegree(x,p))
        for e in self.edges:
            self.labelDegrees.append(LabelDegree(e))
        self.labelDegrees.sort(key = lambda ld: ld.degree)

        self.labelDegrees_idx = 0

        self.allDegrees = sorted(set([0] + [ld.degree for ld in self.labelDegrees]))

    def refine1a(self):
        for ld in self.labelDegrees:
            if ld.aboutVertex:
                x = ld.vertex
                p = ld.vertexLabel
                x.label2.add(p)
            else:
                e = ld.edge
                x = e.origin
                r = e.label
                x.label3.add(r)

        for x in self.vertices:
            key = (tuple(sorted(x.label2)), tuple(sorted(x.label3)))
            if key not in self.B.departingSubblocks1:
                self.B.departingSubblocks1[key] = DList()
            self.B.elements.remove(x)
            self.B.departingSubblocks1[key].add(x)

        if len(self.B.departingSubblocks1) == 1:
            key = list(self.B.departingSubblocks1)[0]
            self.B.elements, self.B.departingSubblocks1[key] = self.B.departingSubblocks1[key], self.B.elements
            self.B.departingSubblocks1.clear()
        else:
            pc = self.B.pComponent
            pc.removeBlock(self.B)
            B2 = FBlock(None, 0)
            for key in self.B.departingSubblocks1:
                SCBlock(B2, self.B.departingSubblocks1[key], pc)
            self.B = B2

        for x in self.vertices:
            x.label2.clear()
            x.label3.clear()

    def refine1b(self, di):
        vertices_tbp = []
        while self.labelDegrees_idx < len(self.labelDegrees) \
                and self.labelDegrees[self.labelDegrees_idx].degree == di:
            ld = self.labelDegrees[self.labelDegrees_idx]
            self.labelDegrees_idx += 1
            if ld.aboutVertex:
                x = ld.vertex
                p = ld.vertexLabel
                x.label2.add(p)
                if not x.processed:
                    vertices_tbp.append(x)
                    x.processed = True
            else:
                e = ld.edge
                r = e.label
                pce = e.pcEdge
                if pce.maxKey(r) == di:
                    x = e.origin
                    pc = e.destination.scBlock.pComponent
                    x.label3.add((r,id(pc)))
                    if not x.processed:
                        vertices_tbp.append(x)
                        x.processed = True

        for x in vertices_tbp:
            bx = x.scBlock
            key = (tuple(sorted(x.label2)), tuple(sorted(x.label3)))
            if key not in bx.departingSubblocks1:
                bx.departingSubblocks1[key] = DList()
            bx.elements.remove(x)
            bx.departingSubblocks1[key].add(x)

        for x in vertices_tbp:
            bx = x.scBlock
            if len(bx.departingSubblocks1) == 0:
                continue

            if len(bx.departingSubblocks1) == 1 and bx.elements.empty():
                key = list(bx.departingSubblocks1)[0]
                bx.departingSubblocks1[key], bx.elements = bx.elements, bx.departingSubblocks1[key]
            else:
                pc = bx.pComponent

                bp = bx.parent
                if bp is not None:
                    bp.subblocks.remove(bx)

                bx2 = FBlock(bp, di)
                if self.B == bx:
                    self.B = bx2

                for key in bx.departingSubblocks1:
                    SCBlock(bx2, bx.departingSubblocks1[key], pc)

                if not bx.elements.empty():
                    bx2.subblocks.add(bx)
                    bx.parent = bx2
                else:
                    pc.removeBlock(bx)
                
            bx.departingSubblocks1.clear()

        for x in vertices_tbp:
            x.label2.clear()
            x.label3.clear()
            x.processed = False

    def refine2(self, di, X):
        vertices_of_X = []
        x = X.elements.head
        while x is not None:
            vertices_of_X.append(x)
            x = x.next

        self.computePComponentEdges(vertices_of_X)
        self.computeSubblocks(di, vertices_of_X)
        self.doSplitting(di, X, vertices_of_X)
        self.clearAuxiliaryInfo(vertices_of_X)

    def computePComponentEdges(self, vertices_of_X):
        for x in vertices_of_X:
            for e in x.comingEdges:
                pce = e.pcEdge
                if pce.departingPCEdge is None:
                    pce.departingPCEdge = PComponentEdge(pce)
                dpce = pce.departingPCEdge
                r = e.label
                d = e.degree
                pce.popKey(r, d)
                dpce.pushKey(r, d)

    def computeSubblocks(self, di, vertices_of_X):
        for x in vertices_of_X:
            for e in x.comingEdges:
                v = e.origin
                r = e.label
                pce = e.pcEdge
                dpce = pce.departingPCEdge
                if pce.maxKey(r) > di:
                    v.label4.add(r)
                if dpce.maxKey(r) > di: 
                    v.label5.add(r)

        for x in vertices_of_X:
            for e in x.comingEdges:
                v = e.origin
                bv = v.scBlock
                if not v.processed:
                    if len(v.label4) > 0 or len(v.label5) > 0:
                        key = (tuple(sorted(v.label4)), tuple(sorted(v.label5)))
                        if key not in bv.departingSubblocks2:
                            bv.departingSubblocks2[key] = DList()
                        bv.elements.remove(v)
                        bv.departingSubblocks2[key].add(v)
                    v.processed = True

    def doSplitting(self, di, X, vertices_of_X):
        Y = X.pComponent
        Y.removeBlock(X)
        PComponent.createPComponent(Y.pPartition, X)

        for x in vertices_of_X:
            for e in x.comingEdges:
                e.pcEdge = e.pcEdge.departingPCEdge
                v = e.origin
                bv = v.scBlock
                if len(bv.departingSubblocks2) == 0:
                    continue

                if len(bv.departingSubblocks2) == 1 and bv.elements.empty():
                    key = list(bv.departingSubblocks2)[0]
                    bv.departingSubblocks2[key], bv.elements = bv.elements, bv.departingSubblocks2[key]
                else:
                    pc = bv.pComponent
                    bp = bv.parent
                    if bp is None or bp.degree < di:
                        if bp is not None:
                            bp.subblocks.remove(bv)
                        bv2 = FBlock(bp, di)
                        if self.B == bv:
                            self.B = bv2

                        for key in bv.departingSubblocks2:
                            SCBlock(bv2, bv.departingSubblocks2[key], pc)

                        if not bv.elements.empty():
                            bv2.subblocks.add(bv)
                            bv.parent = bv2
                        else: 
                            pc.removeBlock(bv)

                    else:
                        for key in bv.departingSubblocks2:
                            SCBlock(bp, bv.departingSubblocks2[key], pc)

                        if bv.elements.empty():
                            bp.subblocks.remove(bv)
                            pc.removeBlock(bv)

                bv.departingSubblocks2.clear()

    def clearAuxiliaryInfo(self, vertices_of_X):
        for x in vertices_of_X:
            for e in x.comingEdges:
                v = e.origin
                v.label4.clear()
                v.label5.clear()
                v.processed = False

                pce = e.pcEdge
                spce = pce.sourcePCEdge
                if spce is not None:
                    spce.departingPCEdge = None
                    pce.sourcePCEdge = None

    def compFP(self):
        self.initialize()
        k = len(self.allDegrees) - 1
        for i in range(k):
            di = self.allDegrees[i]
            if self.verbose:
                print("di = " + str(di) + "\n")

            if di == 0:
                self.refine1a()
            else:
                self.refine1b(di)

            if self.verbose:
                print("After refine1:\n\nB: %s\n\nP: %s\n" % (str(self.B),str(self.P)))

            while not self.P.compoundComponents.empty():
                Y = self.P.compoundComponents.head
                X = Y.smallerBlock()
                self.refine2(di, X)
                if self.verbose:
                    print("After refine2:\n\nB: %s\n\nP: %s\n" % (str(self.B),str(self.P)))

        return self.B

def test():
    scenarios = [[10,100,100,5,5,0.33,0.20],
                 [10,100,1000,5,5,0.33,0.20],
                 [10,100,1000,1,1,0.33,0.50],
                 [10,100,1000000,1,1,0.33,0.10],
                 [1,1000,100,1,1,0.33,0.001],
                 [1,1000,100,1,1,0.33,0.01]]

    for [N,n,l,nSV,nSE,prP,prE] in scenarios:
      print("TESTING SCENARIO [N,n,l,nSV,nSE,prP,prE] = ", [N,n,l,nSV,nSE,prP,prE])

      for k in range(N):
        G = FuzzyGraph()
        SV = range(nSV)
        SE = range(nSE)
        V = range(n)

        for v in V:
            for p in SV:
                if random.random() > prP:
                    d = random.randint(1,l) * 1.0 / l
                    G.addVertexLabel(v, p, d)

        for x in V:
            for y in V:
                if y != x:
                    for r in SE:
                        if random.random() > prE:
                            d = random.randint(1,l) * 1.0 / l
                            G.addEdge(r, x, y, d)


        inst = CompFP(G)
        inst.compFP()
        print("Done a random test #" + str(k+1) + ".")

if __name__ == "__main__":
    verbose = "-v" in sys.argv or "--verbose" in sys.argv

    G = FuzzyGraph()
    G.read()
    inst = CompFP(G)
    inst.verbose = verbose
    B = inst.compFP()
    sys.stdout.write(str(B) + "\n")

