# author: Linh Anh Nguyen, nguyen@mimuw.edu.pl
# created: 2021-07-09
# last modified: 2025-06-16

import sys
import random
import time
from queue import PriorityQueue

class DList: # doubly linked list
    def __init__(self):
        self.head = None
        self.tail = None
        self.size = 0

    def add(self, elem): # at the font
        elem.prev = None
        elem.next = self.head
        if self.head is not None:
            self.head.prev = elem
        self.head = elem
        if self.tail is None:
            self.tail = elem
        self.size += 1

    def remove(self, elem):
        p = elem.prev
        n = elem.next
        if p is not None:
            p.next = n
        if n is not None:
            n.prev = p
        elem.prev = None
        elem.next = None
        if self.head == elem:
            self.head = n
        if self.tail == elem:
            self.tail = p
        self.size -= 1

    def addDList(self, dlist): # at the front
        if dlist.empty():
            return
            
        if self.head is None:
            self.head = dlist.head
            self.tail = dlist.tail
            self.size = dlist.size
        else:
            dlist.tail.next = self.head
            self.head.prev = dlist.tail
            self.head = dlist.head
            self.size += dlist.size
        
    def empty(self):
        return self.head is None

    def __str__(self):
        buf = "["
        first = True
        x = self.head
        while x is not None:
            if not first:
                buf += ", "
            first = False
            buf += str(x)
            x = x.next
        buf += "]"
        return buf

#-------------------------------------------------------------------------------

class Vertex:
    def __init__(self, ID):
        self.ID = ID
        self.label = {}
        self.block = None
        self.next = None
        self.prev = None
        self.comingEdges = {}
        self.processed = False

    def addLabel(self, p, d):
        assert p not in self.label
        self.label[p] = d

    def __str__(self):
        return "vertex(%s:%s)" % (str(self.ID), str(self.label))

class Edge:
    def __init__(self, r, x, y, d, bE = None):
        self.label = r
        self.origin = x
        self.destination = y
        self.degree = d
        self.blockEdge = bE
        if r not in y.comingEdges:
            y.comingEdges[r] = []
        y.comingEdges[r].append(self)

    def __str__(self):
        return "edge(%s, %s, %s, %.2f)" % (str(self.origin.ID), str(self.destination.ID), str(self.label), self.degree)

class Block:
    def __init__(self, vertices, partition, superBlocks):
        self.vertices = vertices
        self.partition = partition
        partition.add(self)
        self.superBlocks = {}
        self.departingSubblocks1 = {}
        self.departingSubblocks2 = {}

        x = self.vertices.head
        while x is not None:
            x.block = self
            x = x.next

        if superBlocks is not None:
            for r in superBlocks:
                self.superBlocks[r] = superBlocks[r]
                superBlocks[r].addBlock(self)

    @staticmethod
    def createBlock(vl):
        assert vl.size > 0
        x = vl.head
        bx = x.block
        Block(vl, bx.partition, bx.superBlocks)
    
    def size(self):
        return self.vertices.size

    def __str__(self):
        buf = "{"

        x = self.vertices.head
        first = True
        while x is not None:
            if not first:
                buf += ", "
            first = False
            buf += str(x.ID)
            x = x.next

        buf += "}"
        return buf
        
    def sortedList(self):
        rs = []
        x = self.vertices.head
        while x is not None:
            rs.append(x.ID)
            x = x.next
        return sorted(rs)

class Partition:
    def __init__(self):
        self.blocks = []

    def add(self, b):
        self.blocks.append(b)

    def size(self):
        return len(self.blocks)

    def __str__(self):
        buf = "{"
        first = True
        for b in self.blocks:
            if not first:
                buf += ", "
            first = False
            buf += str(b)
        buf += "}"
        return buf

    def sortedList(self):
        return sorted([b.sortedList() for b in self.blocks])
        
class SuperBlock:
    def __init__(self, superPartition):
        self.superPartition = superPartition
        self.blocks = []
        self.next = None
        self.prev = None
        superPartition.addSuperBlock(self)

    def size(self):
        return len(self.blocks)

    def compound(self):
        return self.size() > 1

    def smallerBlock(self):
        assert self.compound()
        b1 = self.blocks[0]
        b2 = self.blocks[1]
        if b1.size() < b2.size():
            return b1
        return b2

    def addBlock(self, b):
        self.blocks.append(b)
        if self.size() == 2:
            self.superPartition.simpleSuperBlocks.remove(self)
            self.superPartition.compoundSuperBlocks.add(self)

    def removeBlock(self, b):
        self.blocks.remove(b)
        if self.size() == 1:
            self.superPartition.compoundSuperBlocks.remove(self)
            self.superPartition.simpleSuperBlocks.add(self)

    @staticmethod
    def createSuperBlock(sp, b, r):
        sb = SuperBlock(sp)
        sb.addBlock(b)
        b.superBlocks[r] = sb

    def __str__(self):
        buf = "{"
        first = True
        for b in self.blocks:
            if not first:
                buf += ", "
            first = False
            buf += str(b)
        buf += "}"
        return buf

class SuperPartition:
    def __init__(self):
        self.compoundSuperBlocks = DList()
        self.simpleSuperBlocks = DList()

    def addSuperBlock(self, sb):
        if sb.compound():
            self.compoundSuperBlocks.add(sb)
        else:
            self.simpleSuperBlocks.add(sb)

    def __str__(self):
        return str(self.compoundSuperBlocks) + "+" + str(self.simpleSuperBlocks)

class BlockEdge:
    def __init__(self, sbe = None, withPriorityQueue = False):
        self.counter = {}
        self.departingBlockEdge = None
        self.sourceBlockEdge = sbe
        self.withPriorityQueue = withPriorityQueue
        if withPriorityQueue:
            self.keys = PriorityQueue()

    def pushKey(self, d):
        if d not in self.counter:
            self.counter[d] = 1
            if self.withPriorityQueue:
                self.keys.put(-d)
        else:
            self.counter[d] += 1

    def popKey(self, d):
        if d in self.counter:
            self.counter[d] -= 1
            if self.counter[d] == 0:
                del self.counter[d]
                
    def maxKey(self):
        if self.withPriorityQueue:
            while len(self.counter) > 0 and \
                    (-self.keys.queue[0]) not in self.counter:
                self.keys.get()
            if len(self.counter) == 0:
                return 0
            return -self.keys.queue[0]
        else:
            if self.counter == {}:
                return 0
            return max(self.counter) 
        
    def __str__(self):
        return str(self.counter)

def testBlockEdge():
    bE = BlockEdge()
    bE.pushKey(4)
    bE.pushKey(5)
    bE.pushKey(5)
    bE.pushKey(6)
    bE.pushKey(4)
    bE.pushKey(1)
    bE.pushKey(1)
    bE.pushKey(2)
    assert bE.maxKey() == 6
    bE.popKey(5)
    assert bE.maxKey() == 6
    bE.popKey(5)
    bE.popKey(6)
    assert bE.maxKey() == 4
    bE.popKey(2)
    bE.popKey(4)
    bE.popKey(4)
    assert bE.maxKey() == 1
    
#-------------------------------------------------------------------------------

class FuzzyGraph:
    def __init__(self):
        self.E = set()
        self.SV = set()
        self.SE = set()
        self.vertices = {} # {ID: vertex, ...}

    def V(self):
        return self.vertices.values()
    
    def getVertex(self, ID):
        if ID not in self.vertices:
            self.vertices[ID] = Vertex(ID)
        return self.vertices[ID]

    def addVertexLabel(self, ID, p, d):
        self.SV.add(p)
        v = self.getVertex(ID)
        v.addLabel(p, d)

    def addEdge(self, r, xID, yID, d):
        self.SE.add(r)
        x = self.getVertex(xID)
        y = self.getVertex(yID)
        self.E.add(Edge(r, x, y, d))

    def read(self, filename = None):
        if filename is not None:
            f = open(filename, "r")
        else:
            f = sys.stdin

        for line in f:
            l = line.rstrip().split()
            if len(l) == 3:
                self.addVertexLabel(l[0], l[1], float(l[2]))
            elif len(l) == 4:
                self.addEdge(l[2], l[0], l[1], float(l[3]))
            elif len(l) != 0:
                assert False, "Bad input file!"

        if filename is not None:
            f.close()

#-------------------------------------------------------------------------------

class CompCB:
    def __init__(self, G, withCountingSuccessors = False):
        self.G = G
        self.withCountingSuccessors = withCountingSuccessors
        self.verbose = False

    def initialize(self):
        self.vertices = self.G.V()
        self.edges = self.G.E

        self.P = Partition()
        self.Q = {}
        msb = {}
        for r in self.G.SE:
            self.Q[r] = SuperPartition()
            msb[r] = SuperBlock(self.Q[r])

        blockEdges = {}
        for x in self.vertices:
            for r in self.G.SE:
                blockEdges[(x,r)] = BlockEdge()

        for e in self.edges:
            e.blockEdge = blockEdges[(e.origin,e.label)]
            e.blockEdge.pushKey(e.degree)

        blocks = {}
        for x in self.vertices:
            key1 = tuple(sorted(x.label.items()))
            tmp = {}
            for r in self.G.SE:
                if not self.withCountingSuccessors:
                    tmp[r] = blockEdges[(x,r)].maxKey()
                else:
                    tmp[r] = tuple(sorted(
                        blockEdges[(x,r)].counter.items()))
            key2 = tuple(sorted(tmp.items()))
            key = (key1,key2)
            if key not in blocks:
                blocks[key] = DList()
            blocks[key].add(x)

        for key in blocks:
            Block(blocks[key], self.P, msb)

    def split(self, X, Y, r):
        vertices_of_X = []
        x = X.vertices.head
        while x is not None:
            vertices_of_X.append(x)
            x = x.next

        self.computeBlockEdges(vertices_of_X, r)
        self.computeSubblocks(vertices_of_X, r)
        self.doSplitting(X, vertices_of_X, Y, r)
        self.clearAuxiliaryInfo(vertices_of_X, r)

    def computeBlockEdges(self, vertices_of_X, r):
        for x in vertices_of_X:
          if r in x.comingEdges:
            for e in x.comingEdges[r]:
                bE = e.blockEdge
                if bE.departingBlockEdge is None:
                    bE.departingBlockEdge = BlockEdge(bE)
                dbE = bE.departingBlockEdge
                bE.popKey(e.degree)
                dbE.pushKey(e.degree)

    def computeSubblocks(self, vertices_of_X, r):
        for x in vertices_of_X:
          if r in x.comingEdges:
            for e in x.comingEdges[r]:
                v = e.origin
                bv = v.block
                if v.processed:
                    continue

                bE = e.blockEdge

                if not self.withCountingSuccessors:
                    dbE = bE.departingBlockEdge
                    d1 = bE.maxKey()
                    d2 = dbE.maxKey()
                    if d1 >= d2:
                        if d2 not in bv.departingSubblocks2:
                            bv.departingSubblocks2[d2] = DList()
                        bv.vertices.remove(v)
                        bv.departingSubblocks2[d2].add(v)
                    else:
                        if d1 not in bv.departingSubblocks1:
                            bv.departingSubblocks1[d1] = DList()
                        bv.vertices.remove(v)
                        bv.departingSubblocks1[d1].add(v)
                else:
                    key = tuple(sorted(bE.counter.items()))
                    if key not in bv.departingSubblocks1:
                        bv.departingSubblocks1[key] = DList()
                    bv.vertices.remove(v)
                    bv.departingSubblocks1[key].add(v)

                v.processed = True

    def doSplitting(self, X, vertices_of_X, Y, r):
        Y.removeBlock(X)
        SuperBlock.createSuperBlock(Y.superPartition, X, r)

        for x in vertices_of_X:
          if r in x.comingEdges:
            for e in x.comingEdges[r]:
                if e.blockEdge.departingBlockEdge is not None:
                    e.blockEdge = e.blockEdge.departingBlockEdge
                v = e.origin
                bv = v.block

                if not self.withCountingSuccessors:
                    if bv.departingSubblocks1 == {} and \
                            bv.departingSubblocks2 == {}:
                        continue
                elif bv.departingSubblocks1 == {}:
                    continue

                if bv.vertices.empty():    
                    if bv.departingSubblocks1 != {}:
                        d = list(bv.departingSubblocks1)[0]
                        bv.vertices = bv.departingSubblocks1[d]
                        del bv.departingSubblocks1[d]
                    else:
                        assert not self.withCountingSuccessors
                        d = list(bv.departingSubblocks2)[0]
                        bv.vertices = bv.departingSubblocks2[d]
                        del bv.departingSubblocks2[d]

                for key in bv.departingSubblocks1:
                    Block.createBlock(bv.departingSubblocks1[key])

                if not self.withCountingSuccessors:
                    for key in bv.departingSubblocks2:
                        Block.createBlock(
                            bv.departingSubblocks2[key])

                bv.departingSubblocks1.clear()
                if not self.withCountingSuccessors:
                    bv.departingSubblocks2.clear()

    def clearAuxiliaryInfo(self, vertices_of_X, r):
        for x in vertices_of_X:
          if r in x.comingEdges:
            for e in x.comingEdges[r]:
                e.origin.processed = False
                bE = e.blockEdge
                sbE = bE.sourceBlockEdge
                if sbE is not None:
                    sbE.departingBlockEdge = None
                    bE.sourceBlockEdge = None

    def compCB(self):
        self.initialize()
        if self.P.size() == 1:
            return self.P

        if self.verbose:
            print("P: ", self.P, "\n")

        changed = True
        while changed:
            changed = False
            for r in self.G.SE:
                while not self.Q[r].compoundSuperBlocks.empty():
                    Y = self.Q[r].compoundSuperBlocks.head
                    X = Y.smallerBlock()
                    if self.verbose:
                        print("Splitting by", X, Y, r)
                    self.split(X, Y, r)
                    changed = True
                    if self.verbose:
                        print("P: ", self.P)
                        print("Q[" + str(r) + "]: ", 
                              self.Q[r], "\n")

        return self.P

#===================================================================
# A NAIVE VERSION FOR TESTING THE CORRECTNESS:
#===================================================================

class FuzzyGraph2:
    def __init__(self):
        self.V = {}
        self.E = {}
        self.SV = set()
        self.SE = set()
        
    def addVertex(self, ID):
        if ID not in self.V:
            self.V[ID] = {}
            self.E[ID] = {}

    def addVertexLabel(self, ID, p, d):
        self.SV.add(p)
        self.addVertex(ID)
        self.V[ID][p] = d

    def addEdge(self, r, xID, yID, d):
        self.SE.add(r)
        self.addVertex(xID)
        self.addVertex(yID)
        if r not in self.E[xID]:
            self.E[xID][r] = {}
        self.E[xID][r][yID] = d

    def read(self, filename = None):
        if filename is not None:
            f = open(filename, "r")
        else:
            f = sys.stdin

        for line in f:
            l = line.rstrip().split()
            if len(l) == 3:
                self.addVertexLabel(l[0], l[1], float(l[2]))
            elif len(l) == 4:
                self.addEdge(l[2], l[0], l[1], float(l[3]))
            elif len(l) == 0:
                pass
            else:
                assert False, "Bad input file!"

        if filename is not None:
            f.close()

class DListElem:
    def __init__(self, value):
        self.next = None
        self.prev = None
        self.value = value
        
    def __str__(self):
        return str(self.value)
        
class CompCB2:
    def __init__(self, G, withCountingSuccessors = False):
        self.G = G
        self.withCountingSuccessors = withCountingSuccessors

    def initialize(self):
        self.vertices = set(self.G.V.keys())
        
        blocks = {}
        for x in self.vertices:
            key = tuple(sorted(self.G.V[x].items()))
            if key not in blocks:
                blocks[key] = set()
            blocks[key].add(x)
            
        self.blocks = DList()
        for key in blocks:
            self.blocks.add(DListElem(blocks[key]))

    def split(self, X, Y):
        blocks = {}
        for x in X:
            tmp = {}
            for r in self.G.E[x]:
                if not self.withCountingSuccessors:
                    maxKey = 0
                    for y in self.G.E[x][r]:
                        if y in Y and self.G.E[x][r][y] > maxKey:
                            maxKey = self.G.E[x][r][y]
                    if maxKey > 0:
                        tmp[r] = maxKey
                else:
                    tmp2 = {}
                    for y in self.G.E[x][r]:
                        if y in Y:
                            d = self.G.E[x][r][y]
                            if d not in tmp2:
                                tmp2[d] = 0
                            tmp2[d] += 1    
                    tmp[r] = tuple(sorted(tmp2.items()))
            key = tuple(sorted(tmp.items()))
            if key not in blocks:
                blocks[key] = set()
            blocks[key].add(x)
            
        blocks2 = DList()
        for key in blocks:
            blocks2.add(DListElem(blocks[key]))
        return blocks2
        
    def compCB(self):
        self.initialize()

        changed = True
        while changed:
            changed = False
            X = self.blocks.head
            while X is not None:
                Y = self.blocks.head
                while Y is not None:
                    tmp = self.split(X.value, Y.value)
                    if tmp.size > 1:
                        changed = True
                        self.blocks.remove(X)
                        self.blocks.addDList(tmp)
                        break
                    Y = Y.next
                if changed:
                    break
                X = X.next
                    
        rs = []
        X = self.blocks.head
        while X is not None:
            rs.append(sorted(X.value))
            X = X.next
        return sorted(rs)

#===================================================================

def testCorrectness():
  scenarios = [
    [2,100,1000,10,3,1,1],
    [2,100,1000,10,3,2,1],
    [2,100,1000,10,3,10,1],
    [2,100,1000,10,3,1,2],
    [2,100,1000,10,3,1,10],
    [2,100,1000,10,3,10,10],

    [2,100,1000,10,10,1,1],
    [2,100,1000,10,10,2,1],
    [2,100,1000,10,10,10,1],
    [2,100,1000,10,10,1,2],
    [2,100,1000,10,10,1,10],
    [2,100,1000,10,10,10,10],

    [2,1000,10000,100,3,1,1],
    [2,1000,10000,100,3,2,1],
    [2,1000,10000,100,3,10,1],
    [2,1000,10000,100,3,1,2],
    [2,1000,10000,100,3,1,10],
    [2,1000,10000,100,3,10,10],

    [2,1000,10000,100,10,1,1],
    [2,1000,10000,100,10,2,1],
    [2,1000,10000,100,10,10,1],
    [2,1000,10000,100,10,1,2],
    [2,1000,10000,100,10,1,10],
    [2,1000,10000,100,10,10,10],
  ]
                 
  for withCountingSuccessors in [False, True]:
    print("TESTING WITH withCountingSuccessors =", withCountingSuccessors)  
    for [N,n,m,p,l,nSV,nSE] in scenarios:
      print("TESTING SCENARIO [N,n,m,p,l,nSV,nSE] = ", [N,n,m,p,l,nSV,nSE])

      for k in range(N):
        G = FuzzyGraph()
        G2 = FuzzyGraph2()
        SV = range(nSV)
        SE = range(nSE)
        V = range(n)

        vertexLabels = set()
        while len(vertexLabels) < p:
          x = random.randrange(n)
          q = random.randrange(nSV)
          vertexLabels.add((x,q))
        for (x,q) in vertexLabels:
          d = random.randint(1,l) * 1.0 / l
          G.addVertexLabel(x, q, d)
          G2.addVertexLabel(x, q, d)
        vertexLabels.clear()

        edges = set()
        while len(edges) < m:
          x = random.randrange(n)
          y = random.randrange(n)
          r = random.randrange(nSE)
          edges.add((x,y,r))
        for (x,y,r) in edges:
          d = random.randint(1,l) * 1.0 / l
          G.addEdge(r, x, y, d)
          G2.addEdge(r, x, y, d)
        edges.clear()

        inst = CompCB(G, withCountingSuccessors)
        P = inst.compCB()
        inst2 = CompCB2(G2, withCountingSuccessors)
        assert P.sortedList() == inst2.compCB()
        print("Done a random test #" + str(k+1) + ", " + 
              str(P.size()) + " blocks.")

#===================================================================

def testPerformance():
  scenarios = [
    [2,100000,1000000,10000,3,1,1],
    [2,100000,1000000,10000,3,2,1],
    [2,100000,1000000,10000,3,10,1],
    [2,100000,1000000,10000,3,1,2],
    [2,100000,1000000,10000,3,1,10],
    [2,100000,1000000,10000,3,10,10],
    [2,100000,2000000,10000,3,1,1],
    [2,100000,3000000,10000,3,1,1],
    [2,100000,4000000,10000,3,1,1],
    [2,100000,5000000,10000,3,1,1],

    [2,100000,100000,1000,10,1,1],
    [2,100000,100000,1000,10,2,1],
    [2,100000,100000,1000,10,10,1],
    [2,100000,100000,1000,10,1,2],
    [2,100000,100000,1000,10,1,10],
    [2,100000,100000,1000,10,10,10],
    [2,100000,200000,1000,10,1,1],
    [2,100000,300000,1000,10,1,1],
    [2,100000,400000,1000,10,1,1],
    [2,100000,500000,1000,10,1,1],

    [2,100000,100000,1000,100,1,1],
    [2,100000,100000,1000,100,2,1],
    [2,100000,100000,1000,100,10,1],
    [2,100000,100000,1000,100,1,2],
    [2,100000,100000,1000,100,1,10],
    [2,100000,100000,1000,100,10,10],
    [2,100000,200000,1000,100,1,1],
    [2,100000,300000,1000,100,1,1],
    [2,100000,400000,1000,100,1,1],
    [2,100000,500000,1000,100,1,1],
  ]

  for withCountingSuccessors in [False, True]:
    print("TESTING WITH withCountingSuccessors =", withCountingSuccessors)  
    for [N,n,m,p,l,nSV,nSE] in scenarios:
      print("TESTING SCENARIO [N,n,m,p,l,nSV,nSE] = ", [N,n,m,p,l,nSV,nSE])

      for k in range(N):
        G = FuzzyGraph()
        SV = range(nSV)
        SE = range(nSE)
        V = range(n)

        vertexLabels = set()
        while len(vertexLabels) < p:
          x = random.randrange(n)
          q = random.randrange(nSV)
          vertexLabels.add((x,q))
        for (x,q) in vertexLabels:
          d = random.randint(1,l) * 1.0 / l
          G.addVertexLabel(x, q, d)
        vertexLabels.clear()

        edges = set()
        while len(edges) < m:
          x = random.randrange(n)
          y = random.randrange(n)
          r = random.randrange(nSE)
          edges.add((x,y,r))
        for (x,y,r) in edges:
          d = random.randint(1,l) * 1.0 / l
          G.addEdge(r, x, y, d)
        edges.clear()

        t1 = time.time()
        inst = CompCB(G, withCountingSuccessors)
        P = inst.compCB()
        t2 = time.time()
        print("Done a random test #" + str(k+1) + 
              ": " + str(t2-t1) + "s, " + 
              str(P.size()) + " blocks.")

#===================================================================

def printResult(rs):
    print("\n".join([" ".join(block) for block in rs]))

if __name__ == "__main__":
    withCountingSuccessors = "-s" in sys.argv or \
                             "--withCountingSuccessors" in sys.argv
    verbose = "-v" in sys.argv or "--verbose" in sys.argv
    naive = "--naive" in sys.argv

    if not naive:
        G = FuzzyGraph()
        G.read()
        inst = CompCB(G, withCountingSuccessors)
        inst.verbose = verbose
        printResult(inst.compCB().sortedList())
    else:
        G = FuzzyGraph2()
        G.read()
        inst = CompCB2(G, withCountingSuccessors)
        printResult(inst.compCB())

#===================================================================

