# author: Linh Anh Nguyen, nguyen@mimuw.edu.pl
# created: 2024-06-10

import sys
from queue import *

class Matrix:
    @staticmethod
    def min(A, B):
        assert type(A) == type(B) == list and len(A) == len(B)
        m = len(A)
        n = len(A[0])
        for i in range(m):
            assert type(A[i]) == type(B[i]) == list and len(A[i]) == len(B[i]) == n
        return [[min(A[i][j], B[i][j]) for j in range(n)] for i in range(m)]

    @staticmethod
    def transposed(A):
        m = len(A)
        n = len(A[0])
        for i in range(m):
            assert type(A[i]) == list and len(A[i]) == n
        return [[A[i][j] for i in range(m)] for j in range(n)]

    @staticmethod
    def print(A):
        print("[", end="")
        n = len(A)
        for i in range(n):
            if i > 0:
                print(" ", end="")
            print(A[i], end="")
            print("," if i < n - 1 else "]")
        
class RLattice: # Residuated lattice [0, 1]
    def __init__(self, kind):
        assert kind in ["G", "L", "P"] # Goedle, Lukasiewicz or Product
        self.kind = kind

    def tnorm(self, x, y):
        if self.kind == "G":
            return min(x, y)
        if self.kind == "L":
            return max(0, x + y - 1)
        if self.kind == "P":
            return x * y

    def residuum(self, x, y):
        if x <= y:
            return 1
        if self.kind == "G":
            return y
        if self.kind == "L":
            return y - x + 1
        if self.kind == "P":
            return y / x

    def biresiduum(self, x, y):
        return min(self.residuum(x, y), self.residuum(y, x))

    def residuum1(self, X, Y):
        return [[self.residuum(x, y) for y in Y] for x in X]

    def biresiduum1(self, X, Y):
        return [[self.biresiduum(x, y) for y in Y] for x in X]

    def composition1(self, X, Y):
        assert len(X) == len(Y)
        return max([self.tnorm(X[i], Y[i]) for i in range(len(X))])
        
    def composition21(self, M, Y):
        return [self.composition1(X, Y) for X in M]

    def composition12(self, X, M):
        return [self.composition1(X, Y) for Y in Matrix.transposed(M)]

    def subset(self, X, Y):
        assert type(X) == type(Y) == list and len(X) == len(Y)
        return min([self.residuum(X[i], Y[i]) for i in range(len(X))])

    def equal(self, X, Y):
        assert type(X) == type(Y) == list and len(X) == len(Y)
        return min([self.biresiduum(X[i], Y[i]) for i in range(len(X))])

class Automaton:
    def __init__(self):
        self.sigma = []
        self.n = None
        self.I = []
        self.F = []
        self.T = {}

    def read(self, filename = None):
        if filename is not None:
            f = open(filename, "r")
        else:
            f = sys.stdin

        self.n = None
        self.T = {}
        
        count = 0
        for line in f:
            line = line.strip()
            if line == "" or line[0] == '%':
                continue
            count += 1
            
            if count == 1:
                self.sigma = line.split()
                for s in self.sigma:
                    self.T[s] = []
            else:
                L = [float(e) for e in line.split()]
                assert self.n == None or count <= 3 + len(self.sigma) * self.n

                if count == 2:
                    self.n = len(L)
                    self.I = L
                elif count == 3:
                    self.F = L
                else:
                    i = (count - 4) // self.n
                    s = self.sigma[i]
                    self.T[s].append(L)

        if filename is not None:
            f.close()

    def __str__(self):
        rs = str(self.sigma) + "\n"
        rs += str(self.I) + "\n"
        rs += str(self.F) + "\n"
        for s in self.T:
            rs += str(self.T[s]) + "\n"
        return rs

    @staticmethod
    def transposed(A):
        At = Automaton()
        At.n = A.n
        At.sigma = A.sigma.copy()
        At.I = A.F.copy()
        At.F = A.I.copy()
        for s in At.sigma:
            At.T[s] = Matrix.transposed(A.T[s])
        return At
        
#===================================================================

def bfbs(A, Ap, k, bis, lattice):
    assert type(A) == type(Ap) == Automaton and A.sigma == Ap.sigma
    assert type(k) == int and k >= 0 or k == "infinity"
    assert type(bis) == bool
    assert type(lattice) == RLattice

    allPairs = set()
    allPairs.add((tuple(A.F), tuple(Ap.F)))
    
    unprocessed = Queue()
    unprocessed.put((A.F, Ap.F))
    
    R = lattice.biresiduum1(A.F, Ap.F) if bis else lattice.residuum1(A.F, Ap.F)

    i = 0
    while k == "infinity" or i < k:
        i += 1
        unprocessed2 = Queue()
        while not unprocessed.empty():
            (F, Fp) = unprocessed.get()
            for s in A.sigma:
                F2  = lattice.composition21(A.T[s], F)
                F2p = lattice.composition21(Ap.T[s], Fp)
                if (tuple(F2), tuple(F2p)) not in allPairs:
                    R = Matrix.min(R, lattice.biresiduum1(F2, F2p) if bis
                                      else lattice.residuum1(F2, F2p))
                    allPairs.add((tuple(F2), tuple(F2p)))
                    unprocessed2.put((F2, F2p))
        unprocessed = unprocessed2
        if unprocessed.empty():
            break

    comparison = 1 # equality/subsethood (depending on bis) between A and Ap
    for (F, Fp) in allPairs:
        d = lattice.composition1(A.I, F)  
        dp = lattice.composition1(Ap.I, Fp)
        comparison = min(comparison, lattice.biresiduum(d, dp) if bis
                                     else lattice.residuum(d, dp))

    return (R, comparison)

#===================================================================
        
if __name__ == "__main__":
    assert len(sys.argv) == 6 or \
           len(sys.argv) == 7 and sys.argv[6] in ["forward", "backward"]
           
    A = Automaton()
    A.read(sys.argv[1])
    Ap = Automaton()
    Ap.read(sys.argv[2])
    k = sys.argv[3] if sys.argv[3] == "infinity" else int(sys.argv[3])
    assert sys.argv[4] in ["true", "True", "T", "1", "false", "False", "F", "0"]
    bis = sys.argv[4] in ["true", "True", "T", "1"]
    lattice = RLattice(sys.argv[5])
    forward = len(sys.argv) == 6 or sys.argv[6] == "forward"

    '''
    print("----------------------\nRead:")
    print("A:"), print(A)
    print("Ap:"), print(Ap)
    print("k =", k)
    print("bis =", bis)
    print("lattice =", lattice.kind)
    print("----------------------")
    '''

    label = "equality:" if bis else "subsethood:"

    if forward:
        (R, comparison) = bfbs(A, Ap, k, bis, lattice)
        Matrix.print(R)
        print(label, comparison)
    else:
        At = Automaton.transposed(A)
        Apt = Automaton.transposed(Ap)
        (Rt, comparison) = bfbs(At, Apt, k, bis, lattice)
        Matrix.print(Rt)
        print(label, comparison)
