(* Original copyrights are below.*)

(* text-widget.sml
 *
 * COPYRIGHT (c) 1991 by AT&T Bell Laboratories.  See COPYRIGHT file for details.
 *
 * A simple text widget: currently this only supports one fixed-width font (9x15).
 *)

structure MyTextWidget :> MY_TEXT_WIDGET =
  struct
    structure W = Widget

    open Geometry EXeneBase W

    fun impossible (f,msg) = raise LibBase.Impossible("TextWidget."^f^": "^msg)

    datatype char_coord = ChrCrd of {col : int, row : int}

    fun min (a : int, b) = if (a < b) then a else b
    fun max (a : int, b) = if (a > b) then a else b

    val fontName = "7x13" (*"9x15"*)
    val pad = 2
    val totPad = pad+pad

  (* Get the character dimensions from a (fixed-width) font *)
    fun fontInfo font = let
	  val {ascent, descent} = Font.fontHt font
	  in
	    (ascent + descent, Font.textWidth font "M", ascent)
	  end

  (* A description of the various size parameters of a text window *)
    datatype text_sz = TSZ of {
	sz : size,
	rows : int, cols : int,
	char_ht : int, char_wid : int, ascent : int
      }

  (* make a text window size descriptor from a window size and font. *)
    fun mkTextSz (sz as SIZE{wid, ht}, font) = let
	  val (charHt, charWid, ascent) = fontInfo font
	  in
	    TSZ{
		sz = sz,
		rows = Int.quot(ht - totPad, charHt),
		cols = Int.quot(wid - totPad, charWid),
		char_ht = charHt, char_wid = charWid, ascent = ascent
	      }
	  end

  (* return true if the character coordinate is in the text window *)
    fun inTextWin (TSZ{rows, cols, ...}, ChrCrd{row, col}) =
	  ((0 <= row) andalso (row < rows)) andalso
	  ((0 <= col) andalso (col < cols))

  (* clip a string to insure that it does not exceed the text length
    fun clipString (TSZ{cols, ...}, col, s) = let
	  val len = String.size s
	  in
	    if ((col + len) <= cols)
	      then s
	      else substring(s, 0, cols-col)
	  end                                                        *)


  (*** The text buffer ***
   * This is a two dimensional array of characters with highlighting information.
   *)
    datatype text_buf = TB of {arr : CharArray.array Array.array}
        fun buildTextBufFromStrs strs rows cols =
              let
                fun loop ([], l) = TB{arr=Array.fromList l}
                  | loop (s::rest, l) =
                     let
                       val arr = CharArray.array(cols, #" ")
                     in
                       (CharArray.copy {src=(CharArray.fromList s), si=0, di=0, len=NONE, dst=arr};
                        loop (rest, arr::l))
                     end
              in
                loop (strs, [])
              end

        fun mapRectExt tb f fromRow toRow fromCol toCol y x =
              let
                fun substr charr i j s =
                     if (i<=j) then
                       (substr charr (i+1) j (s ^ (Char.toString (CharArray.sub (charr, i)))))
                     else
                       s
                fun mapRow row charr =
                     (let
                        fun map_f i last =
                          if (i <= last) then
                            (f {row=(row-y), col=(i-x), s=(substr charr i last "")})
                          else
                            ()
                      in
                        map_f fromCol toCol
                      end)
                val TB {arr} = tb
              in
                if (fromRow <= toRow) then
                  (mapRow fromRow (Array.sub (arr, fromRow));
                   mapRectExt tb f (fromRow+1) toRow fromCol toCol y x)
                else
                  ()
              end

        fun findStr (TB{arr}) crow ccol max_col =
             let
               val _ = (Int.toString crow) ^ "<-row, col->" ^ (Int.toString ccol)
               fun firstNoSpace charr i =
                    if i = 0 then 0
                    else if (CharArray.sub (charr, i-1)) = #" " then i
                         else firstNoSpace charr (i-1)
               fun lastNoSpace  charr i =
                    if i = max_col then max_col
                    else if (CharArray.sub (charr, i+1)) = #" " then i
                         else lastNoSpace charr (i+1)
               fun copyChars charr i j l = 
                    if i>j then l
                    else copyChars charr (i+1) j (l @ [(CharArray.sub (charr, i))])
               val charr = Array.sub (arr, crow)
             in
               if ((CharArray.sub (charr, ccol)) = #" ") then
                 ""
               else
                 String.implode (copyChars charr (firstNoSpace charr ccol) (lastNoSpace charr ccol) [])
             end


  (*** The text window ***
   * This is a dumb text window that supports drawing text in normal and
   * highlighted mode text.
   *)
    datatype text_win = TW of {
	root : Widget.root,
	win : EXeneBase.window,
	font : EXeneBase.font,
	draw_text : {col : int, row : int, s : string} -> unit,
	clear_blk : {start_row : int, end_row : int, size:text_sz} -> unit
       }

      local
	open Drawing

      (* Blt a block of text within a line *)
	fun charBlt (win, TSZ{char_ht, char_wid, sz=SIZE{wid, ...}, ...}) = let
	      val pixelBlt = pixelBltEvt (drawableOfWin win) defaultPen
	      fun blt {row, from, to, nchars} = let
		    val Y = (char_ht * row) + pad
		    in
		      pixelBlt {
			  src = WSRC win,
			  src_rect = RECT{
			      x = pad + from*char_wid, y = Y,
			      wid = nchars*char_wid, ht = char_ht
			    },
			  dst_pos = PT{x=pad + to*char_wid, y=Y}
			}
		    end
	      in
		blt
	      end (* charBlt *)

      (* Blt a block of text by lines *)
	fun lineBlt (win, TSZ{char_ht, char_wid, sz=SIZE{wid, ...}, ...}) = let
	      val pixelBlt = pixelBltEvt (drawableOfWin win) defaultPen
	      val textWid = wid - totPad
	      fun blt {from, to, nlines} = let
		    val fromY = (char_ht * from) + pad
		    val toY = (char_ht * to) + pad
		    in
		      pixelBlt {
			  src = WSRC win,
			  src_rect = RECT{
			      x = pad, y = fromY,
			      wid = textWid, ht = (char_ht * nlines)
			    },
			  dst_pos = PT{x=pad, y=toY}
			}
		    end
	      in
		blt
	      end (* lineBlt *)

      (* a stipple pattern for the cursor *)
	val cursorStippleData = (16, [[
                "0x8888", "0x2222", "0x1111", "0x4444",
                "0x8888", "0x2222", "0x1111", "0x4444",
                "0x8888", "0x2222", "0x1111", "0x4444",
                "0x8888", "0x2222", "0x1111", "0x4444"
              ]])

      in

    (* make a text window of the given size *)
      fun mkTextWin (root, win, font, size) = let
	    val TSZ{sz=SIZE{wid, ht}, rows, cols, char_ht, char_wid, ascent} = size
	    val (pen, normalStipple) = let
		  val black = blackOfScr(EXeneWin.screenOfWin win)
		  val white = whiteOfScr(EXeneWin.screenOfWin win)
                  val stipple = tile root "lightGray"
		  in
		    (newPen[PV_Foreground black, PV_Background white],
		     newPen[PV_Foreground black, PV_FillStyle_Stippled, PV_Stipple stipple])
		  end
	    fun ccToPt {row, col} =
		  {x = (col * char_wid) + pad, y = (row * char_ht) + pad}
	    fun drawText (clear, draw) {row, col, s} = let
		  val {x, y} = ccToPt {row=row, col=col}
		  in
		    clear (RECT{x=x, y=y, wid=char_wid*(String.size s), ht=char_ht});
		    draw (PT{x=x, y=y+ascent}, s)
		  end

	    val clrArea = clearArea (drawableOfWin win)

	    fun clearBlk {start_row, end_row, size} =
                let
                  val TSZ{sz=SIZE{wid, ht}, char_ht, ...} = size
		  val {x, y} = ccToPt {row=start_row, col=0}
		in
		    clrArea (RECT{
			x=x, y=y, wid=(wid-totPad), ht=(end_row - start_row)*char_ht
		      })
		end
	    in
	      TW{
		  root = root,
		  win = win,
		  font = font,
		  draw_text = drawText (clrArea, drawString (drawableOfWin win) pen font),
		  clear_blk = clearBlk
		}
	    end (* mkTextWin *)

      end (* local *)


  (*** The internal text widget state ***
   * The internal state of the text widget consists of the current size, a text
   * buffer, a text window and a cursor.
   *)
    datatype text = TXT of {
	size : text_sz,
        row: int,
        col: int,
        all_rows: int, 
        all_cols: int,
	txt_buf : text_buf,
	txt_win : text_win
      }

  (* redraw damaged lines (but not the cursor) *)
  fun redrawText (TXT{size, txt_buf, txt_win, col, row, all_rows, all_cols}) damage =
       let
         val TSZ {sz, rows, cols, char_ht, char_wid, ...} = size
         fun splitDamage [] = (1,0,0,0)
           | splitDamage (r::rest) =
              let
                fun rect2coord (RECT{x, y, wid, ht}) =
                  let
                    val topLn = Int.quot (y - pad, char_ht)
                    val botLn = Int.quot ((y - pad) + ht + (char_ht-1), char_ht)
                    val minC = Int.quot (x - pad, char_wid)
                    val maxC = Int.quot ((x - pad) + wid + (char_wid-1), char_wid)
                  in
                    (topLn, botLn, minC, maxC)
                  end
                fun split c [] = c
                  | split (fr, tr, fc, tc) (r::rest) =
                     let
                       val (fr', tr', fc', tc') = rect2coord r
                     in
                       (Int.min (fr, fr'),
                        Int.max (tr, tr'),
                        Int.min (fc, fc'),
                        Int.max (tc, tc'))
                     end
              in
                split (rect2coord r) rest
              end
         val (fr, tr, fc, tc) = splitDamage damage
         val TW {draw_text, clear_blk, ...} = txt_win
         val fromRow = row + fr
         val toRow   = min (all_rows - 1, row + tr)
         val fromCol = col + fc
         val toCol   = min (all_cols - 1, col + tc)
       in
         mapRectExt txt_buf draw_text fromRow toRow fromCol toCol row col
       end

  fun redrawAllText (TXT{size, txt_buf, txt_win, col, row, all_rows, all_cols}) =
       let
         val TSZ {sz, rows, cols, char_ht, char_wid, ...} = size
         val TW {draw_text, clear_blk, ...} = txt_win
         val fromRow = row
         val toRow   = min (all_rows - 1, rows + row - 1)
         val fromCol = col
         val toCol   = min (all_cols - 1, cols + col - 1)
       in
         (clear_blk {start_row=0, end_row=rows, size=size};
         mapRectExt txt_buf draw_text fromRow toRow fromCol toCol row col)
       end

  (* resize the text buffer and text window *)
    fun resize (TXT{txt_buf, txt_win, row, col, all_rows, all_cols, ...}, font, RECT{wid, ht, ...}) =
        let
	  val newSize = mkTextSz(SIZE{wid=wid, ht=ht}, font)
          val TSZ {rows, cols, ...} = newSize
          val new_row = if (rows + row - all_rows > 0) then max (0, all_rows - rows)
                        else row
          val new_col = if (cols + col - all_cols > 0) then max (0, all_cols - cols)
                        else col
	in
	  TXT {size = newSize,
               txt_buf = txt_buf,
	       txt_win = txt_win,
               row = new_row,
               col = new_col,
               all_rows = all_rows,
               all_cols = all_cols
	      }
	end (* resize *)


  (* scroll the text from line "from" up "n" lines. *)
    fun scrollUp txt =
        let
          val TXT {size, row, col, all_rows, all_cols, txt_buf, txt_win} = txt
          val TSZ {rows, cols, ...} = size
          val new_row = if (row > 0) then
                          row - 1
                        else
                          row
          val new_txt = TXT {size=size, row=new_row, col=col, all_rows=all_rows, all_cols=all_cols, txt_buf=txt_buf, txt_win=txt_win}
        in
          (if (not (new_row = row)) then redrawAllText new_txt else ();
           new_txt)
        end


  (* scroll the text starting at line "from" down "n" lines. *)
    fun scrollDown txt =
        let
          val TXT {size, row, col, all_rows, all_cols, txt_buf, txt_win} = txt
          val TSZ {rows, cols, ...} = size
          val new_row = if (row + rows < all_rows) then
                          row + 1
                        else
                          row
          val new_txt = TXT {size=size, row=new_row, col=col, all_rows=all_rows, all_cols=all_cols, txt_buf=txt_buf, txt_win=txt_win}
        in
          (if (not (new_row = row)) then redrawAllText new_txt else ();
           new_txt)
        end


    fun scrollLeft txt =
        let
          val TXT {size, row, col, all_rows, all_cols, txt_buf, txt_win} = txt
          val TSZ {rows, cols, ...} = size
          val new_col = if (col > 0) then
                          col - 1
                        else
                          col
          val new_txt = TXT {size=size, row=row, col=new_col, all_rows=all_rows, all_cols=all_cols, txt_buf=txt_buf, txt_win=txt_win}
        in
          (if (not (new_col = col)) then redrawAllText new_txt else ();
           new_txt)
        end


    fun scrollRight txt =
        let
          val TXT {size, row, col, all_rows, all_cols, txt_buf, txt_win} = txt
          val TSZ {rows, cols, ...} = size
          val new_col = if (col + cols < all_cols) then
                          col + 1
                        else
                          col
          val new_txt = TXT {size=size, row=row, col=new_col, all_rows=all_rows, all_cols=all_cols, txt_buf=txt_buf, txt_win=txt_win}
        in
          (if (not (new_col = col)) then redrawAllText new_txt else ();
           new_txt)
        end

  (* makes empty line and draw "str" at "pos" in normal mode *)
  (* col always 0*)
    fun writeStrs (txt, strs) =
        let
          fun countRowsCols ([], rows, cols) = (rows, cols)
            | countRowsCols (s::rest, rows, cols) =
                if List.length s > cols then
                  countRowsCols (rest, rows + 1, List.length s)
                else
                  countRowsCols (rest, rows + 1, cols)

          val TXT {size, txt_buf, txt_win, ...} = txt
          val (new_rows, new_cols) = countRowsCols (strs, 0, 0)
          val (new_all_rows, new_all_cols) = (new_rows, new_cols)
          val tb_new = buildTextBufFromStrs strs new_all_rows new_all_cols
          val new_txt = TXT {size=size, row=0, col=0, all_rows=new_all_rows, all_cols=new_all_cols, txt_buf=tb_new, txt_win=txt_win}
        in
          (redrawAllText new_txt; new_txt)
        end

        fun findClickedString (TXT{size=TSZ{char_ht, char_wid, ...}, txt_buf, all_rows, all_cols, row, col, ...}) (PT{x, y}) =
             let
               val crow = Int.quot (y - pad, char_ht)
               val ccol = Int.quot (x - pad, char_wid)
             in
               if (crow+row < all_rows) andalso (ccol+col < all_cols) then
                 findStr txt_buf (crow+row) (ccol+col) (all_cols-1)
               else
                 ""
             end


  (*** The text widget ***
   * The text widget is represented by a request/reply pair of communication
   * channels.
   *)

    datatype req_msg
      = ScrollUp
      | ScrollDown
      | ScrollLeft
      | ScrollRight
      | WriteStrs of string list

    datatype text_widget = TW of {
	widget : widget,
	cmd : req_msg -> unit,
        onClickEvt: string CML.event
      }

    fun mkStrs n =
         let
           fun add l 0 = l
             | add l m = add ([#" "]::l) (m-1)
         in
           add [] n
         end
         
    fun strsToCharLists strs =
         let
           fun stc [] l = l
             | stc (s::rest) l = stc rest ([String.explode s] @ l)
         in
           stc strs []
         end

  (* create a new text widget *)
    fun mkTextWidget root {rows : int, cols : int} = let
	  val rows = max(rows, 1) and cols = max(cols, 1)
	  val reqCh = CML.channel()
          val onClickCh = CML.channel()
	  val reqEvt = CML.recvEvt reqCh
          val onClickEvt = CML.recvEvt onClickCh
	  val font = Font.openFont (displayOf root) fontName
	  val (charHt, charWid, _) = fontInfo font
	  fun realize {env, win, sz} = let
		open Interact
		val InEnv{ci, co, m, ...} = ignoreKey env
		val tsz = mkTextSz (sz, font)
                val TSZ {sz, rows, cols, ...} = tsz
		val text = TXT{
			size = tsz,
                        row = 0,
                        col = 0,
                        all_rows = rows,
                        all_cols = cols,
			txt_buf = buildTextBufFromStrs (mkStrs cols) rows cols,
			txt_win = mkTextWin (root, win, font, tsz)
		      }
		fun server text = let
                      fun handleM msg = (case (msgBodyOf msg) of
                              MOUSE_FirstDown {pt,but,...} => (
                                let
                                  val s = findClickedString text pt
                                in
                                  (CML.sendPoll (onClickCh, s);
                                   server text)
                                end)
                            | _ => (server text))
		      fun handleCI msg = (case (msgBodyOf msg)
			   of (CI_Redraw damage) => (
				redrawText text damage;
				server text)
			    | (CI_Resize newR) =>
				server (resize (text, font, newR))
			    | (CI_OwnDeath) => ()
			    | _ => impossible("realize",
				  "[TextWidget: unexpected CI message]"))
		      fun handleReq (ScrollUp) =
			    server (scrollUp text)
			| handleReq (ScrollDown) =
			    server (scrollDown text)
			| handleReq (ScrollLeft) =
			    server (scrollLeft text)
			| handleReq (ScrollRight) =
			    server (scrollRight text)
			| handleReq (WriteStrs strs) =
			    server (writeStrs (text, strsToCharLists strs))
		      in
			CML.sync (CML.choose [
			    CML.wrap(ci, handleCI),
			    CML.wrap(m, handleM),
			    CML.wrap(reqEvt, handleReq)
			  ])
		      end
		in
		  XDebug.xspawn("textWidgetServer", fn () => server text);
		  ()
		end
	  in
	    TW{
		widget = mkWidget{
		    root = root,
                    args= fn () => {background = NONE},
		    boundsOf = fn () => {
			  x_dim = DIM{
			      base=totPad, incr=charWid, min=1, nat=cols, max=NONE
			    },
			  y_dim = DIM{
			      base=totPad, incr=charHt, min=1, nat=rows, max=NONE
			    }
		        },
		    realize = realize
		  },
		cmd = (fn req => CML.send(reqCh, req)),
                onClickEvt = onClickEvt
	      }
	  end (* mkTextWidget *)

    fun widgetOf (TW{widget, ...}) = widget

    fun scrollUp    (TW{cmd, ...}) = cmd (ScrollUp)
    fun scrollDown  (TW{cmd, ...}) = cmd (ScrollDown)
    fun scrollLeft  (TW{cmd, ...}) = cmd (ScrollLeft)
    fun scrollRight (TW{cmd, ...}) = cmd (ScrollRight)

    fun insertText (TW{cmd, ...}) strs = cmd (WriteStrs strs)

    fun onClickEvt (TW{onClickEvt, ...}) = onClickEvt

  end (* MyTextWidget *)


structure ScrollableText: SCROLLABLE_TEXT =
struct

    datatype scroll_text = STXT of {
                             box: Box.box_layout,
                             textWidg: MyTextWidget.text_widget
                           }

    fun mkScrollableTextWidget  root {rows : int, cols : int} =
      let        
        val textWidg = MyTextWidget.mkTextWidget root {rows=rows, cols=cols}
        val textWidgF = Frame.mkFrame {color=SOME (Widget.EXB.blackOfScr (Root.screenOf root)),
                                       width=1,
                                       widget=MyTextWidget.widgetOf textWidg}
        val hScroll = Scrollbar.mkHScrollbar root {color=NONE, sz=5}
        val vScroll = Scrollbar.mkVScrollbar root {color=NONE, sz=5}
        val box_top    = Box.HzCenter [Box.WBox (Frame.widgetOf textWidgF), Box.WBox (Scrollbar.widgetOf vScroll)]
        val box_bottom = Box.HzCenter [Box.WBox (Scrollbar.widgetOf hScroll), Box.Glue {nat=5,min=5,max=SOME(5)}]
        val layout = Box.mkLayout root (Box.VtCenter [box_top, box_bottom])

        fun handleVScroll scroll_evt =
             case (scroll_evt) of
               Scrollbar.ScrUp (r) => MyTextWidget.scrollUp textWidg
             | Scrollbar.ScrDown (r) => MyTextWidget.scrollDown textWidg
             | Scrollbar.ScrStart (r) => ()
             | Scrollbar.ScrMove (r) => ()
             | Scrollbar.ScrEnd (r) => ()

        fun handleHScroll scroll_evt =
             case (scroll_evt) of
               Scrollbar.ScrUp (r) => MyTextWidget.scrollLeft textWidg
             | Scrollbar.ScrDown (r) => MyTextWidget.scrollRight textWidg
             | Scrollbar.ScrStart (r) => ()
             | Scrollbar.ScrMove (r) => ()
             | Scrollbar.ScrEnd (r) => ()

        fun loop () =
          let
          in
            CML.sync (CML.choose [
	                    CML.wrap(Scrollbar.evtOf hScroll, handleHScroll),
                            CML.wrap(Scrollbar.evtOf vScroll, handleVScroll)
                                 ]);
            loop ()
          end

      in
        XDebug.xspawn("scrollableText", fn () => loop ());
        STXT {
          box = layout,
          textWidg = textWidg
        }
      end (*mkScrollableTextWidget*)

    fun widgetOf (STXT{box, ...}) = Box.widgetOf box

    fun insertText (STXT{textWidg, ...}) strs = MyTextWidget.insertText textWidg strs

    fun onClickEvt (STXT{textWidg, ...}) = MyTextWidget.onClickEvt textWidg

end (* ScrollableText *)
