structure ObligsMgrShell:
    sig
      structure W: WIDGET
      structure G: MY_GENERATOR
      type obmgr_shell
      val mkObligsMgrShell: (W.root * W.view * W.arg list) ->
                            G.keyinfo list ->
                            G.proofinfo list ->
                            (unit -> unit) ->
                            string
                            -> obmgr_shell
      val shellOf: obmgr_shell -> Shell.shell
    end =
struct
  structure W = Widget
  structure S = SigsShell
  structure P = StepsShell
  structure G = MyGenerator
  structure A = Attrs
  
  type sigs = (S.sigs_shell option * string * string list) list
  type oblig = (string list * string list * string list * bool)
  type steps = (P.steps_shell option * string * int * oblig list) list
  
  fun prepareSigs [] = []
    | prepareSigs (G.KI(n, b)::rest) = (NONE, n, b)::(prepareSigs rest)
  
  fun prepareSteps [] = []
    | prepareSteps (G.PI(k, n, obs)::rest) =
      let
        val n = k ^ " " ^ n
        fun countNotDone ([], i) = i
          | countNotDone ((l,r,p,d)::rest, i) =
             let
               val i' = if d then 0 else 1
             in
               countNotDone (rest, i + i')
             end
      in
        (NONE, n, countNotDone (obs, 0), obs)::(prepareSteps rest)
      end

  type obmgr_shell = Shell.shell * sigs * steps
      
  fun mkObligsMgrShell (root, view, args) ks ps do_quitMgrShell fileName =
    let
      val dummy_sigs  = prepareSigs ks
      val sigs  = ref dummy_sigs
      val dummy_steps = prepareSteps ps
      val steps = ref dummy_steps


      fun progress_info cnt obs =
            let
                  val all = List.length obs
                  val done_str = Int.toString (all - cnt)
                  val done_str = if cnt = all then "nothing"
                                 else if cnt = 0 then "all"
                                      else done_str
                  val all_str = Int.toString all
                  val s_info = 
                       " (" ^ done_str ^ " of " ^ all_str ^ " done)"
            in
              s_info
            end
      fun updateFirst (SOME (shel)) ((l, r, p, ok)::rest) =
            let
              val (p', ok')  = P.changes shel
              val n = if ok then
                         if ok' then  0
                         else         1
                      else
                         if ok' then ~1
                         else         0
            in
              ((l, r, p', ok')::rest, n)
            end
        | updateFirst _ obs = (obs, 0)

      fun do_next s f choose =
          let
            fun ble s [] = []
              | ble s ((e as (sh, s', cnt, oldObs))::rest) =
                  if s = s' then
                    let
                      val (obs, n) = updateFirst sh oldObs
                      val _ = f (progress_info (cnt+n) obs)
                    in
                      case choose obs of
                        NONE => (sh, s', cnt+n, obs)::rest
                      | SOME (ob, obs') =>
                           case sh of
                             SOME(shel) =>
                                ((SOME (P.show shel (s, ob)), s', cnt+n, obs')::rest)
                           | NONE => e::rest
                    end
                  else e::(ble s rest)
                  
            val new_steps = ble s (!steps)
          in
            steps := new_steps
          end

      fun do_next_oblig s f =
          let
            fun next [] = NONE
              | next [ob] = NONE
              | next (ob::ob'::rest) = SOME (ob', ob'::rest @ [ob])
          in
            do_next s f next
          end
      
      fun do_next_to_proof s f =
          let
            fun next [] = NONE
              | next (e::obs) =
                 let
                   fun search [] _ = NONE
                     | search ((ob as (l, r, p, ok))::rest) acc =
                        if ok then search rest (acc @ [ob])
                        else SOME (ob, ob::rest @ acc)
                 in
                   search obs [e]
                 end
          in
            do_next s f next
          end

      fun closeSigsShell s = 
          let
            fun destroy s [] = []
              | destroy s ((e as (sh, s', body))::rest) =
                  if s = s' then
                      case sh of
                        SOME(shel) =>
                             (Shell.destroy (S.shellOf shel);
                              (NONE, s', body)::rest)
                      | NONE => e::rest
                  else e::(destroy s rest)
          in
            sigs := destroy s (!sigs)
          end

      fun closeStepsShell s = 
          let
            fun destroy s [] = []
              | destroy s ((e as (sh, s', cnt, obs))::rest) =
                  if s = s' then
                      case sh of
                        SOME(shel) =>
                          (let
                             val (obs', n) = updateFirst sh obs
                           in
                             (Shell.destroy (P.shellOf shel);
                              (NONE, s', cnt+n, obs')::rest)
                           end)
                      | NONE => e::rest
                  else e::(destroy s rest)
          in
            steps := destroy s (!steps)
          end

      fun quitMgrShell () =
          let
            fun quitS [] = []
              | quitS ((sh, s, bdy)::rest) =
                   case sh of
                     SOME (shel) =>
                             (Shell.destroy (S.shellOf shel);
                              (NONE, s, bdy)::(quitS rest))
                   | NONE => (sh, s, bdy)::(quitS rest)
                       
            fun quitP [] = []
              | quitP ((sh, s, cnt, obs)::rest) =
                   case sh of
                     SOME (shel) =>
                          (let
                             val (obs', n) = updateFirst sh obs
                           in
                             (Shell.destroy (P.shellOf shel);
                              (NONE, s, cnt+n, obs')::(quitP rest))
                           end)
                   | NONE => (sh, s, cnt, obs)::(quitP rest)
          in
            sigs  := quitS (!sigs);
            steps := quitP (!steps);
            do_quitMgrShell ()
          end

      fun present_sig s =
          let
            fun present s [] = []
              | present s ((e as (sh, s', body))::rest) =
                 if s = s' then
                   let
                     val args = args @
                          [(A.attr_title, A.AV_Str ("GPP - Signature "^s)),
                           (A.attr_title, A.AV_Str ("GPP - Signature "^s))]
                     val sh' = case sh of
                                 SOME (sshell) =>
                                    (Shell.unmap (S.shellOf sshell);
                                     Shell.map (S.shellOf sshell);
                                    (* Shell.setWMHints (S.shellOf sshell)
                                       (Shell.mkHints { size_hints=[],
                                                        wm_hints=[W.EXW.ICCC.HINT_NormalState]});*)
                                     sh)
                               | NONE =>
                                    SOME (S.mkSigsShell
                                              (root, view, args)
                                              (s, body)
                                              (fn () => closeSigsShell s)
                                              (present_sig))
                   in
                     (sh', s', body)::rest
                   end
                 else e::(present s rest)
          in
            sigs := present s (!sigs)
          end

      fun present_step s =
          let
            fun getFirst [] = ([], [], [], false)
              | getFirst (e::rest) = e

            fun present s [] = []
              | present s ((e as(sh, s', cnt, obs))::rest) =
                 if s = s' then
                   let
                     val args = args @
                           [(A.attr_title, A.AV_Str ("GPP - Proof Obligations for "^s)),
                            (A.attr_title, A.AV_Str ("GPP - Proof Obligations for "^s))]
                     val sh' = case sh of
                                 SOME (sshell) =>
                                    (Shell.unmap (P.shellOf sshell);
                                     Shell.map (P.shellOf sshell);
                                     sh)
                               | NONE =>
                                    SOME (StepsShell.mkStepsShell
                                           (root, view, args)
                                           (s, getFirst obs)
                                           (fn f => fn () => do_next_oblig s f)
                                           (fn f => fn () => do_next_to_proof s f)
                                           (fn () => closeStepsShell s)
                                           (present_sig)
                                           (progress_info cnt obs))
                   in
                     (sh', s', cnt, obs)::rest
                   end
                 else e::(present s rest)
          in
            steps := present s (!steps)
          end

      fun do_sig () =
          let
            open SimpleMenu
            fun mkItem (_, s, _) = 
                  MenuItem (s, fn () => present_sig s)
          in
            MENU (map mkItem (!sigs))
          end
            
      fun do_step () =
          let
            open SimpleMenu
            fun mkItem (_, s, cnt, obs) = 
                let
                  val all = List.length obs
                  val all_str = Int.toString all
                  val s_info = s ^ " (" ^ all_str ^ ")"
                in
                  MenuItem (s_info, fn () => present_step s)
                end
          in
            MENU (map mkItem (!steps))
          end

      val quitBttn = Button.textCmd
                       (root, view, [(A.attr_label, A.AV_Str "quit")])
                       quitMgrShell

      val (sigsMenuBttn, sigsMenuEvt) =
                       MenuButton.mkMenuButton root ("Signatures", do_sig())
      val (stepsMenuBttn, stepsMenuEvt) =
                       MenuButton.mkMenuButton root ("Proof obligations for (how many)", do_step())  

      fun loop () = loop ((CML.select [sigsMenuEvt, stepsMenuEvt]) ())

      val layout = Box.mkLayout root (Box.VtCenter [
                                  Box.HzTop [
                                             (*Box.WBox (Shape.mkRigid sigsMenuBttn),*)
                                             Box.WBox (sigsMenuBttn),
                                             Box.Glue {nat=5,min=5,max=NONE},
                                             Box.WBox (stepsMenuBttn)
                                            ],
                                  Box.WBox (Divider.horzDivider (root, view, [])),
                                  Box.WBox (Button.widgetOf quitBttn)
                                                   ])
      val args = args @ [(A.attr_title, A.AV_Str ("GPP - Sigs&Obligs for "^fileName)),
                         (A.attr_title, A.AV_Str ("GPP - Sigs&Obligs for "^fileName))]
      val shell = Shell.shell (root, view, args) (Box.widgetOf layout)
    in
      Shell.init shell;
      CML.spawn loop;
      (shell, !sigs, !steps)
    end
    
    fun shellOf (shell, _, _) = shell
end;
