signature MY_GENERATOR =
sig                     (* name,   text of    *)
  datatype keyinfo = KI of string * string list
                          (* kind,    name,      implementation,specification,proof[not used],done*)
  datatype proofinfo = PI of string * string * ((string list * string list * string list * bool) list)
  
  val generate: string list -> (keyinfo list * proofinfo list) option 
end;

structure MyGenerator : MY_GENERATOR =
struct
  datatype keyinfo = KI of string * string list
  datatype proofinfo = PI of string * string * ((string list * string list * string list * bool) list)
  
  datatype result = FOUND | NOTFOUND
  
  fun generate ([]) = SOME ([], [])
    | generate (fileName::rest) = 
      let
        fun cutEmpty [] = []
          | cutEmpty (e::rest) = if e = "\n" then cutEmpty rest
                                 else e::(cutEmpty rest)
        val ls = cutEmpty (ObligsStream.run fileName)
        val res = obligsStream2infoList ls
      in
        case res of
          NONE => NONE
        | SOME (ks, ps) =>
            let
              val res' = generate rest
            in
              case res' of
                NONE => NONE
              | SOME (ks', ps') =>
                   SOME (ks @ ks', ps @ ps')
            end
      end
      
  and obligsStream2infoList ls =
      let
        val step_ms = ["***SIGNATURE\n", "***TOPDECLARATION\n"]
      
        fun tail (e::rest) = rest
        
        fun head (e::rest) = e
      
        fun is_in e [] = false
          | is_in e (e'::rest) =
              if e = e' then true
              else is_in e rest

        fun cutLn s = 
              if (substring (s, (size s) - 1, 1)) = "\n"
              then substring (s, 0, (size s) - 1)
              else s
              
        fun cutLns [] = []
          | cutLns (s::rest) = (cutLn s)::(cutLns rest)

        fun takeUnless [] ps _ = ([], ps, NOTFOUND)
          | takeUnless (e::rest) ps ms =
               if is_in e ms then (e::rest, ps, FOUND)
               else takeUnless rest (ps @ [e]) ms

        fun parseSig ("***SIGNATURE\n"::name::body) =
            let
              val (rest, bdy, res) = takeUnless body [] ["***END_OF\n"]
            in
              ([KI(cutLn name, cutLns bdy)], [], tail rest)
            end

        fun sameLeftRight ([], []) = true
          | sameLeftRight (s::rest, s'::rest') = (s = s') andalso
                                                 sameLeftRight (rest, rest')
          | sameLeftRight _ = false

        fun parseOblig ls =
            let
              val (ls', left, res)  = takeUnless (tail ls) [] ["***RIGHT\n"]
              val (ls'', right, res') = takeUnless (tail ls') [] ["***OBLIG_END\n"]
              val (cleft, cright) = (cutLns left, cutLns right)
              val ob = (cleft, cright, [], sameLeftRight (cleft, cright))
            in
              case head ls'' of
                "***OBLIG\n" => (ob, ls'')
              | other => (ob, tail ls'')
            end

        fun parseObligs (obs, ls) =
             if (head ls) = "***OBLIG\n" then
               let
                 val (ob, rest) = parseOblig (tail ls)
               in
                 parseObligs (obs @ [ob], rest)
               end
             else (obs, ls)

        fun parseStep (txt::kind::name::obligs) =
            let
              val (obs, rest) = parseObligs ([], obligs)
            in
              ([], [PI (cutLn kind, cutLn name, obs)], rest)
            end

        fun parseStepOrSig ls =
             case head ls of
               "***SIGNATURE\n" =>  parseSig ls
             | other => parseStep ls
             
        fun parseStepsAndSigs (kis, pis,  ls) =
            let
              val (kis', pis', ls') = parseStepOrSig ls
              val (nexts, prevs, res) = takeUnless ls' [] step_ms
            in
              case res of
                FOUND =>
                  parseStepsAndSigs (kis @ kis', pis @ pis', nexts)
              | NOTFOUND =>
                  (kis @ kis', pis @ pis', ls')
            end

        val (nexts, prevs, res) = takeUnless ls [] step_ms
        val (kis, pis, rest) = case res of
                                 NOTFOUND => ([], [], prevs)
                               | FOUND => parseStepsAndSigs ([], [], nexts)

        fun evalOK ls =
            let
              val (nexts, prevs, res) = takeUnless ls [] ["val it = () : unit\n"]
            in
              case res of
                FOUND => true
              | NOTFOUND => false
            end

      in
        if evalOK rest then SOME (kis, pis)
        else NONE
      end
end;
