signature ELEM =
sig
   type id
   val  equal: id * id -> bool

axiom forall i =>                                                      
        equal (i, i)
  and forall (i, i') =>                                                
        equal (i, i') implies  equal (i', i)
  and forall (i, i', i'') => 
        equal (i, i') andalso equal (i', i'') implies 
          equal (i, i'')
end;

signature NODE = 
sig
  include ELEM
end;

signature NONDIRECTED_EDGE = 
sig
   include ELEM
   structure Node: NODE
   val left:  id -> Node.id
   val right: id -> Node.id
   val mkEdge: Node.id * Node.id -> id
   val are_connected: id * id -> bool

axiom forall (i, i') =>                                                
        Node.equal (left (mkEdge (i, i')), i) andalso
        Node.equal (right (mkEdge (i, i')), i')

axiom forall (e, e') =>                                                
        equal (e, e')
        iff
        ((Node.equal (left e, left e') andalso Node.equal (right e, right e'))
         orelse
         (Node.equal (left e, right e') andalso Node.equal (right e, left e')))

axiom forall (e, e') =>                                                
        are_connected (e, e')
        iff
        (exists (ee, ee') =>
           equal (e, ee) andalso equal (e', ee') andalso
           Node.equal (right ee, left ee'))
end;

signature SET =
sig
   type set
   structure Elem: ELEM
   val empty: set
   val equal: set * set-> bool
   val is_in: Elem.id * set -> bool
   val add: Elem.id * set -> set
   val remove: Elem.id * set -> set
   val count: set -> int
   val is_empty: set -> bool
   val foldL : ('b * 'b -> bool) -> (Elem.id -> 'b -> 'b) -> 'b -> set -> 'b

axiom forall (i, i', s, s') =>                                         (*A1*)
        Elem.equal (i, i') andalso equal (s, s') implies
          equal (add (i, s), add (i', s'))
          andalso
          equal (remove (i, s), remove (i', s'))

axiom forall (i, s) =>                                                 (*A2*)
        equal (s, empty) implies
          not (is_in (i, s))
  and forall (s, i) =>                                                 (*A3*)
        is_in (i, s)
        iff
        (exists s' =>
           not (is_in (i, s'))
           andalso
           equal (s, add (i, s')))

axiom forall (s, s') =>                                                (*A4*)
       equal (s, s')
       iff
       (forall i => 
           is_in (i, s) iff is_in (i, s')) 

axiom forall (i, s) =>                                                 (*A5*)
        is_in (i, s) implies 
          equal (add (i, s), s)

axiom forall (i, s) =>                                                 (*A6*)
        not (is_in (i, s)) implies 
          equal (remove (i, s), s)

axiom forall (i, s) =>                                                 (*A7*)
        not (is_in (i, remove (i, s)))

axiom forall (i, i', s) =>                                             (*A8*)
        not (Elem.equal (i, i')) implies
        ((is_in (i, s) iff is_in (i, add (i', s)))
         andalso
         (is_in (i, s) iff is_in (i, remove (i', s))))
        
axiom forall s =>                                                      (*A9*)
        is_empty s iff equal (s, empty)

axiom forall s =>                                                      (*A10*)
        equal (s, empty) implies 
          count s = 0
  and forall (s, i) =>                                                 (*A11*)
        is_in (i, s) implies
          count s = count (remove (i, s)) + 1

axiom forall (eq, f, b, s) =>                                          (*A12*)
        equal (s, empty) implies
          eq (foldL eq f b s, b)
  and forall (eq, f, b, s, e) =>                                       (*A13*)
        is_in (e, s) implies
            eq (foldL eq f b s, foldL eq f (f e b) (remove (e, s)))
end;

signature SET_OF_NODES =
sig
   include SET
   structure Node: NODE
   sharing Elem = Node
end;

signature SET_OF_EDGES =
sig
   include SET
   structure Edge: NONDIRECTED_EDGE
   sharing Elem = Edge
end;

signature SET_OF_SETS_OF_NODES =
sig
   include SET
   structure Set: SET_OF_NODES
   sharing type Elem.id = Set.set
   
   structure Node : NODE
   sharing Node = Set.Node
   
   val is_elem_in: Node.id * set -> bool

axiom forall (i, ss) =>                                                
        is_elem_in (i, ss)
        iff
        (exists s =>
           is_in (s, ss)
           andalso
           Set.is_in (i, s))
end;


signature CYCLE_LIST =
sig
  include SET
  val are_neighbours: Elem.id * Elem.id * set -> bool

  local
      val orderNo: Elem.id * set -> int
    axiom forall (i, i', s) =>
            Elem.equal (i, i') implies
              orderNo (i, s) = orderNo (i', s)
    axiom forall (i, s) =>
            is_in (i, s) iff (orderNo (i, s) > 0)
    axiom forall (i, i', s) =>
            is_in (i, s) andalso not (is_in (i', s)) implies
              orderNo (i, s) < orderNo (i', add (i', s))
  in
    axiom forall (s, s') =>
            equal (s, s')
            iff
            forall (i, i') =>
              are_neighbours (i, i', s) iff are_neighbours (i, i', s')

    axiom forall (i, i', s) =>
            are_neighbours (i, i', s)
            iff
            (
             ((* ...,i,i',... *)
              orderNo (i, s) < orderNo (i', s)
              andalso 
              (forall j => 
                 orderNo (j, s) <= orderNo (i, s)
                 orelse
                 orderNo (i', s) <= orderNo (j, s))
             ) 
             orelse 
             ((* i',...,i *)
              orderNo (i', s) < orderNo (i, s) 
              andalso 
              (forall j => 
                 is_in (j, s) implies
                   orderNo (i', s) <= orderNo (j, s)
                   andalso
                   orderNo (j, s) <= orderNo (i, s))
             )
            )

    axiom forall (eq, f, b, s, e) =>
            ((is_in (e, s)) andalso 
             (forall j => 
                orderNo (j, s) <= orderNo (e, s))) 
            implies
               eq (foldL eq f b s, foldL eq f (f e b) (remove (e, s)))
  end (*of local*)
end;

signature CYCLE_LIST_OF_NODES =
sig
   include CYCLE_LIST
   structure Node: NODE
   sharing Elem = Node
end;

signature CYCLE_LIST_OF_EDGES =
sig
   include CYCLE_LIST
   structure Edge: NONDIRECTED_EDGE
   sharing Elem = Edge
end;

signature NONDIRECTED_GRAPH =
sig
   structure Node : NODE
   structure Edge : NONDIRECTED_EDGE
   structure Nodes: SET_OF_NODES
   structure Edges: SET_OF_EDGES
   sharing Nodes.Node = Node = Edge.Node
       and Edge = Edges.Edge

   type graph

   val empty: graph

   val isNode     : graph * Node.id -> bool
   val isEdge     : graph * Edge.id -> bool
   val equal      : graph * graph -> bool
   val isEmpty    : graph -> bool
   val neighNodes : graph * Node.id -> Nodes.set
   val allNodes   : graph -> Nodes.set
   val addNode    : graph * Node.id -> graph
   val removeNode : graph * Node.id -> graph
   val neighEdges : graph * Node.id -> Edges.set
   val allEdges   : graph -> Edges.set
   val addEdge    : graph * Edge.id -> graph
   val removeEdge : graph * Edge.id -> graph

axiom forall (g, n, n') =>
        Node.equal (n, n') implies
          equal (addNode (g, n), addNode (g, n'))
          andalso
          equal (removeNode (g, n), removeNode (g, n'))
  and forall (g, e, e') =>
        Edge.equal (e, e') implies
          equal (addEdge (g, e), addEdge (g, e'))
          andalso
          equal (removeEdge (g, e), removeEdge (g, e'))

axiom forall (g, g') =>
        equal (g, g')
        iff
        (
          (forall n =>
             isNode (g, n) iff isNode (g', n))
          andalso
          (forall e =>
             isEdge (g, e) iff isEdge (g', e))
        )
           
axiom forall (g, n) => 
        isNode (g, n) implies
          equal (addNode (g, n), g)
          
  and forall (g, e) =>
        isEdge (g, e) implies
          equal (addEdge (g, e), g)
          
  and forall (g, n) => 
        not (isNode (g, n)) implies
          equal (removeNode (g, n), g)
          
  and forall (g, e) =>
        not (isEdge (g, e)) implies
          equal (removeEdge (g, e), g)
          
axiom forall (g, n, n') =>
        not (Node.equal(n, n')) implies
          (isNode (g, n) iff isNode (addNode (g, n'), n))
          andalso
          (isNode (g, n) iff isNode (removeNode (g, n'), n))
  and forall (g, n, n', n'') =>
        (not (Node.equal(n, n'))) andalso (not (Node.equal(n, n''))) implies
          (isNode (g, n) iff isNode (addEdge (g, Edge.mkEdge (n', n'')), n))
          andalso
          (isNode (g, n) iff isNode (removeEdge (g, Edge.mkEdge (n', n'')), n))
  and forall (g, n, n', n'') =>
        (not (Node.equal(n, n'))) andalso (not (Node.equal(n, n''))) implies
          (isEdge (g, Edge.mkEdge (n', n'')) 
          iff isEdge (addNode (g, n), Edge.mkEdge (n', n'')))
          andalso
          (isEdge (g, Edge.mkEdge (n', n'')) 
          iff isEdge (removeNode (g, n), Edge.mkEdge (n', n'')))
  and forall (g, e, e') =>
        not (Edges.Edge.equal (e, e')) implies
          (isEdge (g, e) iff isEdge (addEdge (g, e'), e))
          andalso
          (isEdge (g, e) iff isEdge (removeEdge (g, e'), e))

axiom forall g =>
        isEmpty g iff equal (g, empty)

axiom forall (g, n) =>
        equal (g, empty) implies
          not (isNode (g, n))
  and forall (g, n) =>
        isNode (g, n)
        iff
        (
          (exists g' =>
            not (isNode (g', n))
            andalso
            equal (g, addNode (g', n)))
          orelse
          (exists (g', e, n') =>
            Edge.equal (e, Edge.mkEdge (n, n'))
            andalso
            equal (g, addEdge (g', e)))
        )

axiom forall (g, n) =>
        not (isNode (removeNode (g, n), n))

axiom forall (g, e) =>
        equal (empty, g) implies
          not (isEdge (g, e))
  and forall (g, e) =>
        isEdge (g, e)
        iff
        (
          exists g' =>
            not (isEdge (g', e))
            andalso
            equal (g, addEdge (g', e))
        )

axiom forall (g, e) =>
        not (isEdge (removeEdge (g, e), e))
  and forall (g, e, n, n') =>
        Edge.equal (e, Edge.mkEdge (n, n')) implies
          not (isEdge (removeNode (g, n), e))

axiom forall (g, n) =>
        Nodes.is_in (n, allNodes g)
        iff
        isNode (g, n)
axiom forall (g, e) =>
        Edges.is_in (e, allEdges g)
        iff
        isEdge (g, e)    

axiom forall (g, n, n') =>
        Nodes.is_in (n', neighNodes (g, n))
        iff
        isEdge (g, Edge.mkEdge (n, n'))
axiom forall (g, n, n') =>
        Edges.is_in (Edge.mkEdge (n, n'), neighEdges (g, n))
        iff
        isEdge (g, Edge.mkEdge (n, n'))
end;


signature SUB_GRAPH =
sig
   structure Graph : NONDIRECTED_GRAPH

   val isSubGraph : Graph.graph * Graph.graph -> bool

axiom forall (sg, g) =>
        isSubGraph (sg, g)
        iff
        ((forall i =>
            Graph.isNode (sg, i) implies
              Graph.isNode (g, i))
         andalso
         (forall e =>
            Graph.isEdge (sg, e) implies
              Graph.isEdge (g, e)))
end;

signature CONNECTED_COMPONENTS =
sig
   structure Node : NODE
   structure Edge : NONDIRECTED_EDGE
   structure Graph : NONDIRECTED_GRAPH  
   structure Components: SET_OF_SETS_OF_NODES

   sharing Node = Graph.Node = Components.Set.Node
       and Edge = Graph.Edge

   local
       structure NodesList: CYCLE_LIST_OF_NODES
       sharing Node = NodesList.Node
       
       val exists_path: Graph.graph * Node.id * Node.id -> bool

       axiom forall (g, n, n') =>
             exists_path (g, n, n')
             iff
             Node.equal (n, n')
             orelse
             exists l =>
               NodesList.are_neighbours (n', n, l)
               andalso
               forall n =>
                 Graph.isNode (g, n) iff NodesList.is_in (n, l)
               andalso
               forall (i, i') =>
                 NodesList.are_neighbours (i, i', l) andalso
                 not (Node.equal (n, i') andalso Node.equal (n', i)) implies
                   Graph.isEdge (g, Edge.mkEdge (i, i'))
   in
      val connected_components: Graph.graph -> Components.set
      
axiom forall g =>
        not (Components.is_in (Components.Set.empty, connected_components g))

axiom forall (g, n) =>
        Components.is_elem_in (n, connected_components g)
        iff
        Graph.isNode (g, n)

axiom forall (g, l, l') =>
        (
          Components.is_in (l,  connected_components g)
          andalso
          Components.is_in (l', connected_components g)
          andalso
          not (Components.Set.equal (l, l'))
        )
        implies
        (
          forall n =>
            Components.Set.is_in (n, l) implies
               not (Components.Set.is_in (n, l'))
        )
            
axiom forall (g, l, n, n') =>
        (
          Components.is_in (l, connected_components g)
          andalso
          Components.Set.is_in (n,  l)
          andalso
          Components.Set.is_in (n', l)
        )
        implies
          exists_path (g, n, n') 
             
 axiom forall (g, l, l', n, n') =>
        (
          Components.is_in (l,  connected_components g)
          andalso
          Components.is_in (l', connected_components g)
          andalso
          not (Components.Set.equal (l, l'))
          andalso
          Components.Set.is_in (n,  l)
          andalso
          Components.Set.is_in (n', l')
        )
        implies
          not (exists_path (g, n, n'))
   end
end;

signature EULER =
sig
   structure Node : NODE
   structure Edge : NONDIRECTED_EDGE
   structure Graph : NONDIRECTED_GRAPH  
   structure EdgesList : CYCLE_LIST_OF_EDGES
      
   sharing Node = Graph.Node = Edge.Node
   sharing Edge = Graph.Edge = EdgesList.Edge

   val is_Euler_cycle: Graph.graph * EdgesList.set -> bool
   val has_Euler_cycle: Graph.graph -> bool
   val Euler_cycle: Graph.graph -> EdgesList.set

axiom forall (g, l, l') =>
        EdgesList.equal (l, l') implies
          is_Euler_cycle (g, l) iff is_Euler_cycle (g, l')
  and forall (g, g') =>
        Graph.equal (g, g') implies
          has_Euler_cycle g iff has_Euler_cycle g'
  and forall (g, g') =>
        Graph.equal (g, g') implies
          EdgesList.equal (Euler_cycle g, Euler_cycle g')

axiom forall (g, l) =>
        is_Euler_cycle (g, l)
        iff
        ( 
          (
            EdgesList.is_empty l
            andalso
            Graph.isEmpty g
          )
          orelse
          (
            (forall e =>
               EdgesList.is_in (e, l) iff Graph.isEdge (g, e))
            andalso
            (forall (e, e') =>
               EdgesList.are_neighbours (e, e', l) implies
                 Edge.are_connected (e, e'))
          )
        )

 axiom forall g =>
        has_Euler_cycle g
        iff
          exists l =>
            is_Euler_cycle (g, l)
    
axiom forall g =>
        has_Euler_cycle g implies
          is_Euler_cycle (g, Euler_cycle g)

axiom forall g => 
        not (has_Euler_cycle g) implies
          EdgesList.equal (Euler_cycle g, EdgesList.empty)
end;

signature HAMILTON =
sig
   structure Node : NODE
   structure Edge : NONDIRECTED_EDGE
   structure Graph : NONDIRECTED_GRAPH
   structure NodesList : CYCLE_LIST_OF_NODES

   sharing Node = Graph.Node = NodesList.Node = Edge.Node
   sharing Edge = Graph.Edge

   val is_Hamilton: Graph.graph * NodesList.set -> bool
   val has_Hamilton: Graph.graph -> bool
   val Hamilton: Graph.graph -> NodesList.set

axiom forall (g, l, l') =>
        NodesList.equal (l, l') implies
          is_Hamilton (g, l) iff is_Hamilton (g, l')
  and forall (g, g') =>
        Graph.equal (g, g') implies
          has_Hamilton g iff has_Hamilton g'
  and forall (g, g') =>
        Graph.equal (g, g') implies
          NodesList.equal (Hamilton g, Hamilton g')

axiom forall (g, h) =>
        is_Hamilton (g, h)
        iff
        (
         forall (n, n') =>
              NodesList.are_neighbours (n, n', h) implies
                 Graph.isEdge (g, Edge.mkEdge (n, n'))
         andalso
         forall n =>       
            Graph.isNode (g, n)
            iff
            NodesList.is_in (n, h)
        )

axiom forall g =>
        has_Hamilton g
        iff
          exists l =>
            is_Hamilton (g, l)

axiom forall g =>
        has_Hamilton g implies
          is_Hamilton (g, Hamilton g)

axiom forall g =>
        not (has_Hamilton g) implies
          NodesList.equal (Hamilton g, NodesList.empty)
end;


signature GRAPH_ALGS  =
sig
   structure SubGraph : SUB_GRAPH
   structure Euler    : EULER
   structure Hamilton : HAMILTON
   structure ConnComp : CONNECTED_COMPONENTS

   sharing SubGraph.Graph = Euler.Graph = Hamilton.Graph = ConnComp.Graph
end;


signature PERMUTATIONS =
sig
  structure Set   : SET
  structure CList : CYCLE_LIST
  sharing Set.Elem = CList.Elem

  local

    val equalSetCList: Set.set * CList.set -> bool
  axiom forall (s, c) =>
          equalSetCList (s, c)
          iff
          forall i =>
            Set.is_in (i, s) iff CList.is_in (i, c)

  in

    val find: Set.set -> (CList.set -> bool) -> CList.set
axiom forall (s, f) =>
        exists c => f c
        andalso
        equalSetCList (s, c)
           implies
             CList.equal (find s f, c)
axiom forall (s, f) =>
        (forall c =>
           equalSetCList (s, c)
           andalso
           not (f c)) implies 
             CList.equal (find s f, CList.empty)
  end
end;
