functor SetOfNodes' (structure N: NODE
                     structure ES: SET
                     sharing N = ES.Elem): sig include SET_OF_NODES
                                               sharing Node = N
                                           end =
struct
  structure Node = N
  open ES
end;


functor SetOfEdges' (structure E: NONDIRECTED_EDGE
                     structure ES: SET
                     sharing E = ES.Elem): sig include SET_OF_EDGES
                                               sharing Edge = E
                                           end =
struct
  structure Edge = E
  open ES
end;


functor GraphAlgs'' (structure N: NODE
                     structure G: NONDIRECTED_GRAPH
                     structure E: EULER
                     structure H: HAMILTON
                     structure C: CONNECTED_COMPONENTS
                     structure S: SUB_GRAPH
                     sharing N = G.Node
                         and G = E.Graph = H.Graph = C.Graph = S.Graph
                                       ): sig
                                            include GRAPH_ALGS
                                            sharing SubGraph = S
                                                and Euler    = E
                                                and Hamilton = H
                                                and ConnComp = C
                                          end =
struct
  structure SubGraph = S
  structure Euler    = E
  structure Hamilton = H
  structure ConnComp = C
end;


functor EdgesList' (structure E: NONDIRECTED_EDGE
                    structure CL: CYCLE_LIST
                    sharing E = CL.Elem): sig include CYCLE_LIST_OF_EDGES
                                              sharing Edge = E
                                          end =
struct
  structure Edge = E
  open CL
end;


functor NodesList' (structure N: NODE
                    structure CL: CYCLE_LIST
                    sharing N = CL.Elem): sig include CYCLE_LIST_OF_NODES
                                              sharing Node = N
                                          end =
struct
  structure Node = N
  open CL
end;


functor NondirectedEdge (structure N: NODE): sig include NONDIRECTED_EDGE
                                                 sharing Node = N
                                             end = 
struct
  structure Node = N
  type id = Node.id * Node.id
  
  fun equal ((l, r), (l', r')) = 
        (Node.equal (l, l') andalso Node.equal (r, r'))
        orelse
        (Node.equal (l, r') andalso Node.equal (r, l'))
  fun left  (l, _) = l
  fun right (_, r) = r
  fun mkEdge (l, r) = (l, r)
  fun are_connected ((l, r), (l', r')) = 
        Node.equal (l, l') orelse Node.equal (l, r') orelse
        Node.equal (r, l') orelse Node.equal (r, r')
end;


functor SetOfElems' (structure E: ELEM
                     structure CL: CYCLE_LIST
                     sharing E = CL.Elem
                    ) : sig include SET
                            sharing Elem = E
                            sharing type set = CL.set
                        end =
  CL;

functor SetOfSetsOfNodes' (structure N: NODE
                           structure SofN: SET_OF_NODES
                           structure E: ELEM
                           structure SofE: SET
                           sharing N = SofN.Node
                           sharing E = SofE.Elem
                           sharing type E.id = SofN.set
                          ): sig include SET_OF_SETS_OF_NODES
                                 sharing Node = N
                                 sharing Set = SofN
                                 sharing Elem = E
                             end =
struct
  structure Node = N
  structure Set = SofN
  open SofE
  
  fun is_elem_in (i: Node.id, s: set) =
       foldL op= 
             (fn e => fn b =>
                if b then true
                else Set.is_in (i, e))
             false s
end;


functor CycleList (structure E: ELEM): sig include CYCLE_LIST
                                           sharing Elem = E
                                       end =
struct
  structure Elem = E
  type set = Elem.id list 
  val empty = []

  fun foldL eq f b [] = b
    | foldL eq f b (e::rest) = foldL eq f (f e b) rest

  fun is_in (i, cl) =
    foldL op= 
          (fn e => fn b => 
             if b then true 
             else Elem.equal (i, e))
          false cl

  fun equal ([]: set, []: set) = true
    | equal (_ , []) = false
    | equal ([], _ ) = false
    | equal (e::l, e'::l') =
      let
        fun shift (e, e'::l') = if Elem.equal (e, e') then e'::l'
                                else shift (e, l' @ [e'])
        fun same_lists ([], []) = true
          | same_lists (_, []) = false
          | same_lists ([], _) = false
          | same_lists (e::l, e'::l') = Elem.equal (e, e') andalso
                                        same_lists (l, l')
      in
        if is_in (e, e'::l') then same_lists (e::l, shift (e, e'::l'))
        else false
      end

  fun add (i, c) = if is_in (i, c) then c
                   else i::c

  fun remove (i, []) = []
    | remove (i, i'::rest) = if Elem.equal (i, i') then rest
                             else i'::remove (i, rest)

  fun count l = foldL op= (fn i => fn cnt => cnt + 1) 0 l

  fun is_empty ([]) = true
    | is_empty (_)  = false

  fun are_neighbours (i, i', e::c) =
      let
        fun are (_, _, []) = false
          | are (_, _, [_]) = false
          | are (e1, e2, e2'::e1'::rest) =
              if Elem.equal (e2', e2) then
                if Elem.equal (e1', e1) then true
                else false
              else are (e1, e2, (e1'::rest))
      in
        if is_in (i, e::c) andalso is_in (i', e::c) then are (i, i', e::c@[e])
        else false
      end
end;


functor NondirectedGraph'' (structure N: NODE
                            structure E: NONDIRECTED_EDGE
                            structure NS: SET_OF_NODES
                            structure ES: SET_OF_EDGES
                            sharing E.Node = N = NS.Node
                                and E = ES.Edge): 
                                         sig include NONDIRECTED_GRAPH
                                             sharing Node = N
                                                 and Edge = E
                                                 and Nodes = NS
                                                 and Edges = ES
                                         end =
struct
  structure Node = N
  structure Edge = E
  structure Nodes = NS
  structure Edges = ES
  
  type graph = Nodes.set * Edges.set
  val empty = (Nodes.empty, Edges.empty)
                  
  fun isNode ((ns, es), i) = Nodes.is_in (i, ns)

  fun isEdge ((ns, es), e) = Edges.is_in (e, es)

  fun equal ((ns, es), (ns', es')) = 
        Nodes.equal (ns, ns') andalso Edges.equal (es, es')

  fun isEmpty (ns, es) = Nodes.is_empty (ns)

  fun neighNodes (g as (ns, es), i) =
    Nodes.foldL Nodes.equal
                (fn n => 
                   fn ns => 
                     if isEdge (g, Edge.mkEdge (n, i)) then
                       Nodes.add (n, ns)
                     else ns)
                Nodes.empty ns

  fun allNodes (ns, _) = ns

  fun addNode ((ns, es), i) = (Nodes.add (i, ns), es)

  fun neighEdges (g as (ns, es), i) =
    Nodes.foldL Edges.equal
                (fn n =>
                   fn es =>
                     let
                       val edge = Edge.mkEdge (i, n)
                     in
                       if isEdge (g, edge) then
                         Edges.add (edge, es)
                       else es
                     end)
                Edges.empty ns

  fun removeEdge ((ns, es), e) = (ns, Edges.remove (e, es))

  fun removeNode (g, i) =
    let
      val (ns', es') =
           Edges.foldL equal
                       (fn e => fn g => removeEdge (g, e)) 
                       g (neighEdges (g, i))
    in
      (Nodes.remove (i, ns'), es')
    end

  fun allEdges (_, es) = es

  fun addEdge ((ns, es), e) = 
       (Nodes.add (Edge.left e, Nodes.add (Edge.right e, ns)), 
        Edges.add (e, es))
end;


functor SubGraph (structure G: NONDIRECTED_GRAPH): sig include SUB_GRAPH
                                                       sharing Graph = G
                                                   end =
struct
  structure Graph = G

  fun isSubGraph (sg, g) =
        Graph.Nodes.foldL op=
                          (fn n => fn b =>
                             if b then
                               Graph.isNode (g, n)
                             else false)
                          true (Graph.allNodes sg)
        andalso
        Graph.Edges.foldL op=
                          (fn e => fn b =>
                             if b then
                               Graph.isEdge (g, e)
                             else false)
                          true (Graph.allEdges sg)
end;


functor Permutations (structure CL: CYCLE_LIST
                      structure S: SET
                      sharing CL.Elem = S.Elem
                                       ): sig
                                            include PERMUTATIONS
                                            sharing CList = CL
                                                and Set   = S
                                          end =
struct
  structure Elem = S.Elem
  structure CList = CL
  structure Set = S
  datatype 'a Option = None | Some of 'a
  type perms = (int * int * (Elem.id list)) list

  fun reverse ([], revs) = revs
    | reverse (i::is, revs) = reverse (is, i::revs)

  fun init' ([i], 1) = [(1, 0, [i])]
    | init' (i::is, n) = (n, 0, i::is)::(init' (is, n-1))
  
  fun init (is, n) = reverse (init' (is, n), [])
  
  fun initialize [] = None
    | initialize is =
       let
         fun count ([], n) = n
           | count(i::is, n) = count (is, n+1)
       in
         Some (init (is, count (is, 0)))
       end

  fun build_perm ([]) = []
    | build_perm ((no, cnt, i::is)::rest) = i::(build_perm rest)

  fun switch_perms (_, []) = None
    | switch_perms (done, (t as (no, cnt, i::is))::rest) =
        if cnt + 1 < no then
          let
            fun tail (i::is) = is
            val is' = is @ [i]
          in
            Some ((init (tail is', no-1)) @ [(no, cnt+1, is')] @ rest)
          end
        else
          switch_perms (done@[t], rest)

  fun next p = (switch_perms ([], p), build_perm p)

  fun find s test =
       let
         val ls = Set.foldL (fn (l, l') => true)
                        (fn e => fn ls => e::ls)
                        [] s
         val p = initialize ls
         fun test_next p =
              let
                val (p_opt, ls) = next p
                fun list2CList ([], cl) = cl
                  | list2CList (e::rest, cl) = 
                       list2CList (rest, CList.add (e, cl))
                val l = list2CList (ls, CList.empty)
              in
                if test l then l
                else case p_opt of
                       None => CList.empty
                     | Some p => test_next p
              end
       in
         case initialize ls of
           None => CList.empty
         | Some p => test_next p
       end
end;


functor Hamilton'' (structure G: NONDIRECTED_GRAPH
                    structure L: CYCLE_LIST_OF_NODES
                    structure P: PERMUTATIONS
                    sharing G.Node = L.Node
                        and P.CList = L
                        and P.Set = G.Nodes
                                          ): sig include HAMILTON
                                                 sharing Graph = G
                                                     and NodesList = L
                                             end =
struct
  structure Node = G.Node
  structure Edge = G.Edge
  structure Graph = G
  structure NodesList = L
  structure Perms = P

  fun is_Hamilton (g, l) =
        Graph.Nodes.foldL op=
                  (fn n => fn b =>
                     if b then
                       Graph.Nodes.foldL op= (fn n' => fn b' =>
                            if b' then
                              if NodesList.are_neighbours (n, n', l) then 
                                Graph.isEdge (g, Edge.mkEdge (n, n'))
                              else true
                            else false)
                                         true (Graph.allNodes g)
                     else false) 
                  true (Graph.allNodes g)
        andalso
        Graph.Nodes.foldL op=
                  (fn n => fn b => 
                     if b then NodesList.is_in (n, l)
                     else false) 
                  true (Graph.allNodes g)         
        andalso
        NodesList.foldL op=
                (fn n => fn b => 
                   if b then Graph.isNode (g, n) else false) 
                true l

  fun Hamilton g =
       Perms.find (Graph.allNodes g) (fn nl => is_Hamilton (g, nl))

  fun has_Hamilton g =
       if Graph.isEmpty g then true
       else not (NodesList.is_empty (Hamilton g))
end;


functor Euler'' (structure G: NONDIRECTED_GRAPH
                 structure E: CYCLE_LIST_OF_EDGES
                 structure P: PERMUTATIONS
                 sharing G.Edge = E.Edge
                     and P.CList = E
                     and P.Set = G.Edges): sig include EULER 
                                               sharing Graph = G
                                               sharing EdgesList = E
                                           end = 
struct
  structure Node = G.Node
  structure Edge = G.Edge
  structure Graph = G
  structure EdgesList = E
  structure Perms = P
  
  fun is_Euler_cycle (g, l) =
        ( EdgesList.is_empty l
          andalso
          Graph.isEmpty g)
        orelse
        ( Graph.Edges.foldL op=
                  (fn e => fn b =>
                     if b then
                       Graph.Edges.foldL op= (fn e' => fn b' =>
                            if b' then
                              if EdgesList.are_neighbours (e, e', l) then 
                                Edge.are_connected (e, e')
                              else true
                            else false)
                                         true (Graph.allEdges g)
                     else false) 
                  true (Graph.allEdges g)
          andalso
          Graph.Edges.foldL op=
                            (fn e => fn b => 
                               if b then EdgesList.is_in (e, l)
                               else false) 
                            true (Graph.allEdges g)         
          andalso
          EdgesList.foldL op=
                          (fn e => fn b => 
                             if b then Graph.isEdge (g, e) 
                             else false) 
                          true l
        )
  
  fun Euler_cycle g =
       Perms.find (Graph.allEdges g) (fn el => is_Euler_cycle (g, el))

  fun has_Euler_cycle g =
       if Graph.isEmpty g then true
       else not (EdgesList.is_empty (Euler_cycle g))
end;


functor ConnComp' (structure G: NONDIRECTED_GRAPH
                   structure C: SET_OF_SETS_OF_NODES
                   sharing G.Node = C.Node): sig include CONNECTED_COMPONENTS
                                                 sharing Graph = G
                                                     and Components = C
                                             end =
struct
  structure Node = G.Node
  structure Edge = G.Edge
  structure Graph = G
  structure Components = C
  structure Set = Components.Set

  fun DFS (g, n, visited) =
    Graph.Nodes.foldL Set.equal
                      (fn n => fn vs =>
                         if Set.is_in (n, vs) then
                           vs
                         else
                           DFS (g, n, vs))
                      (Set.add (n, visited)) (Graph.neighNodes (g, n))
      
  fun connected_components g =
     Graph.Nodes.foldL Components.equal
                       (fn n => fn ccs => 
                          if Components.is_elem_in (n, ccs) then 
                            ccs
                          else 
                            Components.add (DFS (g, n, Set.empty), ccs))
                       Components.empty (Graph.allNodes g)
end;


functor SetOfElems (structure E: ELEM): sig include SET
                                            sharing Elem = E
                                        end =
  SetOfElems' (structure E = E
               structure CL = CycleList (structure E = E));


functor SetOfNodes (structure N: NODE): sig include SET_OF_NODES
                                            sharing Node = N
                                        end =
  SetOfNodes' (structure N = N
               structure ES = SetOfElems (structure E = N));


functor SetOfEdges (structure E: NONDIRECTED_EDGE): sig include SET_OF_EDGES
                                                        sharing Edge = E
                                                    end =
  SetOfEdges' (structure E = E
               structure ES = SetOfElems (structure E = E));


functor NodesList (structure N: NODE): sig include CYCLE_LIST_OF_NODES
                                           sharing Node = N
                                       end =
  NodesList' (structure N = N
              structure CL = CycleList (structure E = N));


functor EdgesList (structure E: NONDIRECTED_EDGE): 
                                       sig include CYCLE_LIST_OF_EDGES
                                           sharing Edge = E
                                       end =
  EdgesList' (structure E = E
              structure CL = CycleList (structure E = E));

                                             
functor SetOfSetsOfNodes (structure N: NODE): sig include SET_OF_SETS_OF_NODES
                                                  sharing Node = N
                                              end =
   SetOfSetsOfNodes' (structure N = N
                      structure SofN = SetOfNodes (structure N = N)
                      structure E: sig include ELEM
                                       sharing type id = SofN.set
                                   end =
                                   struct
                                     type id = SofN.set
                                     val equal = SofN.equal
                                   end
                      structure SofE = SetOfElems (structure E = E));


functor ConnComp (structure G: NONDIRECTED_GRAPH): 
                                              sig include CONNECTED_COMPONENTS
                                                  sharing Graph = G
                                              end =
  ConnComp' (structure G = G
             structure C = SetOfSetsOfNodes (structure N = G.Node));


functor Hamilton' (structure G: NONDIRECTED_GRAPH
                   structure L: CYCLE_LIST_OF_NODES
                   sharing G.Node = L.Node): sig include HAMILTON
                                                 sharing Graph = G
                                                     and NodesList = L
                                             end =
  Hamilton'' (structure G = G
              structure L = L
              structure P = Permutations (structure CL = L
                                          structure S = G.Nodes));


functor Hamilton (structure G: NONDIRECTED_GRAPH): sig include HAMILTON
                                                       sharing Graph = G
                                                   end =
  Hamilton' (structure G = G
             structure L = NodesList (structure N = G.Node));


functor Euler' (structure G: NONDIRECTED_GRAPH
                structure E: CYCLE_LIST_OF_EDGES
                sharing G.Edge = E.Edge): sig include EULER 
                                              sharing Graph = G
                                              sharing EdgesList = E
                                          end = 
  Euler'' (structure G = G
           structure E = E
           structure P = Permutations (structure CL = E
                                       structure S = G.Edges));


functor Euler (structure G: NONDIRECTED_GRAPH): sig include EULER
                                                     sharing Graph = G
                                                 end =
    Euler' (structure G = G
            structure E = EdgesList (structure E = G.Edge));


functor NondirectedGraph' (structure N: NODE
                           structure E: NONDIRECTED_EDGE
                           sharing E.Node = N): 
                                         sig include NONDIRECTED_GRAPH
                                             sharing Node = N
                                                 and Edge = E
                                         end =
   NondirectedGraph'' (structure N = N
                       structure E = E
                       structure NS = SetOfNodes (structure N = N)
                       structure ES = SetOfEdges (structure E = E));


functor NondirectedGraph (structure N: NODE): 
                         sig include NONDIRECTED_GRAPH
                             sharing Node = N
                         end =
  NondirectedGraph' (structure N = N
                     structure E = NondirectedEdge (structure N = N));


functor GraphAlgs' (structure N: NODE
                    structure G: NONDIRECTED_GRAPH
                    sharing N = G.Node): 
                   sig include GRAPH_ALGS
                       sharing SubGraph.Graph = G
                   end =
    GraphAlgs'' (structure N = N
                 structure G = G
                 structure E = Euler (structure G = G)
                 structure H = Hamilton (structure G = G)
                 structure C = ConnComp (structure G = G)
                 structure S = SubGraph (structure G = G));


functor GraphAlgs (structure N: NODE): sig include GRAPH_ALGS
                                           sharing SubGraph.Graph.Node = N 
                                       end = 
  GraphAlgs' (structure N = N
              structure G = NondirectedGraph (structure N = N));
