(*
$File: Parsing/LexBasics.sml $
$Date: 1992/04/07 14:58:39 $
$Revision: 1.11 $
$Locker:  $
*)

(*$LexBasics: BASIC_IO REPORT PRETTYPRINT FLAGS CRASH LEX_BASICS*)
functor LexBasics(structure BasicIO: BASIC_IO
		  structure Report: REPORT
		  structure PP: PRETTYPRINT
		  structure Flags: FLAGS
		  structure Crash: CRASH
		 ): LEX_BASICS =
  struct
    datatype pos = POSITION of unit -> {file: string, line: int, column: int,
					getLine: int -> string
				       }
		 | DUMMY
    

    datatype SourceReader =
	SOURCE_READER of {name: string,
			  clearFn: unit -> unit,
			  lexingFn: int -> string,
			  positionFn: int -> pos
			 }

   (* We store an entire file (or the entire source text string) as a list
      of lines, paired with the absolute character position of the first
      character of the line (counting from 0). All the lines have a "\n" at
      the end, except the last. For ordinary text files, the last line will
      be "". For weirder files (with EOF in the middle of the line), it
      will be some text without a "\n". Oh: we must remove tabs in here
      (see also lexFromStdIn). *)

    datatype SourceText =
      SOURCE_TEXT of {filename: string,
		      lines: {absCharacter: int, line: string} list
		     }
		      
   (* Reports, diags etc. *)

    type Report = Report.Report

    infix //
    val op // = Report.//

    local
      fun spaces n = String.create n " "

      fun untabifyL(col, "\t" :: rest) =
	    let
	      val gap = 8 - col mod 8
	    in
	      spaces gap :: untabifyL(col + gap, rest)
	    end

	| untabifyL(col, x :: rest) =
	    x :: untabifyL(col + 1, rest)

	| untabifyL(_, nil) = nil
    in
      fun untabify indent line = implode(untabifyL(indent, explode line))
    end

    fun stringToSourceText(filename, string) =
      let
	fun process "" = [""]
	  | process string =
	      case String.search String.MatchCase "\n" string 0
		of OK n =>
		     String.extract 0 (n+1) string (* Include `\n'. *)
		     :: process(String.extract (n+1) (size string) string)

		 | Fail() =>
		     [string]

	val lines = map (untabify 0) (process string)

	fun withAbsCharacter(n, (x :: xs)) =
	      {absCharacter=n, line=x} :: withAbsCharacter(n+size x, xs)
	  | withAbsCharacter(_, nil) = nil
      in
	SOURCE_TEXT{filename=filename, lines=withAbsCharacter(0, lines)}
      end


   (* I suspect doing that for an entire file might be rather painful. So,
      we read a file a chunk at a time and build a sequence of SourceTexts.
      Therefore, we need a SourceText concatenation function. *)

    fun bumpChars(n: int, {absCharacter, line} :: rest) =
          {absCharacter=absCharacter+n, line=line} :: bumpChars(n, rest)
      | bumpChars(_, nil) = nil

    fun concat(SOURCE_TEXT{filename, lines=list1},
	       SOURCE_TEXT{lines=list2, ...}
	      ) =
      case (rev list1, list2)
	of ({absCharacter=lastChar, line=lastLine} :: revFirstLines,
	    {line=firstLine2, ...} :: restLines
	   ) =>
	     let
	       val joinedLine =
		 {absCharacter=lastChar, line=lastLine ^ firstLine2}
	     in
	       SOURCE_TEXT{
		 filename=filename,
		 lines=rev revFirstLines
		       @ (joinedLine
			  :: bumpChars(lastChar + size lastLine, restLines)
			 )
	       }
	     end

	 | _ => Crash.impossible "LexBasics.concat"

    val BUFFER_SIZE = 1024		(* Say... *)

    fun fileToSourceText file =
      let
	val stream = BasicIO.open_in file	(* Io raised here gets dealt
						   with in any calling
						   context properly. *)

	val _ = BasicIO.println("[loading " ^ file ^ "]")

	fun read() =
	  let
	    val text = BasicIO.input(stream, BUFFER_SIZE)
	    val st = stringToSourceText(file, text)
	  in
	    if size text = BUFFER_SIZE
	    then concat(st, read())
	    else st
	  end

	val st = read()
      in
	BasicIO.close_in stream; st
      end

   (* `positionFn' is suspended, for efficiency. It's used in
      lexFromSourceText (to turn a completed SourceText into a position
      function), and we also use it when reading interactively and
      making up a SourceText as we go. *)

    fun positionFn(sourceTextRef, absPos) =
					(* Must start from very top... *)
      let
	fun pr n =
	  let
	    val ref(SOURCE_TEXT{lines, ...}) = sourceTextRef
	  in
	    Report.print(
	      Report.decorate("getLine " ^ Int.string n ^ ": ",
			      List.foldR (fn {line, ...} =>
					    fn rep =>
					      Report.line line // rep
					 ) Report.null lines
			     )
	    )
	  end

	fun getLine n =
	  let
	    val ref(SOURCE_TEXT{lines, ...}) = sourceTextRef
	  in
	    if Flags.DEBUG_ERRORPRINT then pr n else ();
	    #line(List.nth (n-1) lines)
	  end
      in
	POSITION(fn () =>
	  let
	    val _ =
	      if Flags.DEBUG_ERRORPRINT then
		 BasicIO.println("positionFn(absPos=" ^ Int.string absPos ^ ")")
	      else ()

	    val ref(SOURCE_TEXT{filename, lines}) = sourceTextRef

	    fun search(n, previousLine, (L as {absCharacter, line}) :: rest) =
		  if absPos < absCharacter
				(* The next line in the list is beyond
				   this character position, so we must want
				   the previous line. *)
		  then
		    case previousLine
		      of Some{absCharacter, line} =>
			   {file=filename, line=n,
			    column=absPos-absCharacter,
			    getLine=getLine
			   }

		       | None =>
			   Crash.impossible "LexBasics.search(previous/1)"
		  else
		    search(n+1, Some L, rest)

	      | search(n, previousLine, nil) =
				(* No more lines, so we must want the
				   previous line. *)
		  (case previousLine
		     of Some{absCharacter, line} =>
			  {file=filename, line=n,
			   column=absPos-absCharacter,
			   getLine=getLine
			  }

		      | None =>
			  Crash.impossible "LexBasics.search(previous/2)"
		  )
	  in
	    search(0, None, lines)
	  end
	)
      end

    fun lexFromSourceText(source as SOURCE_TEXT{filename, lines=allLines}) =
      let
	val theLines = ref allLines

	fun lexingFn _ =
	  case !theLines
	    of nil => ""
	     | {line, ...} :: rest => (theLines := rest; line)
      in
	SOURCE_READER{name=filename,
		      clearFn=fn () => (),
		      lexingFn=lexingFn,
		      positionFn=fn absPos => positionFn(ref source, absPos)
		     }	(* When we're lexing from a source file, we don't have
			   the `clearFn' functionality to reset the list of
			   input lines - it's too much hassle to implement.
			   However, errors during file read are fatal anyway,
			   so it's not important. Not that `clearFn' mustn't
			   panic: it gets called right at the start of any
			   parse, even for file/string input.
			     The ref here is superfluous; positionFn takes
			   a ref for handling incremental input properly. *)
      end

    val lexFromString =
      lexFromSourceText o (fn text => stringToSourceText("<string>", text))

    val lexFromFile =
      lexFromSourceText o fileToSourceText


   (* lexFromStdIn is more complex. (Yes: even more complex.) We are lexing as
      we go along and building a SourceText line by line. Oh, it's here that
      we get to eliminate tabs as well. They're nuked as we read the lines of
      source, which saves us the hassle of dealing with them later. Don't you
      dare use other than 8-character tabs. *)

    val prompt = ". "

    fun lexFromStdIn() =
      let
	val name = "std_in"
	val theSource = ref(SOURCE_TEXT{filename=name, lines=nil})

	fun getLine() =
	  let
	   (* Prompt and get the line: *)

(* arkady#*)
	    val _ = (BasicIO.print prompt;Outstream.flush(Outstream.std_out))
(* end of arkady#*)

	    val line =
	      untabify (String.size prompt) (Instream.inputLine std_in)

	   (* The lines (and character positions) that we've got so far,
	      in reverse order (latest at front of list): *)
	    val oldLines =
	      case !theSource of SOURCE_TEXT{lines, ...} => rev lines

	   (* Absolute position of the new line: *)
	    val absPos =
	      case oldLines
		of nil => 0
		 | {absCharacter, line} :: _ => absCharacter + size line

	    val _ =
	      if Flags.DEBUG_ERRORPRINT then
		BasicIO.println("absPos(" ^ String.string line ^ ") = "
				^ Int.string absPos
			       )
	      else ()

	   (* The new list of lines: *)
	    val newLines = rev({absCharacter=absPos, line=line} :: oldLines)

	   (* Update the SourceText: *)
	    val _ = (theSource := SOURCE_TEXT{filename=name, lines=newLines})
	  in
	    line
	  end
      in
	SOURCE_READER{
	  name=name,
	  clearFn=fn () => (theSource := SOURCE_TEXT{filename=name, lines=nil}),
	  lexingFn=fn _ => getLine(),
	  positionFn=fn absPos => positionFn(theSource, absPos)
	}
      end

   (* highlight underlines part of a line. Oh, the line will probably
      still carry the trailing `\n'. *)

    fun highlight(line, column, width) =
      let
	val _ =
	  if Flags.DEBUG_ERRORPRINT then
	    BasicIO.println("highlight " ^ String.string line
			    ^ ", " ^ Int.string column
			    ^ " =-> " ^ Int.string width
			   )
	  else
	    ()

	val line = String.dropR "\n" line
	val width = if width = 0 then 1 else width
				(* Eliminate any 0-width error fields. *)
      in
	Report.indent(
	  String.size prompt,
	     Report.line line
	  // Report.line(String.create column " " ^ String.create width "^")
	)
      end
      
(*arkady#*)
    fun sourceFromPositions {left, right} =
      case (left, right) of
        (POSITION posLfn, POSITION posRfn) =>
        let
          val {file, line=line1, column=column1, getLine} = posLfn()
          val {line=line2, column=column2, ...} = posRfn()
          
          fun min n n' = if n < n' then n else n'
          
          fun findNoBlank ([], col) = col
            | findNoBlank (s::rest, col) =
                if s = " " then findNoBlank (rest, col + 1)
                else if s = "\t" then findNoBlank (rest, col + 5)
                     else col
          
          fun findIntend (fromL, toL, n) = 
                if fromL > toL then n
                else
                  let
                    val n' = findNoBlank (String.explode (getLine fromL), 0)
                    val n = min n' n
                  in
                    findIntend (fromL+1, toL, n)
                  end

          fun cutFrom   (line, from)     = String.extract from (String.size line) line
          fun cutTo     (line, to  )     = String.extract 0 to line
          fun cutFromTo (line, from, to) = String.extract from to line
          
          val line2start = findNoBlank (String.explode (getLine line2), 0)
          val cutBlanks = (findIntend (line1+1, line2-1, line2start))
          
          fun listLines (fromL, toL) =
               if fromL > toL then []
               else
                 let
                   val line   = getLine fromL
                   val line'  = String.extract cutBlanks (String.size line) line
                   val line'' = if line' = "\n" then " \n" else line'
                 in
                   line'' :: (listLines (fromL+1, toL))
                 end

          val spacesBeforeLine2 = if cutBlanks < line2start then String.create (line2start - cutBlanks) " " else ""
        in
          if line1 = line2 then
            [(String.dropR "\n" (cutFromTo (getLine line1, column1, column2))) ^ "\n"]
          else
            [((String.dropR "\n" (cutFrom (getLine line1, column1))) ^ "\n")]
            @ (listLines (line1 + 1, line2 - 1)) @
            [spacesBeforeLine2 ^ ((String.dropR "\n" (cutFromTo (getLine line2, line2start, column2))) ^ "\n")]
        end
      | (_, _) => ["[unknown source]\n"]
(*end of arkady#*)

    fun reportPosition{left, right} =
      case (left, right)
	of (POSITION posLfn, POSITION posRfn) =>
	     let
	       val {file, line=line1, column=column1, getLine} = posLfn()
	       val {line=line2, column=column2, ...} = posRfn()
				(* `file' and `getLine' had better match. *)

	       val theLine1 = getLine line1
	       val theLine2 = getLine line2
	       val len1 = String.size theLine1 - 1	(* remove `\n'. *)
	       val len2 = String.size theLine2 - 1

	       fun highlightAll(fromL, toL) =
		 if fromL > toL then
		   Report.null
		 else
		   let
		     val line = getLine fromL
		     val width = String.size line - 1
		   in
		        highlight(line, 0, width)
		     // highlightAll(fromL+1, toL)
		   end
	     in
	       Report.line(file ^ ", line " ^ Int.string line1
			   ^ ", column " ^ Int.string column1 ^ ":"
			  )
	       // (if line1 = line2 then
		     highlight(theLine1, column1, column2-column1)
		   else
			highlight(theLine1, column1, len1-column1)
		     // highlightAll(line1+1, line2-1)
		     // highlight(theLine2, 0, column2)
		  )
	     end

         | (_, _) =>
	     Report.line "[Position unknown]"

    type StringTree = PP.StringTree

    fun layoutPos(POSITION f) =
	  let
	    val {file, line, column, getLine} = f()
	  in
	    PP.NODE{start="POSITION(", finish=")",
		    indent=3, childsep=PP.RIGHT ",",
		    children=[PP.LEAF("\"" ^ file ^ "\""),
			      PP.LEAF("line " ^ Int.string line),
			      PP.LEAF("column " ^ Int.string column),
			      PP.LEAF("\"" ^ getLine line ^ "\"")
			     ]
		   }
	  end

      | layoutPos DUMMY = PP.LEAF "DUMMY"

    val FIELD_WIDTH = 18

    fun pad str = if size str mod 18 <> 0 then pad(str ^ " ") else str

    fun thinking x = () (*BasicIO.print(pad(x ^ " "))*)
    fun shifting x = () (*BasicIO.print(pad("S." ^ x ^ " "))*)
    fun reducing x = () (*BasicIO.print(pad("R." ^ x ^ " "))*)

    exception LEXICAL_ERROR of pos * string
			(* Used to signal lexical errors to the parser. *)
  end;
