(* Static objects - Definition pages 16-22 *)

(*
$File: Common/StatObject.sml $
$Date: 1992/12/30 14:48:20 $
$Revision: 1.18 $
$Locker: birkedal $
*)

(*$StatObject:
	SORTED_FINMAP IDENT VAR CON EXCON SCON LAB TYNAME TYCON
	TYVAR TIMESTAMP LIST_HACKS FLAGS REPORT PRETTYPRINT CRASH
	STATOBJECT STATOBJECT_PROP
*)

functor StatObject(structure SortedFinMap: SORTED_FINMAP
		   structure Ident: IDENT

		   structure Var: VAR
		     sharing type Var.id = Ident.id
			 and type Var.strid = Ident.strid
			 and type Var.longid = Ident.longid

		   structure Con: CON
		     sharing type Con.longid = Ident.longid

		   structure Excon: EXCON
		     sharing type Excon.longid = Ident.longid

		   structure SCon: SCON
		   structure Lab: LAB
		   structure TyName: TYNAME

		   structure TyCon: TYCON
		     sharing type TyCon.tycon = TyName.tycon

		   structure TyVar: TYVAR

		   structure Timestamp: TIMESTAMP
		   structure ListHacks: LIST_HACKS
		   structure Flags: FLAGS
		   structure Report: REPORT

		   structure PP: PRETTYPRINT
		     sharing type PP.Report = Report.Report
			 and type SortedFinMap.StringTree = PP.StringTree

		   structure Crash: CRASH
		  ) =
  struct
    (********
    Type variables
    *********
    We use the syntactic class of tyvars to
    represent explicit tyvars.
    ********)

    type SyntaxTyVar = TyVar.SyntaxTyVar

    datatype TyVar =
      ORDINARY of {id: int, equality: bool, imperative: bool, overloaded: bool}
    | EXPLICIT of SyntaxTyVar

    val isIn = General.uncurry List.member

   (* The rho and field variables for flexible records: *)

    datatype RowVar = ROWVAR of Timestamp.stamp
         and FieldVar = FIELDVAR of Timestamp.stamp

    (********
    Generate fresh type variables
    *********
    The id field must always be positive, negative id's are used for naming bound
    type variables in type schemes.
    ********)

   (* The timestamp id is an integer here, not a Timestamp.stamp because of
      the pos/neg convention which someone seems to be using. Blame David
      for this one, not me. *)

    local
      val r = ref 0
    in
      fun freshTyVar {equality, imperative, overloaded} =
	ORDINARY{id=(r := !r + 1; !r),
		 equality=equality,
		 imperative=imperative,
		 overloaded=overloaded
		}
    end

    fun refreshTyVar (ORDINARY{equality, imperative, overloaded, ...}) =
	freshTyVar {equality = equality, imperative = imperative, overloaded = overloaded}
      | refreshTyVar (EXPLICIT SyntaxTyVar) =
	let
	  val equality   = TyVar.isEquality   SyntaxTyVar
	  val imperative = TyVar.isImperative SyntaxTyVar
	in
	  freshTyVar {equality = equality, imperative = imperative, overloaded = false}
	end

    fun impTyVar (ORDINARY{imperative, ...}) = imperative
      | impTyVar (EXPLICIT syntaxTyVar) = TyVar.isImperative syntaxTyVar

    fun overloadedTyVar (ORDINARY{overloaded, ...}) = overloaded
      | overloadedTyVar _ = false

    (********
    Type constructors
    ********)

    type tycon = TyCon.tycon

    (********
    Make an explicit type variable
    ********)

    val mkExplicitTyVar = EXPLICIT

    val freshTyName = TyName.freshTyName

    (********
    Type names
    ********)

    type TyName = TyName.TyName

    (********
    Record labels
    ********)

    type lab  = Lab.lab

    (****
    Types
    ****)

   (* The record typing stuff comes pretty verbatim from some of Mads' example
      code - which is to say, I don't understand a lot of it. Here it is
      anyway; future generations might find it interesting. *)

   (* Records may be flexible, with rho variables terminating them. We also
      implement field variables, though I have yet to be convinced that they
      are necessary. The labels are ordered. *)

    datatype RecType = NILrec			(* Rigid. *)
	  	     | VARrec of RowVar		(* Flexible perhaps (depending
						   on whether the RowVar is
						   instanced...) *)
      		     | ROWrec of Lab.lab * Field * RecType

	 and Field = ABSENTfield	(* Wny not just "PRESENT('a)" ? *)
	           | PRESENTfield of Type
	           | VARfield of FieldVar

         and Type = TYVAR of TyVar
		  | ARROW of Type * Type
		  | RECTYPE of RecType
		  | CONSTYPE of Type list * TyName.TyName

    type FunType  = Type
     and ConsType = Type

    fun freshRow() = VARrec(ROWVAR(Timestamp.new()))
    and freshField() = VARfield(FIELDVAR(Timestamp.new()))

   (* For prettyprinting and the like it's most convenient to be able
      to change a RecType into a (lab, Field) SortedMap with optional
      RowVar. *)

    fun sanitiseRecType r
          : (Lab.lab, Field) SortedFinMap.map * RowVar Option =
      case r
	of NILrec => (SortedFinMap.empty, None)
	 | VARrec rv => (SortedFinMap.empty, Some rv)
	 | ROWrec(lab, field, r') =>
	     let
	       val (map, rvOpt) = sanitiseRecType r'
	     in
	       (SortedFinMap.add Lab.< (lab, field, map), rvOpt)
	     end

   (* Field insertion. The lab*field list must be ordered over the labs. *)

    fun insertFields(fields: (Lab.lab * Field) list, r): RecType =
      case (fields, r)
	of (nil, _) => r
	 | ((l, f) :: rest, NILrec) =>
	     ROWrec(l, f, insertFields(rest, NILrec))

	 | ((l, f) :: rest, VARrec v) =>
	     ROWrec(l, f, insertFields(rest, VARrec v))

	 | ((l, f) :: rest, ROWrec(l', f', r')) =>
	     if l = l' then
	       Crash.impossible "StatObject.insertFields"
	     else if Lab.<(l, l') then
	       ROWrec(l, f, insertFields(rest, r))
	     else
	       ROWrec(l', f', insertFields((l, f) :: rest, r'))

    fun addField (lab, ty) recType =
      insertFields([(lab, PRESENTfield ty)], recType)

   (* A sort-of compose function: apply a ty->ty to a record. *)

    fun recType_map (f: Type -> Type) (r: RecType): RecType =
      case r
	of NILrec => NILrec

	 | VARrec v => VARrec v

	 | ROWrec(lab, field, r') =>
	     ROWrec(lab,
		    case field
		      of ABSENTfield => ABSENTfield
		       | PRESENTfield ty => PRESENTfield(f ty)
		       | VARfield v => VARfield v,
		    recType_map f r'
		   )

   (* A sort-of fold function: apply a ty*'b->'b along a RecType,
      yielding a 'b. *)

    fun recType_fold (f: Type * 'b -> 'b) (x: 'b) (r: RecType): 'b =
      case r
	of NILrec => x
	 | VARrec _ => x
	 | ROWrec(lab, field, r') =>
	     case field
	       of ABSENTfield => recType_fold f x r'
	        | PRESENTfield ty => recType_fold f (f(ty, x)) r'
	        | VARfield _ => recType_fold f x r'

    (* existsRecVarsType t returns true iff there exists either a field or a 
       row variable in the type t *)
    fun existsRecVarsType t =
      case t of 
	(TYVAR _) => false
      | (ARROW(t1, t2)) => (existsRecVarsType t1) orelse (existsRecVarsType t2)
      | (RECTYPE r) => 
	  (case r of
	     NILrec => false
           | VARrec _ => true
           | ROWrec(_, f, r') => 
	       (case f of
		  ABSENTfield => false
		| PRESENTfield ty => existsRecVarsType ty
		| VARfield _ => true) 
	       orelse 
	       (case r' of
		  NILrec => false
		| VARrec _ => true
		| ROWrec _ => existsRecVarsType (RECTYPE r')
	       )
	  )
      | CONSTYPE(tylist, _) => List.exists existsRecVarsType tylist
	

    (********
    TypeSchemes
    *********
    The type variable list is such that the id's of ALL type variables are negative.
    This avoids any renaming to do with name capture and also gives a normal form
    for typeschemes so that equality can be used safely.
    ********)

    datatype TypeScheme = FORALL of TyVar list * Type

    (********
    Type functions
    *********
    The type variable list is such that the id's of ALL type variables are negative.
    This avoids any renaming to do with name capture and also gives a normal form
    for type functions so that equality can be used safely.
    ********)

    datatype TypeFcn = TYPEFCN of {tyvars: TyVar list, tau: Type}

    local
      val tv = freshTyVar{equality=false, imperative=false, overloaded=false}
    in
      val bogus_TypeFcn = TYPEFCN{tyvars=[tv], tau=TYVAR tv}
    end

    (********
    Toplevel printing and PrettyPrinter hooks
    ********)

   (* Type printing, including association info for type variable names.
      One of these is created for each top-level item printed; it is
      passed around and side-effected. *)

    datatype TVNames = NAMES of {tv: int, letter: int} list ref
		     | NONE		(* NONE -> don't bother. *)

    fun newTVNames() = NAMES(ref nil)

    val ordA = String.ord "a"

    fun pr_TyVarPRETTY names (ORDINARY{equality, imperative, overloaded, id}) =
          let
	    val boring =
	      if id < 0 then "S" ^ Int.string(~id) else "U" ^ Int.string id
	  in
	    (if overloaded then "OVERLOADED" else "") 
	    ^ (if equality then "''" else "'")
	    ^ (if imperative then "_" else "")
	    ^ (case names
		 of NAMES(L as ref L') =>
		      if Flags.DEBUG_TYVARS then
			boring
		      else
			(let
			   val {letter, ...} =
			     List.first (fn {tv, ...} => (tv = id)) L'
			 in
			   String.chr(ordA + letter)
			 end
			 handle List.First _ =>
			   let
			     val len = List.size L'
			   in
			     L := L' @ [{tv=id, letter=len}];
			     String.chr(ordA + len)
			   end
			)

		  | NONE =>
		      boring
	      )
	  end

      | pr_TyVarPRETTY _ (EXPLICIT syntaxTyVar) =
	  TyVar.pr_tyvar syntaxTyVar

    val pr_TyVar = pr_TyVarPRETTY NONE

    fun pr_RowVar(ROWVAR stamp) = "'r" ^ Timestamp.print stamp
    fun pr_FieldVar(FIELDVAR stamp) = "'f" ^ Timestamp.print stamp

    fun pr_TypePRETTY names ty =
      case ty
	of TYVAR tv =>
	     pr_TyVarPRETTY names tv

	 | RECTYPE r =>		(* See if we can print it as `a * b * ...'
				   rather than `{1: a, 2: b, ...}' *)
	     let
	       val (m, rv) = sanitiseRecType r
	     in
	       case (SortedFinMap.matches
		       (fn (i, lab) => Lab.is_LabN(lab, i+1)) m,
		     rv
		    )
		 of (true, None) =>	(* A possible (t1 * t2 * ...) type,
					   and no rowvar. *)
		   	(* Careful: "{1=x}" does *not* print as "(x)" (MacQueen
			   got this one wrong), and "{ }" should be "unit". We
			   don't do this folding at all if there's a row var. *)
		      let
			val fields = SortedFinMap.rangeSORTED m
		      in
			case fields
			  of nil => "unit"	(* Hard-wired *)
			   | [x] => "{1: " ^ pr_FieldPRETTY names x ^ "}"

			   | _ =>
			       List.stringSep "(" ")" " * "
					      (pr_FieldPRETTY names) fields
		      end

		  | _ =>		(* Have to do the general print. *)
		      let
			val finish =
			  case rv of Some v => " ... " ^ pr_RowVar v ^ "}"
			           | None => "}"
		      in
			PP.flatten1(
			  SortedFinMap.layoutMap
			    {start="{", eq=": ", sep=", ", finish=finish}
			    (PP.layoutAtom Lab.pr_Lab)
			    (PP.layoutAtom(pr_FieldPRETTY names))
			    m
			)
		      end
	     end

	 | ARROW(t1, t2) =>
	     implode ["(", pr_TypePRETTY names t1,
		      " -> ", pr_TypePRETTY names t2, ")"
		     ]

	 | CONSTYPE(tys, tyname) =>
	     (case tys
		of nil =>
		     TyName.pr_TyName tyname

		 | [ty] =>
		     implode [pr_TypePRETTY names ty, " ",
			      TyName.pr_TyName tyname
			     ]

		 | _ =>
		     implode [List.stringSep "(" ") " ", "
			        (pr_TypePRETTY names) tys,
			      TyName.pr_TyName tyname
			     ]
	     )

    and pr_FieldPRETTY names field =
      case field
	of ABSENTfield => "<absent>"
	 | PRESENTfield ty => pr_TypePRETTY names ty
	 | VARfield fv => pr_FieldVar fv

    val pr_Type = pr_TypePRETTY NONE

    fun pr_TypeSchemePRETTY names (FORALL(_, ty)) = pr_TypePRETTY names ty

    val pr_TypeScheme = pr_TypeSchemePRETTY NONE

    fun pr_TypeFcnPRETTY names (TYPEFCN{tyvars, tau}) =
      case tyvars
	of nil =>
	     {vars="", body=pr_TypePRETTY names tau}

	 | [tv] =>
	     {vars=pr_TyVarPRETTY names tv, body=pr_TypePRETTY names tau}

	 | _ =>
	     {vars=List.stringSep "(" ")" ", " (pr_TyVarPRETTY names) tyvars,
	      body=pr_TypePRETTY names tau
	     }

(* mju#1.6 *)
    fun pr_TypeFcn typefcn =
	let
	    val names = newTVNames()
	    val {body,...} = pr_TypeFcnPRETTY names typefcn
	in
	    body
	end
(* endmju#1.6 *)


    type StringTree = PP.StringTree

    fun layoutTyName tyname = PP.LEAF(TyName.pr_TyName tyname)
    and layoutTyVar tv = PP.LEAF(pr_TyVar tv)


    and layoutType(CONSTYPE([], tyname)) =
	  layoutTyName tyname

      | layoutType(CONSTYPE(ty_list, tyname)) = 
	PP.NODE{start="(", finish=") " ^ (TyName.pr_TyName tyname), indent=1,
		children=(map layoutType ty_list),
		childsep=PP.LEFT ", "
		}

      | layoutType(RECTYPE r) =
	  let
	    val (m, rv_opt) = sanitiseRecType r

	    val finish = case rv_opt of Some rv => " ... " ^ pr_RowVar rv ^ "}"
	      			      | None => "}"
	  in
	    SortedFinMap.layoutMap
	      {start="{", eq=" : ", sep=", ", finish=finish}
	      (PP.layoutAtom Lab.pr_Lab) layoutField m
	  end

      | layoutType(ARROW(ty, ty')) =
	  PP.NODE{start="(", finish=")", indent=1,
		  children=[layoutType ty, layoutType ty'],
		  childsep=PP.LEFT " -> "
		 }

      | layoutType(TYVAR tv) = layoutTyVar tv

    and layoutField ABSENTfield = PP.LEAF "<absent>"
      | layoutField(PRESENTfield ty) = layoutType ty
      | layoutField(VARfield fv) = PP.layoutAtom pr_FieldVar fv

    fun layoutTypeFcn(TYPEFCN {tyvars, tau}) =
      PP.NODE{start=List.stringSep "LAMBDA (" "). " ", " pr_TyVar tyvars,
	      finish="", indent=0, childsep=PP.NONE,
	      children=[layoutType tau]
	     }

    fun layoutTypeScheme(FORALL(_, tau)) = layoutType tau

    fun layoutTyNameSet tyname_list =
      PP.NODE{start="{", finish = "}", indent=1,
	      children=(map layoutTyName tyname_list),
	      childsep=PP.LEFT ", "
	      }
    (********
    Find free type variables
    ********)

    fun tyvarsTy (TYVAR tyvar) = [tyvar]
      | tyvarsTy (RECTYPE r) =
	  recType_fold (fn (ty, tyvars) => ListHacks.union(tyvarsTy ty, tyvars))
	  	       nil r

      | tyvarsTy (ARROW(ty,ty')) =
	  ListHacks.union(tyvarsTy ty, tyvarsTy ty')

      | tyvarsTy (CONSTYPE(types,_))= 
	  List.foldL
	  (fn ty => fn tyvars => ListHacks.union(tyvarsTy ty , tyvars))
	  [] types

    fun tyvarsTySch (FORALL(tyvarlist, ty)) =
      ListHacks.minus(tyvarsTy ty, tyvarlist)

    (********
    Get the overloaded tyvar if there is one (there should be at most one)
    ********)
    
    fun getOverloadedTyVar typ =
      let 
	val otvs = List.all overloadedTyVar (tyvarsTy typ)
      in
	case otvs of 
	  [] => None
	| [otv] => Some otv
	| _ => Crash.impossible "StatObject.getOverloadTyVar"
      end

    (********
     Find out whether a type is imperative or not
    ********)

    fun isImperativeType tau = 
      List.forAll impTyVar (tyvarsTy tau)

    (********
    Construct and destruct types
    ********)

    fun mkTypeTyVar tv = TYVAR tv
    and unTypeTyVar(TYVAR tv) = Some(tv)
      | unTypeTyVar _ = None

    val mkTypeRecType = RECTYPE

    fun unTypeRecType(RECTYPE t) = Some t
      | unTypeRecType _ = None

    fun mkTypeFunType fty = fty
    and unTypeFunType(ty as (ARROW(ty',ty''))) = Some(ty)
      | unTypeFunType(_) = None

    fun mkTypeConsType cty = cty
    and unTypeConsType(ty as (CONSTYPE(types, tyname))) = Some(ty)
      | unTypeConsType _ = None

    (********
    Construct and destruct record types
    ********)

    val emptyRecType = NILrec		(* "{}" *)
    val emptyFlexRecType = freshRow	(* "{...}" *)

    local
      val ONE = Lab.mk_IntegerLab 1
      and TWO = Lab.mk_IntegerLab 2
    in
      fun mkTypePair (ty,ty') =
	RECTYPE(addField (ONE, ty) (addField (TWO, ty') emptyRecType))

      and unRecPair r =
	    case sanitiseRecType r
	      of (m, None) =>
		   (case SortedFinMap.lookup m ONE
		      of None =>
			   Crash.impossible "StatObject.unRecPair(L=?)"

		       | Some ABSENTfield =>
			   Crash.impossible "StatObject.unRecPair(L=a)"

		       | Some(VARfield _) =>
			   Crash.impossible "StatObject.unRecPair(L=v)"

		       | Some(PRESENTfield tyL) =>
			   (case (SortedFinMap.lookup m TWO)
			      of None =>
				   Crash.impossible "StatObject.unRecPair(R=?)"

			       | Some ABSENTfield =>
				   Crash.impossible "StatObject.unRecPair(R=a)"

			       | Some(VARfield _) =>
				   Crash.impossible "StatObject.unRecPair(R=v)"

			       | Some(PRESENTfield tyR) => (tyL, tyR)
			   )
		   )

	       | (_, Some _) =>		(* It's flexible: punt *)
		   Crash.impossible "StatObject.unRecPair(flexible)"
    end

    fun sortedLabsOfRecType r =
          (case sanitiseRecType r
	     of (m, _) => SortedFinMap.domSORTED m
	  )

    val TypeUnit = RECTYPE emptyRecType

    (********
    Construct and destruct function types
    ********)

    fun mkFunType(ty,ty') = ARROW(ty,ty')

    and unFunType(ARROW(ty,ty')) = Some(ty,ty')
      | unFunType(_) = None

    (********
    Construct and destruct constructed types
    ********)

    fun mkConsType(typel, name) = CONSTYPE(typel,name)

    and unConsType(CONSTYPE(typel,name)) = Some(typel,name)
      | unConsType(_) = None

    (********
    Derived constructors and destructors
    ********)

    fun mkTypeRef t = CONSTYPE([t], TyName.tyName_REF)

    val TypeExn = CONSTYPE([], TyName.tyName_EXN)

    fun isTypeExn(CONSTYPE([], name)) = (name = TyName.tyName_EXN)
      | isTypeExn _ = false

    val mkTypeArrow = ARROW

    fun isTypeArrow(ARROW _) = true
      | isTypeArrow _ = false

    fun unTypeArrow(ARROW(t, t')) = Some(t, t')
      | unTypeArrow _ = None

    (********
    Special constants
    ********)

    type scon = SCon.scon

    (********
    Typing of special constants
    ********)

    val TypeInt    = CONSTYPE([], TyName.tyName_INT)
    and TypeReal   = CONSTYPE([], TyName.tyName_REAL)
    and TypeString = CONSTYPE([], TyName.tyName_STRING)

    fun GetTypescon scon =
      case scon
	of SCon.INTEGER _ => TypeInt
	 | SCon.STRING _  => TypeString
	 | SCon.REAL _    => TypeReal

    (*******
    The type bool
    *******)
    
    val TypeBool = CONSTYPE([], TyName.tyName_BOOL)

    (********
    Substitutions
    *********
    `Oldest' substitutions are at the head of the list
    ********)

    datatype SubstEntry = TYsubst of TyVar * Type
			| ROWsubst of RowVar * RecType
			| FIELDsubst of FieldVar * Field

    datatype Substitution = SUBSTITUTION of SubstEntry list

    (********
    The identity substitution
    ********)

    val Id: Substitution = SUBSTITUTION nil

    val bogus_Subst = Id

    (********
    Make a (type) substitution
    ********)

    fun typeSubstitution(tv: TyVar, ty: Type): Substitution =
      SUBSTITUTION [TYsubst(tv, ty)]

    (********
    Composition of substitutions.
    ********)

    infixr oo

    fun (S1: Substitution) oo (S2: Substitution): Substitution =
      case (S1, S2)
	of (SUBSTITUTION list1, SUBSTITUTION list2) =>
	     SUBSTITUTION(list2 @ list1)

    (********
    Apply a substitution to a type (infix operator)
    ********)

    infixr on

   (* Application of substitutions (of which there are three kinds: ty,
      row, field). *)

    fun (S: Substitution) on (ty: Type): Type =
      let
	fun tv_Subst (tv, ty) t =
	  let
	    val f = tv_Subst(tv, ty)
	  in
	    case t
	      of TYVAR tv' => if tv=tv' then ty else TYVAR tv'
	       | ARROW(ty1, ty2) => ARROW(f ty1, f ty2)
	       | RECTYPE r => RECTYPE(recType_map f r)
	       | CONSTYPE(types, tyname) => CONSTYPE(map f types, tyname)
	  end

	fun rv_Subst (rv, row) t =
	  let
	    val f = rv_Subst(rv, row)
	  in
	    case t
	      of TYVAR _ => t
	       | ARROW(ty1, ty2) => ARROW(f ty1, f ty2)

	       | RECTYPE r =>	(* I guess this is Remy's algorithm. *)
		   RECTYPE(
		     case r
		       of NILrec => NILrec
			| VARrec rv' => if rv = rv' then row else r
			| ROWrec(lab, field, r') =>
			    let
			      fun R(NILrec, acc: (Lab.lab * Field) list) =
				    merge_Rows(acc, NILrec)

				| R(r as VARrec rv', acc) =
				    if rv = rv' then
				      merge_Rows(acc, row)
				    else
				      merge_Rows(acc, r)

				| R(ROWrec(lab', field', r''), acc) =
				    R(r'', acc @ [(lab', F field')])

			      and F ABSENTfield = ABSENTfield
				| F(PRESENTfield ty) = PRESENTfield(S on ty)
				| F(f as VARfield _) = f
			    in
			      R(r', [(lab, F field)])
			    end
		   )

	       | CONSTYPE(types, tyname) => CONSTYPE(map f types, tyname)
	  end

	and merge_Rows(nil: (Lab.lab * Field) list, row2) =
	      row2

	  | merge_Rows((l, f) :: rest, NILrec) =
	      ROWrec(l, f, merge_Rows(rest, NILrec))

	  | merge_Rows((l, f) :: rest, VARrec rv) =
	      ROWrec(l, f, merge_Rows(rest, VARrec rv))

	  | merge_Rows(fields1 as (l1, f1) :: rest,
		       row2 as ROWrec(l2, f2, row2')
		      ) =
	      if l1 = l2 then Crash.impossible "merge_Rows"
	      else if Lab.<(l1, l2)
	      then ROWrec(l1, f1, merge_Rows(rest, row2))
	      else ROWrec(l2, f2, merge_Rows(fields1, row2'))

	fun fv_Subst (fv, field) t =
	  let
	    val f = fv_Subst(fv, field)

	    fun fv_Subst_row row =
	      case row
		of NILrec => NILrec
		 | VARrec rv => VARrec rv
		 | ROWrec(lab, field', row') =>
		     ROWrec(lab,
			    case field'
			      of ABSENTfield => ABSENTfield
			       | PRESENTfield ty => PRESENTfield(f ty)
			       | VARfield fv' =>
				   if (fv = fv') then field else field',
			    fv_Subst_row row'
			   )
	  in
	    case t
	      of TYVAR _ => t
	       | ARROW(ty1, ty2) => ARROW(f ty1, f ty2)
	       | RECTYPE r => RECTYPE(fv_Subst_row r)
	       | CONSTYPE(types, tyname) => CONSTYPE(map f types, tyname)
	  end
      in
	case S of SUBSTITUTION items =>
	  List.foldL
	  (fn TYsubst(tv, ty) => tv_Subst(tv, ty)
	    | ROWsubst(rv, row) => rv_Subst(rv, row)
	    | FIELDsubst(fv, field) => fv_Subst(fv, field)
	  ) ty items
      end

    fun S_on_Row(S, row: RecType): RecType =
      case S on RECTYPE row
	of RECTYPE row' => row'
         | _ => Crash.impossible "S_on_Row"

    (********
    Type schemes
    ********)

    local
      (********
      Rename a list of TyVars
      *********
      This function is used to rename bound TyVars so as to obtain
      a normal form for type schemes.
      ********)

      fun rename_TypeScheme_TyVars (tyvar_list : TyVar list) : TyVar list * Substitution =
	let
	  fun rename [] x = []
	    | rename ((tv as ORDINARY{equality, imperative, overloaded, ...}) :: rest) id =
	      (tv, ORDINARY {id = id, equality = equality, imperative = imperative, overloaded = overloaded})
	      :: rename rest (id - 1)
	    | rename ((tv as EXPLICIT syntaxTyVar) :: rest) id =
	      let
		val equality   = TyVar.isEquality   syntaxTyVar
		val imperative = TyVar.isImperative syntaxTyVar
	      in
		(tv, ORDINARY{id = id, equality = equality, imperative = imperative, overloaded = false})
		 :: rename rest (id - 1)
	      end

	  val renaming_list : (TyVar * TyVar) list  = rename tyvar_list (~1)
	  val renamed_tyvars = map #2 renaming_list

	  val S = List.foldL
	    (fn (tv1, tv2) => fn S => typeSubstitution(tv1, mkTypeTyVar tv2) oo S)
	       Id renaming_list
	in
	  (renamed_tyvars, S)
	end
    in
      fun mkTypeScheme (tyvar_list, ty) =
	let
	  val (renamed_tyvars, S) = rename_TypeScheme_TyVars tyvar_list
	in
	  FORALL(renamed_tyvars, S on (ty: Type))
	end

      and unTypeScheme (FORALL(tyvar_list, ty)) = (tyvar_list, ty)

      (********
      Make a type into a typescheme with no bound variables
      ********)

      fun Type_in_TypeScheme (ty : Type) = FORALL([], ty)

      val Type_in_TypeScheme =
	if Flags.DEBUG_TYPES then
	  (fn ty =>
	     let
	       val sigma = Type_in_TypeScheme ty
		 
	       val tree =
		 PP.NODE{start="Type_in_TypeScheme: ", finish="", indent=0,
			 children=[layoutType ty, layoutTypeScheme sigma],
			 childsep=PP.LEFT " => "
			}
			   
	       val report = PP.reportStringTree tree
	     in
	       Report.print report;
	       sigma
	     end
	   )
	else
	  Type_in_TypeScheme
    end
 
    (********
    Close a typescheme using the specified type variables,
    do NOT quantify over overloaded type variables.
    ********)

    fun Close (tyvar_list, FORALL(_, ty)) : TypeScheme =
      let 
        val notoverloaded_tyvar_list =  List.all (fn tv => not (overloadedTyVar tv)) tyvar_list
      in
        mkTypeScheme(notoverloaded_tyvar_list, ty)
      end

    (********
    Get an instance of a TypeScheme
    ********)

    fun instance (FORALL(tyvars, ty)) =
      let
	val S =
	  List.foldL
	  (fn tv => fn S => typeSubstitution(tv, mkTypeTyVar(refreshTyVar tv)) oo S)
	  Id tyvars
      in 
	S on ty
      end

    (********
    Apply a substitution to a type scheme
    ********)

    fun onScheme (S : Substitution, FORALL(bounds, ty)) =
      FORALL(bounds, S on ty)

    (********
    Rename a list of TyVars
    *********
    This function is used to rename bound TyVars so as to obtain
    a normal form for type functions.
    ********)

    fun rename_TypeFcn_TyVars (tyvar_list : TyVar list) : TyVar list * Substitution =
      let
	fun rename [] id = []
	  | rename (tv :: rest) id =
	    (tv, ORDINARY{id = id, equality = false, imperative = false, overloaded = false})
	     :: rename rest (id - 1)

	val renaming_list  = rename tyvar_list (~1)
	val renamed_tyvars = map #2 renaming_list

	val S = List.foldL
	  (fn (tv1, tv2) => fn S => typeSubstitution(tv1, mkTypeTyVar tv2) oo S)
	  Id renaming_list
      in
	(renamed_tyvars, S)
      end

    local
      (********
      Local exception which is raised when a type cannot admit equality
      ********)

      exception NotEquality

      (********
      Make a type into an equality type (if possible)
      ********)

      fun make_eq(TYVAR(tv as ORDINARY{imperative, overloaded, ...})) =
	  let
	    val tv' = 
	      freshTyVar{equality=true, imperative=imperative, overloaded=overloaded}
	  in
	    typeSubstitution(tv, TYVAR tv')
	  end

	| make_eq(TYVAR(EXPLICIT syntaxTyVar)) =
	    if TyVar.isEquality syntaxTyVar then Id else raise NotEquality

	| make_eq(RECTYPE r) =
	    recType_fold (fn (tau, S) => S oo make_eq(S on tau)) Id r

	| make_eq(CONSTYPE(ty_list, tyname)) =
	    if tyname = TyName.tyName_REF then
	      Id		(* "ref" is a special case; take it out
				   straight away, otherwise we'll damage
				   any tyvars within the arg to "ref". *)
	    else if TyName.equality(tyname) then
	      List.foldL
	      (fn tau => fn S => S oo make_eq(S on tau))
	      Id ty_list
	    else
	      raise NotEquality

	| make_eq(ARROW _) = raise NotEquality
    in
      fun make_equality tau =
	Some(make_eq tau) handle NotEquality => None
    end

    (*******
    Unify types (tau,tau') s.t. range (subst) intersection restr_tyvars = empty
    *******)

    fun restricted_unify (restricted_tyvars : TyVar list) (tau, tau') =
    let

      (********
      This local exception is raised when a unification error occurs
      ********)

      exception Unify

      (********
      The `occurs' check for tyvars in tys
      ********)

      fun occurs_tv_in_Type(tv: TyVar, ty: Type): bool =
	let
	  fun occursType(TYVAR tv') =
	        (tv = tv')
	    | occursType(ARROW(ty1, ty2)) =
	        occursType ty1 orelse occursType ty2
	    | occursType(CONSTYPE(tys, _)) =
	        List.foldL (fn ty => fn b => b orelse occursType ty) false tys
	    | occursType(RECTYPE r) =
	        recType_fold (fn (ty, b) => b orelse occursType ty) false r
	in
	  occursType ty
	end

     (* row-var and field-var occurs checks are more complex since the fields
        contain types which contain records which ... and so on. Actually, I'm
	not sure I see why an occurs-check on rv's and fv's is needed... *)

      fun occurs_rv_in_RecType(rv, row) =
	case row
	  of NILrec => false
	   | VARrec rv' => (rv = rv')
	   | ROWrec(_, f, row') => occurs_rv_in_Field(rv, f)
				   orelse occurs_rv_in_RecType(rv, row')

      and occurs_rv_in_Field(rv, field) =
	case field
	  of ABSENTfield => false
	   | PRESENTfield ty => occurs_rv_in_Type(rv, ty)
	   | VARfield _ => false

      and occurs_rv_in_Type(rv, ty) =
	case ty
	  of TYVAR _ => false

	   | ARROW(ty1, ty2) => occurs_rv_in_Type(rv, ty1)
	       			orelse occurs_rv_in_Type(rv, ty2)

	   | RECTYPE row => occurs_rv_in_RecType(rv, row)

	   | CONSTYPE(tys, _) =>
	       List.foldL
	       (fn ty => fn result => occurs_rv_in_Type(rv, ty) orelse result)
	       false tys

      fun occurs_fv_in_Field(fv, field) =
	case field
	  of ABSENTfield => false
	   | PRESENTfield ty => occurs_fv_in_Type(fv, ty)
	   | VARfield _ => false

      and occurs_fv_in_Type(fv, ty) =
	case ty
	  of TYVAR _ => false

	   | ARROW(ty1, ty2) => occurs_fv_in_Type(fv, ty1)
				orelse occurs_fv_in_Type(fv, ty2)

	   | RECTYPE row => occurs_fv_in_RecType(fv, row)

	   | CONSTYPE(tys, _) =>
	       List.foldL
	       (fn ty => fn result => occurs_fv_in_Type(fv, ty) orelse result)
	       false tys

      and occurs_fv_in_RecType(fv, row) =
	case row
	  of NILrec => false
	   | VARrec _ => false
	   | ROWrec(_, f, row') => occurs_fv_in_Field(fv, f)
	       			   orelse occurs_fv_in_RecType(fv, row')

      (*******
      Make a type tau into an overloaded type iff tau is a type variable exluding
      explicit type variables which yields unification error.
      Remark: It is superfluous to mark every type variable as overloaded in an ARROW, CONS or
      RECTYPE, hence not done, as substituting such a type for an overloaded
      type variable is catered for in the resolvation phase.
      ********)

      fun make_overloaded (TYVAR (tv as ORDINARY{equality, imperative, ...})) =
	  typeSubstitution(tv, 
		 (mkTypeTyVar o freshTyVar)
		    {equality = equality, imperative = imperative, overloaded = true})
	| make_overloaded (TYVAR (EXPLICIT _)) = raise Unify
	| make_overloaded _ = Id

      (********
      Make a TyVar into an imperative TyVar (if possible)
      ********)

      fun makeImpTyVar (ORDINARY{equality, overloaded, ...}) =
	  freshTyVar {equality = equality, imperative = true, overloaded = overloaded}

	| makeImpTyVar (EXPLICIT syntaxTyVar) =
	  if TyVar.isImperative syntaxTyVar then
	    EXPLICIT syntaxTyVar
	  else
	    raise Unify

      (********
      Make a type into an imperative type (if possible)
      ********)

      fun make_imperative(TYVAR tv) =
	    typeSubstitution(tv, TYVAR(makeImpTyVar tv))

	| make_imperative(RECTYPE r) =
	    recType_fold (fn (tau, S) => S oo make_imperative(S on tau)) Id r

	| make_imperative(CONSTYPE(ty_list, tyname)) =
	    List.foldL	
	    (fn tau => fn S => S oo make_imperative(S on tau)) Id ty_list

	| make_imperative(ARROW(tau, tau')) =
	    let
	      val S  = make_imperative tau
	      val S' = make_imperative (S on tau')
	    in
	      S' oo S
	    end

      (********
      Check the attributes of a TyVar are satisfied
      *********
      We assume the `occurs' check has already been done
      ********)

      fun checkAttributes (r as {equality, imperative, overloaded, ...}, tau) =
	let
	  val S =
	    if equality then
	      case make_equality(tau) of
		Some(S) => S | None => raise Unify
	    else
	      Id

	  val S' =
	    if imperative then
	      make_imperative(S on tau)
	    else
	      Id

	  val S'' = 
            if overloaded then 
	      make_overloaded( (S' oo S) on tau)
            else
	      Id

	  val tyvar =
	    ORDINARY r
	in
	  if List.member tyvar restricted_tyvars then
	    raise Unify
	  else
	    typeSubstitution(ORDINARY r, (S'' oo S' oo S) on tau) oo S'' oo S' oo S
	end

      (********
      Check if we can safely unify an explicit TyVar and type
      ********)

      fun unifyExplicit (syntaxTyVar, TYVAR(EXPLICIT syntaxTyVar')) =
	  if syntaxTyVar = syntaxTyVar' then Id else raise Unify

        (* Notice below that unification of overloaded type variable and
	   explicit type variable is not allowed *)

	| unifyExplicit (syntaxTyVar, TYVAR(tv as ORDINARY r)) =
	  let
	    val {equality, imperative, overloaded, ...} = r
	    val S =
	      if List.member tv restricted_tyvars then raise Unify
	      else if overloaded then raise Unify
	      else typeSubstitution(ORDINARY r, TYVAR(EXPLICIT syntaxTyVar))
	  in
	    case
	      (equality, imperative,
	       TyVar.isEquality syntaxTyVar,
	       TyVar.isImperative syntaxTyVar)
	    of
	      (false, false,  _  ,  _  ) => S
	    | ( true, false, true,  _  ) => S
	    | (false,  true,  _  , true) => S
	    | (true , true , true, true) => S
	    | ( _   ,  _   ,  _  ,  _  ) => raise Unify
	  end

	| unifyExplicit (_, _) = raise Unify

      (********
      Check if we can safely unify a TyVar and Type
      ********)

      fun unifyTyVar (EXPLICIT syntaxTyVar, tau') =
	    let
	      fun legal (tau' as TYVAR(tv)) subst =
		  if List.member tv restricted_tyvars then raise Unify
		  else subst
		| legal tau' subst = subst
	    in
	      legal tau' unifyExplicit(syntaxTyVar, tau')
	    end

	| unifyTyVar (tv as ORDINARY r, TYVAR(tv' as ORDINARY r')) =
	    if tv <> tv' then
	      if List.member tv restricted_tyvars then
		if List.member tv' restricted_tyvars then
		  raise Unify
		else
		  checkAttributes(r', TYVAR tv)
	      else
		checkAttributes(r, TYVAR tv')
	    else
	      Id

	| unifyTyVar (tv as ORDINARY r, tau') =
	    if occurs_tv_in_Type(tv, tau')
	       orelse List.member tv restricted_tyvars
	    then raise Unify
	    else checkAttributes(r, tau')

      (********
      Unify two types
      ********)

      fun unifyType(TYVAR tv, tau') = unifyTyVar(tv, tau')
	| unifyType(tau, TYVAR tv') = unifyTyVar(tv', tau)
	| unifyType(RECTYPE r, RECTYPE r') = unifyRecType(r, r')
	| unifyType(ARROW pair, ARROW pair') = unifyFunType(pair, pair')
	| unifyType(CONSTYPE pair, CONSTYPE pair') = unifyConsType(pair, pair')
	| unifyType(_, _) = raise Unify

      and unifyRecType(r1, r2) = unifyRow(r1, r2)
(* --Birkedal --- why on earth is the code duplicated ??
(r1, r2) =	(* This lifted from Mads' code. *)
	case (r1, r2)
	  of (NILrec, NILrec) => Id
	   | (row, VARrec rv) => rowSubstitution(rv, row)
	   | (VARrec rv, row) => rowSubstitution(rv, row)

	   | (ROWrec(lab1, field1, r1'), ROWrec(lab2, field2, r2')) =>
	       if lab1 = lab2 then
		 let
		   val S1 = unifyField(field1, field2)
		   val S2 = unifyRecType(S_on_Row(S1, r1'), S_on_Row(S1, r2'))
		 in
		   S2 oo S1
		 end
	       else if Lab.<(lab1, lab2) then	(* Pad record 2, try again *)
		 let
		   val (S, r2') = extract(lab1, r2)
		 in
		   unifyRow(S_on_Row(S, r1), r2') oo S
		 end
	       else				(* Pad record 1, try again *)
		 let
		   val (S, r1') = extract(lab2, r1)
		 in
		   unifyRow(r1', S_on_Row(S, r2)) oo S
		 end

	   | (ROWrec(lab1, field1, r1'), NILrec) =>
	       let				(* Pad record 2, try again *)
		 val (S, r2') = extract(lab1, r2)
	       in
		 unifyRecType(S_on_Row(S, r1), r2') oo S
	       end

	   | (NILrec, ROWrec(lab2, field2, r2')) =>
	       let				(* Pad record 1, try again *)
		 val (S, r1') = extract(lab2, r1)
	       in
		 unifyRecType(r1', S_on_Row(S, r2)) oo S
	       end
--Birkedal 
*)

      and rowSubstitution(rv: RowVar, row: RecType): Substitution =
	if Flags.DEBUG_FLEXRECORDS then
	  let
	    val t = PP.NODE{start="rowSubstitution(", finish=")", indent=0,
			    children=[PP.layoutAtom pr_RowVar rv,
				      layoutType(RECTYPE row)
				     ],
			    childsep=PP.LEFT " => "
			   }
	  in
	    Report.print(PP.reportStringTree t);
	    rowSubstitution'(rv, row)
	  end
	else
	  rowSubstitution'(rv, row)

      and rowSubstitution'(rv, row) =
	if occurs_rv_in_RecType(rv, row)
	then raise Unify
	else SUBSTITUTION [ROWsubst(rv, row)]

      and fieldSubstitution(fv: FieldVar, field: Field): Substitution =
	if Flags.DEBUG_FLEXRECORDS then
	  let
	    val t = PP.NODE{start="fieldSubstitution(", finish=")", indent=0,
			    children=[PP.layoutAtom pr_FieldVar fv,
				      layoutField field
				     ],
			    childsep=PP.LEFT " => "
			   }
	  in
	    Report.print(PP.reportStringTree t);
	    fieldSubstitution'(fv, field)
	  end
	else
	  fieldSubstitution'(fv, field)

      and fieldSubstitution'(fv, field) =
	if occurs_fv_in_Field(fv, field)
	then raise Unify
	else SUBSTITUTION [FIELDsubst(fv, field)]

      and extract(lab: Lab.lab, row: RecType): Substitution * RecType =
	let
	  val _ =
	    if Flags.DEBUG_FLEXRECORDS then
	      let
		val tree =
		  PP.NODE{start="extract: ", finish="", indent=0,
			  children=[PP.LEAF(Lab.pr_Lab lab),
				    layoutType(RECTYPE row)
				   ],
			  childsep=PP.LEFT " from "
			 }
	      in
		Report.print(PP.reportStringTree tree)
	      end
	    else ()

	  fun rowVar NILrec = None
	    | rowVar(VARrec rv) = Some rv
	    | rowVar(ROWrec(_, _, row')) = rowVar row'

	  val result =
	    case rowVar row
	      of None => (Id, ROWrec(lab, ABSENTfield, row))
	       | Some rv =>
		   let
		     val littleRow = ROWrec(lab, freshField(), freshRow())
		     val S = rowSubstitution(rv, littleRow)
		   in
		     (S, S_on_Row(S, row))
		   end
	in
	  if Flags.DEBUG_FLEXRECORDS then
	    let
	      val tree =
		PP.NODE{start="giving:  ", finish="", indent=0,
			children=[layoutType(RECTYPE(#2 result))],
			childsep=PP.NONE
		       }
	    in
	      Report.print(PP.reportStringTree tree);
	      result
	    end
	  else
	    result
	end

      and unifyRow(row1: RecType, row2: RecType): Substitution =
	if Flags.DEBUG_FLEXRECORDS then
	  let
	    val t = PP.NODE{start="unifyRow(", finish=")", indent=0,
			    children=[layoutType(RECTYPE row1),
				      layoutType(RECTYPE row2)
				     ],
			    childsep=PP.RIGHT "; "
			   }
	  in
	    Report.print(PP.reportStringTree t);
	    unifyRow'(row1, row2)
	  end
	else
	  unifyRow'(row1, row2)

      and unifyRow'(row1, row2) =
	case (row1, row2)
	  of (NILrec, NILrec) => Id
	   | (_, VARrec i) => rowSubstitution(i, row1)
	   | (VARrec i, _) => rowSubstitution(i, row2)

	   | (ROWrec(lab1, field1, row1'), ROWrec(lab2, field2, row2')) =>
	       if lab1 = lab2 then
		 let
		   val S1 = unifyField(field1, field2)
		   val S2 = unifyRow(S_on_Row(S1, row1'), S_on_Row(S1, row2'))
		 in
		   S2 oo S1
		 end
	       else if Lab.<(lab1, lab2) then	(* Pad out row2, try again *)
		 let
		   val (S, row2') = extract(lab1, row2)
		 in
		   unifyRow(row2', S_on_Row(S, row1) (*, row2'*)) oo S
		 end
	       else				(* Pad out row1, try again *)
		 let
		   val (S, row1') = extract(lab2, row1)
		 in
		   unifyRow(row1', S_on_Row(S, row2)) oo S
		 end

	   | (ROWrec(lab1, field1, row1'), NILrec) =>
	       let				(* Pad out row2, try again *)
		 val (S, row2') = extract(lab1, row2)
	       in
		 unifyRow(row2', S_on_Row(S, row1) (*, row2'*)) oo S
	       end

	   | (NILrec, ROWrec(lab2, field2, row2')) =>
	       let				(* Pad out row1, try again *)
		 val (S, row1') = extract(lab2, row1)
	       in
		 unifyRow(row1', S_on_Row(S, row2)) oo S
	       end

      and unifyField(ABSENTfield, ABSENTfield) = Id
        | unifyField(PRESENTfield ty1, PRESENTfield ty2) = unifyType(ty1, ty2)
	| unifyField(VARfield i, field2) = fieldSubstitution(i, field2)
	| unifyField(field1, VARfield i) = fieldSubstitution(i, field1)
	| unifyField _ = raise Unify

      and unifyFunType((tau1, tau2), (tau3, tau4)) =
	let
	  val S1 = unifyType(tau1, tau3)
	  val S2 = unifyType(S1 on tau2, S1 on tau4)
	in
	  S2 oo S1
	end

      and unifyConsType((ty_list, tyname), (ty_list', tyname')) =
	if tyname = tyname' then
	  (* Note that tyname=tyname' implies length(ty_list)=length(ty_list') *)
	  List.foldL
	  (fn (tau, tau') => fn S => S oo unifyType(S on tau, S on tau'))
	  Id (ListPair.zip(ty_list, ty_list'))
	else
	  raise Unify
    in
      (********
      Unify two types, handling the exception generated if we fail
      ********)
	Some(unifyType(tau,tau'))
	handle Unify => None
    end

    val unify = restricted_unify nil


    (********
    TypeSchemes : Definition p 19 sec 4.5
    ********)

    fun TySch_generalises_Type (sigma, tau') : bool =
      let
	val tau = instance sigma
	val fv_tau' = tyvarsTy tau'
      in
    (* prohibit substitutions to be made on fv_tau ty -
     * from definition of generalisation (defn. sec 4.5 pg 19)
     *       sigma >- tau' if a subst can be found s.t. 
     * (subst on tau) = tau' (with restrictions on attributes) *)

	  case restricted_unify fv_tau' (tau,tau') of
	      Some _ => true
	    | None   => false
      end

    fun TySch_generalises_TySch (sigma, sigma') : bool =
      let
	val tau' = instance sigma'
      in
	TySch_generalises_Type (sigma,tau')
      (* note that (instance sigma') renames all tyvars in sigma and so 
       * no bound tyvar in sigma can be contained in the bound name instances
       * of tau' *)
      end

    (********
    Unqualified identifiers
    ********)

    type id = Ident.id

    (********
    Function for decomposing a qualified identifier and a qualified var
    (pre-elab and post-elab respectively).
    ********)

    val decomposeLongId  = Ident.decompose
    val decomposeLongVar = Var.decompose

    (********
    Unqualified variables
    ********)

    type var = Var.var

    (********
    Conversion of ids to vars
    ********)

    val mk_var = Var.mkVar

    (********
    Qualified identifiers
    ********)

    type longid = Ident.longid

    (********
    Qualified variables
    ********)

    type longvar = Var.longvar

    (********
    Qualified constructors
    ********)

    type longcon = Con.longcon

    (********
    Qualified exception constructors
    ********)

    type longexcon = Excon.longexcon

    (********
    Conversion of longids to longvars, longcons and longexcons
    ********)

    val mk_longvar   = Var.mk_longvar
    and mk_longcon   = Con.mk_longcon
    and mk_longexcon = Excon.mk_longexcon

    (********
    Structure identifiers
    ********)

    type strid = Ident.strid

    (********
    Bogus values used to build dummy nodes on the syntax tree
    ********)

    val bogusVar   = Var.bogus
    and bogusCon   = Con.bogus
    and bogusExcon = Excon.bogus

    (********
    TyName sets
    ********)

    type TyNameSet = TyName list

    fun singleTyNameSet t = [t]

    fun isemptyT nil    = true
      | isemptyT (_::_) = false

    val emptyTyNameSet     = nil
    and TyNameSetUnion     = ListHacks.union
    and TyNameSetMinus     = ListHacks.minus
    and TyNameSetIntersect = ListHacks.intersect
    and TyNameSetFold      = List.foldL o General.curry
    and eqTyNameSet        = ListHacks.eqSet

    (********
    Get TyNames
    ********)

    fun TyNamesTy(TYVAR _) =
          emptyTyNameSet

      | TyNamesTy(RECTYPE r) =
	  recType_fold
	  (fn (ty, tynames) => TyNameSetUnion(TyNamesTy ty, tynames))
	  emptyTyNameSet r

      | TyNamesTy(ARROW(ty, ty')) =
	  TyNameSetUnion(TyNamesTy ty, TyNamesTy ty')

      | TyNamesTy(CONSTYPE(types, tyname)) = 
	  List.foldL
	  (fn ty => fn tynames => TyNameSetUnion(TyNamesTy ty, tynames))
	  [tyname] types

    fun TyNamesTySch (FORALL(_, ty)) = TyNamesTy ty

    fun mkTypeFcn(tyvar_list: TyVar list, tau: Type) =
      let
	val (renamed_tyvars, S) = rename_TypeFcn_TyVars tyvar_list
      in
	TYPEFCN{tyvars = renamed_tyvars, tau = S on tau}
      end

    (*******
    We need only check if the type in an type function admits equality because
    the bound type variables have already been renamed to admit equality.
    *******)

    fun admits_equality (TYPEFCN {tau, ...}) : bool =
      case (make_equality tau) of
	Some _ => true
      | None => false

    fun TyNamesTypeFcn (TYPEFCN {tau,...}) = TyNamesTy tau

    fun grounded_TypeFcn (typefcn : TypeFcn, tynameset : TyNameSet) : bool =
      case (TyNameSetMinus(TyNamesTypeFcn typefcn, tynameset)) of
	[] => true
      | _  => false

    fun TyName_in_TypeFcn (tyname : TyName) : TypeFcn =
      let
	fun make_list 0 = []
	  | make_list x =
	    let
	      val tyvar = 
		freshTyVar {equality = false, imperative = false, overloaded = false}
	    in
	      tyvar :: make_list (x-1)
	    end

	val tyvar_list = make_list (TyName.arity tyname)
	val type_list  = map mkTypeTyVar tyvar_list
	val constype   = mkConsType(type_list, tyname)
	val tau = mkTypeConsType constype
      in
	mkTypeFcn(tyvar_list, tau)
      end

    fun applyTypeFcn (TYPEFCN {tyvars, tau}, tau_list : Type list) : Type =
      let
	val renaming_list  = map (fn tv => (tv, refreshTyVar tv)) tyvars

	val S = List.foldL
	  (fn (tv1, tv2) => fn S => typeSubstitution(tv1, mkTypeTyVar tv2) oo S)
	  Id renaming_list
	val tau' = S on tau

	val pairs =
	  ListPair.zip(map #2 renaming_list, tau_list)
	val S' =
	  List.foldL
	  (fn (tv, tau) => fn S => typeSubstitution(tv,tau) oo S)
	  Id pairs
      in
	S' on tau'
      end

    fun arity_TypeFcn (TYPEFCN {tyvars, tau}) : int =
      List.size tyvars

    fun unTyName_TypeFcn (TYPEFCN {tyvars, tau}) : TyName Option =
	case tau of
	    CONSTYPE (_,t) => Some t
	  | _              => None

    (*********
    Type realisations (Definition pg.33 secn. 5.6)
    *********)

    type tyrea = TyName -> TypeFcn

    val id_tyrea = TyName_in_TypeFcn

    fun mktyrea (t,theta) =
      fn t' => if t=t' then theta else id_tyrea t'

    fun mktyrea_class (ts, theta) =
      fn t => if List.member t ts then theta else id_tyrea t

    fun restrict_tyrea tset tyrea =
      fn t => if List.member t tset then tyrea t else id_tyrea t

    fun tyrea_on_TyName tyrea t = tyrea t

    and tyrea_on_Type tyrea ty =
      case ty
	of ty' as TYVAR _ => ty'
         | RECTYPE r =>
	     RECTYPE(recType_map (tyrea_on_Type tyrea) r)

         | ARROW(ty1, ty2) =>
	     ARROW(tyrea_on_Type tyrea ty1, tyrea_on_Type tyrea ty2)

         | CONSTYPE(tylist, t) =>
            (* definition pg 19. sec. 4.4 Types and Type Functions
	       beta-conversion carried out after substitution in substituting
	       type functions for type names *)
	     let
	       val theta = tyrea_on_TyName tyrea t
	     in
	       applyTypeFcn(theta, map (tyrea_on_Type tyrea) tylist)
	     end

    fun tyrea_on_TypeFcn tyrea (TYPEFCN {tyvars, tau}) =
      TYPEFCN {tyvars = tyvars, tau = tyrea_on_Type tyrea tau}

    fun tyrea_on_TypeScheme tyrea (FORALL (tyvars, tau)) =
      FORALL(tyvars, tyrea_on_Type tyrea tau)

    fun oo_tyrea (tyrea1 : tyrea, tyrea2 : tyrea) : tyrea =
      (tyrea_on_TypeFcn tyrea1) o (tyrea_on_TyName tyrea2)

    (********
    Function which generates a type realisation from a TyNameSet.
    *********
    All the TyNames in the TyNameSet are renamed to fresh TyNames with
    their equality attributes set to be true.
    ********)

    local
      fun make_equality tyname =
	let
	  val name    = TyName.name tyname
	  val arity   = TyName.arity tyname
	  val tyname' =
	    TyName.freshTyName {name = name, arity = arity, equality = true}
	  val tyrea =
	    mktyrea(tyname, TyName_in_TypeFcn tyname')
	in
	  tyrea
	end
    in
      fun generate_tyrea (tyname_list : TyNameSet) : tyrea =
	List.foldL
	(fn tyname => fn tyrea => oo_tyrea(tyrea, make_equality tyname))
	id_tyrea tyname_list
    end

    (********
    Function which checks if the typescheme of a contructor violates equality
    *********
    Respecting equality for constructors is as defined in the Definition on page 21.
    All the tynames in the TyNameSet are considered to admit equality. No explicit
    type variables should ever be present.
    ********)

    fun violates_equality (TyNameSet : TyNameSet, ts : TypeScheme) : bool =
      let
	fun violates(TYVAR(ORDINARY _)) = false

	  | violates(TYVAR(EXPLICIT _)) =
	      Crash.impossible "StatObject.violates"

	  | violates(RECTYPE r) =
	      recType_fold
	      (fn (tau, res) => res orelse violates tau)
	      false r

	  | violates(CONSTYPE(ty_list, tyname)) =
	      if TyName.equality(tyname) orelse isIn(tyname, TyNameSet) then
		List.foldL
		(fn tau => fn res => res orelse violates tau)
		false ty_list
	      else
		true

	  | violates(ARROW _) = true

	val (tyvar_list, tau) = unTypeScheme ts
      in
	case unTypeArrow(tau) of
	  None =>
	    false
	| Some(tau', _) =>
	    let
	      val TYPEFCN{tau, ...} = mkTypeFcn(tyvar_list, tau')
	    in
	      violates tau
	    end
      end

    (********
    Printing functions for longid's (for error reporting)
    ********)

    val pr_id = Ident.pr_id
    val pr_longid = Ident.pr_longid
  end;

functor Test(structure SortedFinMap: SORTED_FINMAP
	     structure Ident: IDENT

	     structure Var: VAR
	       sharing type Var.id = Ident.id
		   and type Var.strid = Ident.strid
		   and type Var.longid = Ident.longid

	     structure Con: CON
	       sharing type Con.longid = Ident.longid

	     structure Excon: EXCON
	       sharing type Excon.longid = Ident.longid

	     structure SCon: SCON
	     structure Lab: LAB
	     structure TyName: TYNAME

	     structure TyCon: TYCON
	       sharing type TyCon.tycon = TyName.tycon

	     structure TyVar: TYVAR

	     structure Timestamp: TIMESTAMP
	     structure ListHacks: LIST_HACKS
	     structure Flags: FLAGS
	     structure Report: REPORT

	     structure PP: PRETTYPRINT
	       sharing type PP.Report = Report.Report
		   and type SortedFinMap.StringTree = PP.StringTree

	     structure Crash: CRASH
	    ) : sig end =
  struct
    structure Unconstrained =
      StatObject(structure SortedFinMap = SortedFinMap
		 structure Ident = Ident
		 structure Var = Var
		 structure Con = Con
		 structure Excon = Excon
		 structure Lab = Lab
		 structure SCon = SCon
		 structure TyName = TyName
		 structure TyCon = TyCon
		 structure TyVar = TyVar
		 structure Timestamp = Timestamp
		 structure ListHacks = ListHacks
		 structure Flags = Flags
		 structure Report = Report
		 structure PP = PP
		 structure Crash = Crash
		)

    structure StatObject : STATOBJECT = Unconstrained
    structure StatObjectProp : STATOBJECT_PROP = Unconstrained
  end;
