
(*
$File: Common/ObligationsEnv.sml $
$Date: 1998/07/11 15:15:23 $
$Revision: 1.10 $
$Locker: arkady $
*)
   
(*$ObligationsEnv:
	OBLIGATIONS_ENV STRID SIGID FUNID FINMAP
*)

functor ObligationsEnv (
                         structure StrId: STRID
                         structure SigId: SIGID
                         structure FunId: FUNID
                         structure GrammarInfo: GRAMMAR_INFO
                         structure SourceInfo: SOURCE_INFO
                         sharing type SourceInfo.info = GrammarInfo.SourceInfo
                         structure FinMap: FINMAP
                        )
                             : sig include OBLIGATIONS_ENV
                                   sharing type strid = StrId.strid
                                       and type longstrid = StrId.longstrid
                                       and type funid = FunId.funid
                                       and type sigid = SigId.sigid
                                       and type source_info = GrammarInfo.PostElabGrammarInfo
                                       and type src_id = SourceInfo.info
                               end
                             =
struct
  exception OGEN_IMPOSSIBLE of string

  type source_info = GrammarInfo.PostElabGrammarInfo
  type src_id = SourceInfo.info
  type sigid = SigId.sigid
  type strid = StrId.strid
  type longstrid = StrId.longstrid
  type funid = FunId.funid

  datatype ID = SIG of sigid | FUN of funid | STR of strid | LSTR of longstrid

  datatype Body = SIGBDY of src_id | SIGIDBDY of string | STRUCTBDY of src_id

  fun BodyOfSig i = 
        case GrammarInfo.getPostElabSourceInfo i of
          None => raise OGEN_IMPOSSIBLE ("no source info")
        | Some (si) => SIGBDY (si)

  fun BodyOfSigId sigid = SIGIDBDY (SigId.pr_SigId sigid)
  
  fun BodyOfStruct i =
        case GrammarInfo.getPostElabSourceInfo i of
          None => raise OGEN_IMPOSSIBLE ("no source info")
        | Some (si) => STRUCTBDY (si)


  datatype wrapped_body = WBDY of {body: Body, wrappers: StrInfoEnv}
       and Obligations = OBLIGS of (wrapped_body * wrapped_body) list
       and ObligsEnv = OE of {GIE: SigInfoEnv, SIE: StrInfoEnv, FIE: FunInfoEnv}
       and SigBody = GB of {body: Body, ext_SIE: StrInfoEnv, SIE: StrInfoEnv}
       and SigInfo = GINF of {Spec: SigBody}
       and StrInfo = SINF of {Spec: SigBody}
       and FunInfo = FINF of {Param: StrInfoEnv, Spec: SigBody}
       and SigInfoEnv = GIENV of (sigid, SigInfo) FinMap.map
       and StrInfoEnv = SIENV of (strid, StrInfo) FinMap.map
       and FunInfoEnv = FIENV of (funid, FunInfo) FinMap.map


  fun wrapped (body, SIE) = WBDY {body=body, wrappers=SIE}

  val emptyObligs = OBLIGS []
  fun singleObligs (wb, wb') = OBLIGS [(wb, wb')]

  fun conc_obligs (OBLIGS obligs, OBLIGS obligs') = OBLIGS (obligs @ obligs')
    

  val emptyGIE = GIENV (FinMap.empty)
  val emptySIE = SIENV (FinMap.empty)
  fun isEmptySIE (SIENV m) = FinMap.isEmpty m
  val emptyFIE = FIENV (FinMap.empty)
  val emptyOE = OE {GIE=emptyGIE, SIE=emptySIE, FIE=emptyFIE}


  fun OE_plus_OE (OE {GIE, SIE, FIE}, OE {GIE=GIE', SIE=SIE', FIE=FIE'}) =
        OE {GIE = GIE_plus_GIE (GIE, GIE'),
            SIE = SIE_plus_SIE (SIE, SIE'),
            FIE = FIE_plus_FIE (FIE, FIE')}

  and OE_plus_GIE (OE {GIE, SIE, FIE}, GIE') =
         OE {GIE = GIE_plus_GIE (GIE, GIE'), SIE = SIE, FIE = FIE}

  and OE_plus_SIE (OE {GIE, SIE, FIE}, SIE') =
         OE {GIE = GIE, SIE = SIE_plus_SIE (SIE, SIE'), FIE = FIE}

  and OE_plus_FIE (OE {GIE, SIE, FIE}, FIE') =
         OE {GIE = GIE, SIE = SIE, FIE = FIE_plus_FIE (FIE, FIE')}

  and lookup_sigid (OE {GIE=GIENV m, ...}, sigid) = 
        case FinMap.lookup m sigid of
          None => raise OGEN_IMPOSSIBLE ("lookup_sigid")
        | Some (GI) => GI
  
  and lookup_strid (OE {SIE=SIENV m, ...}, strid) =
        case FinMap.lookup m strid of
          None => raise OGEN_IMPOSSIBLE ("lookup_strid " ^ StrId.pr_StrId strid)
        | Some (SI) => SI

  and lookup_funid (OE {FIE=FIENV m, ...}, funid) =
        case FinMap.lookup m funid of
          None => raise OGEN_IMPOSSIBLE ("lookup_funid")
        | Some (FI) => FI
  
  and lookup_longstrid (OE {SIE, ...}, longstrid) =
    let
      val (strid_list, strid) = StrId.explode_longstrid longstrid
      fun traverse SIE [] = SIE
        | traverse SIE (strid::rest) =
           let
             val SIENV m = SIE
             val Spec = case FinMap.lookup m strid of
                          None => raise OGEN_IMPOSSIBLE ("traverse in lookup_longstrid")
                        | Some (SINF{Spec}) => Spec
             val GB {SIE=SIE', ...} = Spec
           in
             traverse SIE' rest
           end
      val SIENV m = traverse SIE strid_list
    in
      case FinMap.lookup m strid of
        None => raise OGEN_IMPOSSIBLE ("lookup_longstrid")
      | Some (SI) => SI
    end


  and mkGB (body, ext_SIE, SIE) = GB {body=body, ext_SIE=ext_SIE, SIE=SIE}
  and unmkGB (GB {body, ext_SIE, SIE}) = (body, ext_SIE, SIE)

  and domGIE (GIENV m) = FinMap.dom m
  and singleGIE (sigid, gb) = 
          GIENV (FinMap.singleton (sigid, GINF {Spec=gb}))
   
  and GIE_plus_GIE (GIENV m, GIENV m') = 
          GIENV (FinMap.plus (m, m'))

  and domSIE (SIENV m) = FinMap.dom m
  and singleSIE (strid, gb) = 
          SIENV (FinMap.singleton (strid, SINF {Spec=gb}))
   
  and SIE_plus_SIE (SIENV m, SIENV m') = 
          SIENV (FinMap.plus (m, m'))

  and cut_domain (SIENV orig_m, dom) =
        let
          fun cut_dom dom (strid, sinf) = EqSet.member strid dom
        in 
          SIENV (FinMap.filter (cut_dom dom) orig_m)
        end

  and domFIE (FIENV m) = FinMap.dom m
  and singleFIE (funid, SIE, gb) = 
          FIENV (FinMap.singleton (funid, FINF {Param=SIE, Spec=gb}))
   
  and FIE_plus_FIE (FIENV m, FIENV m') = 
          FIENV (FinMap.plus (m, m'))

  fun Spec_of_OE (oe, SIG sigid) =
         let
           val GINF {Spec} = lookup_sigid (oe, sigid)
         in
           Spec
         end
    | Spec_of_OE (oe, FUN funid) =
         let
           val FINF {Param, Spec} = lookup_funid (oe, funid)
         in
           Spec
         end
    | Spec_of_OE (oe, STR strid) =
         let
           val SINF {Spec} = lookup_strid (oe, strid)
         in
           Spec
         end
    | Spec_of_OE (oe, LSTR lstrid) =
         let
           val SINF {Spec} = lookup_longstrid (oe, lstrid)
         in
           Spec
         end
      
  fun Param_of_OE (oe, funid) =
         let
           val FINF {Param, Spec} = lookup_funid (oe, funid)
         in
           Param
         end

  fun SIE_of_OE (oe, id) =
         let
           val GB {body, ext_SIE, SIE} = Spec_of_OE (oe, id)
         in
           SIE
         end


  fun generate_obligs (str_SIE as SIENV str_m, sig_SIE as SIENV sig_m) = 
                                      (*impl |= spec*)
       let
         val common_ids = EqSet.intersect (domSIE str_SIE) (domSIE sig_SIE)
         fun BodyAndExtStrAndSIE m strid =
               let
                 val Some (SINF {Spec=GB{body=b, ext_SIE=e, SIE=s}}) = 
                                                      FinMap.lookup m strid
               in
                 (*body, ext_SIE, SIE*)
                 (b, e, s)
               end
         fun gen [] _ _ = emptyObligs
           | gen (strid::rest) str_m sig_m =
              let
                val (str_body, str_ext_SIE, str_SIE) =
                                            BodyAndExtStrAndSIE str_m strid
                val (sig_body, sig_ext_SIE, sig_SIE) =
                                            BodyAndExtStrAndSIE sig_m strid
                val oblig = singleObligs (wrapped (str_body, str_ext_SIE),
                                          wrapped (sig_body, sig_ext_SIE))
              in
                conc_obligs (oblig,
                             conc_obligs (generate_obligs (str_SIE, sig_SIE),
                                          gen rest str_m sig_m
                                         )
                            )
              end
       in
         gen (EqSet.list common_ids) str_m sig_m
       end

  
  fun cutSigStruct n sp s =
       let
         val s' = String.extract n (String.size s) s
         val s' = if (String.skipSpaces (String.dropR "\n" s')) = "" then "\n"
                  else sp ^ s'
       in
         if (String.nth (String.size s' - 1) s') = "\n" then
           s'
         else
           (s' ^ "\n")
       end

  fun cutEnd s =
       let
         val s' = String.extract 0 (String.size s - 4) s
         val s' = if s' = "" then "\n" else s'
       in
         if (String.nth (String.size s' - 1) s') = "\n" then
           s'
         else
           (s' ^ "\n")
       end

  fun cut isSig [] = []
    | cut isSig s = 
     let
       fun first (s::rest) = (s, rest)
       fun last [s] = ([], s)
         | last (s::rest) =
            let
              val (ss, l) = last rest
            in
              (s::ss, l)
            end

       val (sp, len) = if isSig then ("   ", 3) else ("      ", 6)
       val (f, rest) = first s
       val f' = cutSigStruct len sp f
       val rest = if (String.skipSpaces (String.dropR "\n" f')) = "" then rest
                  else f'::rest
       val (prev, l) = last (rest)
       val l' = cutEnd l
     in
       if (String.skipSpaces (String.dropR "\n" l')) = "" then prev
       else prev @ [l']
     end


  fun output_cut_body intend print body =
      let
        fun print' intd s = (print intd; print s)
      in
        case body of
          SIGBDY (src) => (map (print' intend) (cut true (SourceInfo.sourceFrom src)); ())
        | SIGIDBDY (s) => print' intend (s ^ "\n")
        | STRUCTBDY (src) => (map (print' intend) (cut false (SourceInfo.sourceFrom src)); ())
      end

  fun output_body intend print body =
      let
        fun print' intd s = (print intd; print s)
      in
        case body of
          SIGBDY (src) => (map (print' intend) (SourceInfo.sourceFrom src); ())
        | SIGIDBDY (s) => print' intend (s ^ "\n")
        | STRUCTBDY (src) => (map (print' intend) (SourceInfo.sourceFrom src); ())
      end

  fun output_signature print (sigid, GINF {Spec = GB {body, ...}}) =
       let
         val _ = print "***SIGNATURE\n"
         val _ = print ((SigId.pr_SigId sigid) ^ "\n")   (*one line*)
         val _ = output_body "" print body
         val _ = print "***END_OF\n"
       in
         ()
       end

  fun output_SigInfoEnv print (GIENV m) =
       let
         val _ = FinMap.ComposeMap (output_signature print) m
       in
         ()
       end


  fun output_strdec intend print withQMARK (strid, SINF {Spec=GB{body, ext_SIE, ...}}) =
     if isEmptySIE ext_SIE then
       let
         val _ = print (intend ^ "structure " ^ (StrId.pr_StrId strid) ^ " :\n")
         val _ = output_body (intend ^ "  ") print body
         val _ = if withQMARK then print (intend ^ "    = ?\n") else ()
       in
         ()
       end
     else
       let
         val _ = print (intend ^ "local\n")
         val _ = output_strdecs (intend ^ "  ") print withQMARK ext_SIE
         val _ = print (intend ^ "in\n")
         val _ = print (intend ^ "  structure " ^ (StrId.pr_StrId strid) ^ " :\n")
         val _ = output_body (intend ^ "    ") print body
         val _ = if withQMARK then print (intend ^ "      = ?\n") else ()
         val _ = print (intend ^ "end\n")
       in
         ()
       end

  and output_strdecs intend print withQMARK (SIENV m) =
       let
         val _ = FinMap.ComposeMap (output_strdec intend print withQMARK) m
       in
         ()
       end

  fun output_wrapped_body print (WBDY{body, wrappers as (SIENV m)}) =
    case (body, FinMap.isEmpty m) of
      (SIGIDBDY (txt), _) =>
        print (txt ^ "\n")
    | (SIGBDY (_), true) =>
        output_body "" print body
    | (SIGBDY (_), false) =>
        let
          val _ = print "sig\n"
          val _ = print "  local\n"
          val _ = output_strdecs "    " print false wrappers
          val _ = print "  in\n"
          val _ = output_cut_body "    " print body
          val _ = print "  end\n"
          val _ = print "end\n"
        in
          ()
        end
    | (STRUCTBDY (_), true) =>
        output_body "" print body
    | (STRUCTBDY (_), false) =>
        let
          val _ = print "struct\n"
          val _ = print "  local\n"
          val _ = output_strdecs "    " print true wrappers
          val _ = print "  in\n"
          val _ = output_cut_body "    " print body
          val _ = print "  end\n"
          val _ = print "end\n"
        in
          ()
        end

  fun output_obligation print (wb, wb') =
       let
         val _ = print "***OBLIG\n"
         val _ = print "***LEFT\n"
         val _ = output_wrapped_body print wb
         val _ = print "***RIGHT\n"
         val _ = output_wrapped_body print wb'
         val _ = print "***OBLIG_END\n"
       in
         ()
       end
  
  fun output_Obligations print (OBLIGS obligs) = 
       let
         val _ = map (output_obligation print) obligs
       in
         ()
       end


  datatype TopObligations = TOPOBLIGS of (string * string * Obligations) list

  val emptyTopObligs = TOPOBLIGS nil

  fun singleTopObligs (kind, name, obligs) = TOPOBLIGS [(kind, name, obligs)]

  fun conc_topobligs (TOPOBLIGS l, TOPOBLIGS l') = TOPOBLIGS (l@l')

  fun additional_obligs (obligs, TOPOBLIGS l) =
        TOPOBLIGS (map (fn (k, n, obs) => (k, n, conc_obligs (obligs, obs))) l)

  fun output_TopOblig print (kind, name, obligs) =
       let
         val _ = print "***TOPDECLARATION\n"
         val _ = print (kind ^ "\n")
         val _ = print (name ^ "\n")
         val _ = output_Obligations print obligs
       in
         ()
       end

  fun output_TopObligations print (TOPOBLIGS l) =
       (map (fn e => output_TopOblig print e) l;
        ())

end;
