(* The interpreter *)

(*
$File: Common/Interpreter.sml $
$Date: 1992/04/07 14:56:13 $
$Revision: 1.35 $
$Locker: birkedal $
*)

(*$Interpreter:
	BASIS PARSE ELABTOPDEC PPTOPDECGRAMMAR
	EVALTOPDEC ERROR_TRAVERSE INFIX_BASIS TOP_LEVEL_REPORT BASIC_IO
	REPORT PRETTYPRINT FLAGS CRASH INTERPRETER
 *)

functor Interpreter(structure Basis: BASIS

		    structure Parse: PARSE
		      sharing type Parse.InfixBasis = Basis.InfixBasis

		    structure ElabTopdec: ELABTOPDEC
		      sharing type ElabTopdec.PreElabTopdec = Parse.topdec
			  and type ElabTopdec.StaticBasis = Basis.StaticBasis

		    structure PPPreElabTopdecGrammar: PPTOPDECGRAMMAR
		      sharing type PPPreElabTopdecGrammar.G.topdec
				   = ElabTopdec.PreElabTopdec

		    structure PPPostElabTopdecGrammar: PPTOPDECGRAMMAR
		      sharing type PPPostElabTopdecGrammar.G.topdec
				   = ElabTopdec.PostElabTopdec

		    structure EvalTopdec: EVALTOPDEC
		      sharing type EvalTopdec.topdec = ElabTopdec.PostElabTopdec
			  and type EvalTopdec.DynamicBasis = Basis.DynamicBasis

(* arkady *)
                    structure ObligTopdec: OBLIGTOPDEC
                      sharing type ObligTopdec.topdec = EvalTopdec.topdec
(* end of arkady *)

		    structure ErrorTraverse: ERROR_TRAVERSE
		      sharing type ErrorTraverse.topdec
				   = ElabTopdec.PostElabTopdec

		    structure InfixBasis: INFIX_BASIS
		      sharing type InfixBasis.Basis = Basis.InfixBasis

		    structure TopLevelReport: TOP_LEVEL_REPORT
		      sharing type TopLevelReport.Basis = Basis.Basis

		    structure BasicIO: BASIC_IO

		    structure Report: REPORT
		      sharing type InfixBasis.Report
				   = Parse.Report
				   = ErrorTraverse.Report
				   = TopLevelReport.Report
				   = Report.Report

		    structure PP: PRETTYPRINT
		      sharing type PP.Report = Report.Report
			  and type InfixBasis.StringTree
			           = PPPreElabTopdecGrammar.StringTree
				   = ElabTopdec.StringTree
				   = EvalTopdec.StringTree
				   = PP.StringTree

		    structure Flags: FLAGS
		    structure Crash: CRASH
		   ): INTERPRETER =
  struct
    datatype Mode = PARSE_ONLY | ELABORATE_ONLY | EVALUATE
(* arkady *)
                  | OBLIGATE of bool
(* end of arkady *)
    type Report = Report.Report
    type Basis = Basis.Basis

    infix //
    val op // = Report.//

    type Pack = EvalTopdec.Pack

(* arkady *)
    structure ObligsEnv = ObligTopdec.ObligationsEnv
    
    val OE = ref ObligsEnv.emptyOE
(* end of arkady *)

    datatype Result =
        SUCCESS of {report: Report, basis: Basis, state: Parse.State}
      | SYNTAX_ERROR of Report
      | SEMANTIC_ERRORS of Report
      | RUNTIME_ERROR of Pack
      | LEGAL_EOF		(* Legal end-of-file. *)

   (* Interpret a topdec. This is a one-shot - we have to iterate in
      order to digest a file of declarations. *)

    fun interpret(oldBasis, control, parsed: Parse.Result): Result =
      let
(* arkady *)
	val (type_it, run_it, obligate_it, asTops) =
	  case control
	    of PARSE_ONLY => (false, false, false, false)
	     | ELABORATE_ONLY => (true, false, false, false)
	     | EVALUATE => (true, true, false, false)
	     | OBLIGATE flag => (true, false, true, flag)
(* end of arkady *)

	fun dealWithTyped(oldBasis, topdec, iBas, sBas, state): Result =
	  let
(* arkady *)
            fun print_line s = Outstream.output (Outstream.std_out, s)
(* end of arkady *)
	    val debugElab =
	      if Flags.DEBUG_ELABTOPDEC
	      then PP.reportStringTree(ElabTopdec.layoutStaticBasis sBas)
	      else Report.null

	    val elabBasis =
	      Basis.B_plus_B(Basis.Inf_in_B iBas, Basis.Stat_in_B sBas)
	  in
(* arkady *)
            if obligate_it then
              let
                val (OE', obligs, OBLIGS) =
                  (
                    if asTops then
                    let
                      val (OE', OBLIGS) =
                         ObligTopdec.topoblig_topdec print_line (!OE, topdec)
                    in
                      (OE', ObligsEnv.emptyObligs, OBLIGS)
                    end
                    else
                    let
                      val (OE', obligs) =
                         ObligTopdec.oblig_topdec (!OE, topdec)
                    in
                      (OE', obligs, ObligsEnv.emptyTopObligs)
                    end
                  )
                  handle ObligsEnv.OGEN_IMPOSSIBLE txt =>
                  (print_line ("\n\nOGEN_IMPOSSIBLE: " ^ txt ^ "\n\n");
                   raise ObligsEnv.OGEN_IMPOSSIBLE txt)
              in
                (
                  OE := OE';
                  if asTops then
                    ObligsEnv.output_TopObligations print_line OBLIGS
                  else
                    ObligsEnv.output_Obligations print_line obligs;
                  Outstream.flush Outstream.std_out;
                  SUCCESS {
                            report = Report.null,
                            basis = elabBasis,
                            state = state
                          }
                )
              end
            else
(* end of arkady *)
	    if run_it then
	      let
		val dBas = EvalTopdec.eval(Basis.Dyn_of_B oldBasis, topdec)

		val debugEval =
		  if Flags.DEBUG_EVALTOPDEC
		  then PP.reportStringTree(EvalTopdec.layoutDynamicBasis dBas)
		  else Report.null

		val evalBasis =
		  Basis.B_plus_B(elabBasis, Basis.Dyn_in_B dBas)

		val fullReport =
		  TopLevelReport.report{basis=evalBasis, bindings=true}
	      in
		SUCCESS{report=debugElab // debugEval // fullReport,
			basis=evalBasis,
			state=state
		       }
	      end
	      handle EvalTopdec.UNCAUGHT pack => RUNTIME_ERROR pack
	    else
	      SUCCESS{report=debugElab
		             // TopLevelReport.report{basis=elabBasis,
						      bindings=false
						     },
		      basis=elabBasis,
		      state=state
		     }
	  end

	fun dealWithParsed(oldBasis, topdec, iBas, state): Result =
	  let
	    val debugParse =
	      if Flags.DEBUG_PARSING then
	        PP.reportStringTree(PPPreElabTopdecGrammar.layoutTopdec topdec)
		// PP.reportStringTree(InfixBasis.layoutBasis iBas)
	      else
		Report.null

	    val parseBasis = Basis.Inf_in_B iBas
	  in
	    if type_it then
	      let
		val (sBas, topdec') =
		  ElabTopdec.elab_topdec(Basis.Stat_of_B oldBasis, topdec)
	      in
		case ErrorTraverse.traverse topdec'
		  of ErrorTraverse.SUCCESS =>
		       (case dealWithTyped(oldBasis, topdec', iBas, sBas, state)
			  of SUCCESS{report, basis=basis', ...} =>
			       SUCCESS{report=debugParse // report,
				       basis=basis',
				       state=state
				      }

			   | other => other
		       )

		   | ErrorTraverse.FAILURE errors =>
		       SEMANTIC_ERRORS(debugParse // errors)
	      end
	    else
	      SUCCESS{report=Report.line "[Parse only]"
		      	     // TopLevelReport.report{basis=parseBasis,
						      bindings=false
						     }
					(* That'll just report the infixes;
					   the other envs will be empty. *)
			     // debugParse,
		      basis=parseBasis,
		      state=state
		     }
	  end
      in
	case parsed
	  of Parse.SUCCESS(iBas, topdec, state) =>
	       dealWithParsed(oldBasis, topdec, iBas, state)

	   | Parse.ERROR error => SYNTAX_ERROR error
	   | Parse.LEGAL_EOF => LEGAL_EOF
      end (* interpret *)


    val B = ref Basis.initialB

    (* `loop' is the main top-level loop. It takes a mode (parse-only,
       parse-and-typecheck, parse-typecheck-run), a lexer state, and
       two failure functions for dealing with compile-time and runtime
       errors. These might cause full failure propagation, or re-entry
       of `loop'. (see interpretWith.) Oh: `loop' also deals with internal
       consistency errors (Crash.CRASH) - we just do a fail_COMPILE which
       isn't quite what we want to do really but is close enough. *)

    fun loop(name: string,
	     mode,
	     state: Parse.State,
	     fail_COMPILE: Report -> unit,
	     fail_RUN: Pack -> unit
	    ) =
      (case interpret(!B, mode, Parse.parse(Basis.Inf_of_B(!B), state))
	 of SUCCESS{report, basis=basis', state=state'} =>
	      (Report.print report;
	       B := Basis.B_plus_B(!B, basis');
	       loop(name, mode, state', fail_COMPILE, fail_RUN)
	      )

	  | SYNTAX_ERROR report => fail_COMPILE report
	  | SEMANTIC_ERRORS report => fail_COMPILE report
	  | RUNTIME_ERROR pack => fail_RUN pack
	  | LEGAL_EOF => BasicIO.println("[end of " ^ name ^ "]")
      ) handle Crash.CRASH =>
	  fail_COMPILE(Report.line "System Crash: Reentering...")

    fun interpretWith(sourceFn, failOnError) =
      fn mode => fn arg =>
	let
	  val source = sourceFn arg	(* the SourceReader. *)

	  val name = Parse.nameOf source

	  fun fail_COMPILE report =
	    (Report.print report;
	     if failOnError then
	       EvalTopdec.FAIL_USE()	(* Fail this nested `use' context
					   because of compile-time error. *)
	     else
	       loop(name, mode, Parse.begin source, fail_COMPILE, fail_RUN)
					(* create new lexing fn. to discard
					   rest of line, and start again. *)
	    )

	  and fail_RUN pack =
	    if failOnError then
	      EvalTopdec.RE_RAISE pack	(* Propagate to calling context *)
	    else
	      (Report.print(Report.line("Failure: " ^ EvalTopdec.pr_Pack pack));
	       loop(name, mode, Parse.begin source, fail_COMPILE, fail_RUN)
					(* Print report, discard rest of line,
					   start again. *)
	      )
	in
	  loop(name, mode, Parse.begin source, fail_COMPILE, fail_RUN)
	end

    val interpretStdIn = interpretWith(Parse.sourceFromStdIn, false)
    val interpretFile = interpretWith(Parse.sourceFromFile, true)
    val interpretString = interpretWith(Parse.sourceFromString, true)
  end;
