(* Elaborator for Core Language Declarations*)

(*
$File: Common/ElabDec.sml $
$Date: 1993/03/05 14:38:18 $
$Revision: 1.69 $
$Locker: birkedal $
*)

(* mikon#1.28 *)        
(*$ElabDec:
	DEC_GRAMMAR ENVIRONMENTS STATOBJECT CON LAB VAR ELABDEC SOURCE_INFO
	ERROR_INFO TYPE_INFO OVERLOADING_INFO GRAMMAR_INFO IDENT
	PPDECGRAMMAR REPORT PRETTYPRINT	FLAGS CRASH
        CORE_TRACE
 *)
(* end mikon#1.28 *)

functor ElabDec(structure IG: DEC_GRAMMAR

		structure OG: DEC_GRAMMAR
		  sharing type IG.lab       = OG.lab
		      and type IG.con       = OG.con
		      and type IG.scon      = OG.scon
		      and type IG.tyvar     = OG.tyvar
		      and type IG.tycon     = OG.tycon
		      and type IG.longtycon = OG.longtycon
		      and type IG.longstrid = OG.longstrid
		      and type IG.excon     = OG.excon

	       (* resolved identifiers *)
		structure R:
		  sig
		    eqtype longvar
		       and longcon
		       and longexcon

		    datatype longid = LONGVAR of longvar
				    | LONGCON of longcon
				    | LONGEXCON of longexcon
		  end
		  sharing type R.longid = OG.longid

		structure Environments: ENVIRONMENTS
		  sharing type Environments.longid    = IG.longid
		      and type Environments.longtycon = IG.longtycon
		      and type Environments.longstrid = IG.longstrid
		      and type Environments.tycon     = IG.tycon
		      and type Environments.valbind   = IG.valbind
(* mikon#1.20 *)
                      and type Environments.match     = IG.match
                      and type Environments.exp       = IG.exp
(* end mikon#1.20 *)
 		      and type Environments.excon     = IG.excon
		      and type Environments.con       = IG.con
		      and type Environments.id        = IG.id
                      and type Environments.pat       = IG.pat

		structure StatObject: STATOBJECT
		  sharing type StatObject.TypeScheme   = Environments.TypeScheme
		      and type StatObject.SyntaxTyVar  = IG.tyvar
		      and type StatObject.longid       = IG.longid
		      and type StatObject.longcon      = R.longcon
		      and type StatObject.longvar      = R.longvar
		      and type StatObject.longexcon    = R.longexcon
		      and type StatObject.TypeFcn      = Environments.TypeFcn
		      and type StatObject.Substitution
			       = Environments.Substitution
		      and type StatObject.Type  = Environments.Type
		      and type StatObject.TyVar = Environments.TyVar
		      and type StatObject.TyName = Environments.TyName
		      and type StatObject.tycon = IG.tycon
		      and type StatObject.scon  = IG.scon
		      and type StatObject.lab   = IG.lab
		      and type StatObject.id    = IG.id

	       (* Unqualified identifiers are always converted into vars *)
		  sharing type StatObject.var = OG.id

	       (* Con needed for compiler info. *)
		structure Con: CON
		  sharing type Con.longid = IG.longid
		      and type StatObject.longcon = Con.longcon
		      and type Environments.con = Con.con

	       (* Lab needed for the equality attribute. *)
		structure Lab: LAB
		  sharing type OG.lab = Lab.lab
		      and type IG.lab = Lab.lab

	       (* We need Var to map vars (PostElabDecGrammar.id) into ids
		  (PreElabDecGrammar.id) for the RECvalbind analysis. *)

		structure Var: VAR
		  sharing type IG.id = Var.id
		      and type OG.id = Var.var

		structure SourceInfo: SOURCE_INFO

		structure ErrorInfo: ERROR_INFO
		  sharing type ErrorInfo.Type  = StatObject.Type
		      and type ErrorInfo.TyVar = StatObject.TyVar
(* mju#1.6 *)
		      and type ErrorInfo.TypeFcn = StatObject.TypeFcn
(* endmju#1.6 *)
		      and type ErrorInfo.id    = Environments.id
		      and type ErrorInfo.longid = StatObject.longid
		      and type ErrorInfo.longstrid = Environments.longstrid
		      and type ErrorInfo.con   = IG.con
		      and type ErrorInfo.excon = IG.excon
		      and type ErrorInfo.tycon = IG.tycon
		      and type ErrorInfo.lab   = IG.lab
		      and type ErrorInfo.longtycon = Environments.longtycon

		structure TypeInfo: TYPE_INFO
		  sharing type TypeInfo.lab     = IG.lab
		      and type TypeInfo.longcon = R.longcon

		structure OverloadingInfo: OVERLOADING_INFO
		  sharing type OverloadingInfo.Type = StatObject.Type

(* mikon#1.28 *)
		structure CoreTrace : CORE_TRACE
		  sharing type CoreTrace.Type    = Environments.Type
		      and type CoreTrace.Env     = Environments.Env
		      and type CoreTrace.Context = Environments.Context
		      and type CoreTrace.TyEnv   = Environments.TyEnv
		      and type CoreTrace.VarEnv  = Environments.VarEnv
		      and type CoreTrace.TyRea   = Environments.tyrea
		      and type CoreTrace.TyVar   = Environments.TyVar
		      and type CoreTrace.TyName   = Environments.TyName
	              and type CoreTrace.Substitution = Environments.Substitution

		structure Trace : sig
		                      type Trace and TraceCOR
				      val TraceCOR_in_Trace : TraceCOR -> Trace
                                      val unTraceCoreTrace : Trace -> TraceCOR Option	
				  end
		  sharing type Trace.TraceCOR = CoreTrace.Trace
(* end mikon#1.28 *)

		structure GrammarInfo: GRAMMAR_INFO
		  sharing type GrammarInfo.ErrorInfo = ErrorInfo.info
		      and type GrammarInfo.TypeInfo = TypeInfo.info
		      and type GrammarInfo.OverloadingInfo = OverloadingInfo.info
		      and type GrammarInfo.PreElabGrammarInfo = IG.info
		      and type GrammarInfo.PostElabGrammarInfo = OG.info
(* mikon#1.28 *)
		      and type GrammarInfo.Trace = Trace.Trace
(* end mikon#1.28 *)

		structure PPInDecGrammar: PPDECGRAMMAR
		  sharing PPInDecGrammar.G = IG

	        structure Report: REPORT

		structure PP: PRETTYPRINT
		  sharing type StatObject.StringTree
			       = Environments.StringTree
			       = PPInDecGrammar.StringTree
			       = PP.StringTree
		      and type PP.Report = Report.Report

		structure Flags: FLAGS
 		structure Crash: CRASH
	       ) : ELABDEC =
  struct

(* mikon#1.12 *)
    val TypeBool = StatObject.TypeBool
(* end mikon#1.12 *)

    type PreElabGrammarInfo  = GrammarInfo.PreElabGrammarInfo
     and PostElabGrammarInfo = GrammarInfo.PostElabGrammarInfo

    val okConv = GrammarInfo.convertGrammarInfo
				(* Convert pre-elab. info into post-elab, with
				   empty Error, Type, and Trace fields *)

    fun errorConv(i: PreElabGrammarInfo, e: ErrorInfo.info)
          : PostElabGrammarInfo =
      GrammarInfo.addPostElabErrorInfo (okConv i) e
				(* Convert pre-elab. to post-elab. adding
				   an error tag at the same time. The assumption
				   (forced by this type conversion) is that
				   there can only be one error tag per info. *)

    fun preOverloadingConv(i: PreElabGrammarInfo, oi: OverloadingInfo.info)
	  : PostElabGrammarInfo =
      GrammarInfo.addPostElabOverloadingInfo (okConv i) oi
                                (* Convert pre-elab. info to post-elab, adding
				   pre-resolvation overloading info at the
				   same time *)
   
    fun postOverloadingConv(i: PostElabGrammarInfo, oi: OverloadingInfo.info)
          : PostElabGrammarInfo =
      GrammarInfo.addPostElabOverloadingInfo i oi
                                (* Add post-resolvation overloading info *)

(* mikon#1.28 *)

    fun addCoreTrace (tr : CoreTrace.Trace) (i : PostElabGrammarInfo) =
	GrammarInfo.addPostElabTrace i (Trace.TraceCOR_in_Trace tr)
				(* Add CoreTrace.Trace injected into the Trace *)

    fun addScheme_and_Sim (scheme : CoreTrace.SchemeTrace, sim : CoreTrace.SimTrace) 
                          (i : PostElabGrammarInfo) =
	addCoreTrace (CoreTrace.TRACE(Some scheme, Some sim)) i
 
    fun addSimTrace (sim : CoreTrace.SimTrace) (i : PostElabGrammarInfo) =
	addCoreTrace (CoreTrace.TRACE(None, Some sim)) i

    fun addSchemeTrace (scheme : CoreTrace.SchemeTrace) (i : PostElabGrammarInfo) =
	addCoreTrace (CoreTrace.TRACE(Some scheme, None)) i

(* end mikon#1.28 *)

(* mikon#1.38 *)

    fun changeSchemeTrace (scheme : CoreTrace.SchemeTrace) (i : PostElabGrammarInfo) = 
	case (GrammarInfo.getPostElabTrace i)
	  of (Some(trace)) =>
	      (case (Trace.unTraceCoreTrace trace)
		 of (Some(CoreTrace.TRACE(scheme_opt, sim_opt))) => 
		     addCoreTrace (CoreTrace.TRACE(Some scheme, sim_opt)) i
		  | (None) => Crash.impossible "ElabDec.changeSchemeTrace(unTrace)")
	   | (None) => addSchemeTrace scheme i
	      
(* end mikon#1.38 *)

    fun lookupIdError(i: PreElabGrammarInfo, longid: StatObject.longid)
          : PostElabGrammarInfo =
      errorConv(i, ErrorInfo.LOOKUP_LONGID longid)

    fun lookupTyConError(i: PreElabGrammarInfo,
			 longtycon: Environments.longtycon
			): PostElabGrammarInfo =
      errorConv(i, ErrorInfo.LOOKUP_LONGTYCON longtycon)

    fun repeatedIdsError(i: PreElabGrammarInfo,
			 rids: ErrorInfo.RepeatedId list): PostElabGrammarInfo =
      errorConv(i, ErrorInfo.REPEATED_IDS rids)

    
    fun overloadingError(i: PostElabGrammarInfo): PostElabGrammarInfo =
      GrammarInfo.addPostElabErrorInfo i ErrorInfo.NOTRESOLVED


    infixr on
    val op on = StatObject.on

    infixr onC
    val op onC = Environments.onC

    infixr onVE
    val op onVE = Environments.onVE

    infixr oo
    val op oo = StatObject.oo

    infixr C_cplus_E
    val op C_cplus_E = Environments.C_cplus_E

    infixr C_cplus_TE
    val op C_cplus_TE = Environments.C_cplus_TE

    infixr C_cplus_VE_and_TE
    val op C_cplus_VE_and_TE = Environments.C_cplus_VE_and_TE

    type PreElabDec = IG.dec and PostElabDec = OG.dec
    type PreElabTy  = IG.ty  and PostElabTy  = OG.ty
(* mikon#4 *)
    type PreElabExp = IG.exp and PostElabExp = OG.exp

    type Type    = StatObject.Type
    type Env     = Environments.Env
    type Context = Environments.Context
    type Substitution = StatObject.Substitution

    fun pr(msg: string, t: PP.StringTree): unit =
      Report.print(Report.decorate(msg, PP.reportStringTree t))

    fun getRepeatedElements ls =
      let
	fun NoOfOccurences x [] = 0
	  | NoOfOccurences x (y::ys) = 
	    if x = y then 1 + NoOfOccurences x ys
	    else NoOfOccurences x ys
      in
	List.all (fn e => (NoOfOccurences e ls) > 1) ls
      end

    fun where list elem =
      case List.index (General.curry (op =) elem) list
	of OK n => n
	 | Fail _ => Crash.impossible "ElabDec.where"

    (* Hooks needed by the compiler:

	  o for a RECORDatpat, a type-info node of type LAB_INFO in each
	    PATROW.

	  o for a CONSpat or constructor LONGIDatpat or constructor IDENTatexp,
	    a type-info node of type CON_INFO.

	  o for a exception constructor IDENTatexp, a type-info node of
	    type EXCON_INFO.
     *)

    local
      open GrammarInfo TypeInfo
    in
     (* MEMO: no duplication checks here (or anywhere else!) *)
      fun addTypeInfo_CON(postElabInfo, C, isArrow, id) =
	let
	  val con = #2(Con.decompose(StatObject.mk_longcon id))
	  val cons = Environments.lookupFellowCons(C, id)
	in
	  addPostElabTypeInfo postElabInfo
	    (CON_INFO{numCons=List.size cons,
		      index=where cons con,
		      functional=isArrow
		     }
	    )
	end

      fun addTypeInfo_EXCON(postElabInfo, tau) =
	(* The excon carries a value if the type tau is functional,
	   and doesn't if it's `exn'. *)
	addPostElabTypeInfo postElabInfo
			    (EXCON_INFO{functional=StatObject.isTypeArrow tau})

      fun addTypeInfo_LAB(postElabInfo, index) =
	addPostElabTypeInfo postElabInfo (LAB_INFO{index=index})
    end

    (********
    Get a fresh type variable
    *********
    Returns a type variable with the imperative, equality and overloading 
    attributes set to be false.
    ********)

    fun freshTyVar() =
      StatObject.freshTyVar{equality=false, imperative=false, overloaded=false}

    fun freshType() = StatObject.mkTypeTyVar(freshTyVar())

   (* When we get elaboration errors we often need to return a bogus type.
      It's best if that's just 'a. *)

    val bogus_Type = freshType

    fun Unify(tau, tau', i): Substitution * PostElabGrammarInfo =
      case StatObject.unify(tau, tau')
	of Some subst => (subst, okConv i)
         | None =>
	     (StatObject.bogus_Subst,
	      errorConv(i, ErrorInfo.UNIFICATION(tau, tau'))
	     )

   (* Traversal of patterns to build a VE. We only do this for `rec'
      bindings. Arguably, we should reject records and so on right here,
      but the caller isn't set up to deal with that (and ignoring them
      would give unbound var errors for actual occuring vars), so we work for
      all patterns. We reject illegal pattern forms in the `rec' post-pass. *)

   (*MEMO: these should all be returning sets. *)

    (********
    Find the variables bound in a valbind
    *********
    There is no checking for multiply declared variables
    ********)

    fun dom_vb(C: Environments.Context, vb: IG.valbind): Environments.id list =
      case vb
	of IG.PLAINvalbind(_, pat, _, vb_opt) =>
	     Environments.dom_pat(C, pat) @ (case vb_opt
				  of Some vb => dom_vb(C, vb)
				   | None => nil
			       )

         | IG.RECvalbind(_, vb) =>
	     dom_vb(C, vb)

    (*********
    Find the TE to be used initially in elaborating a datbind
    **********
    We determine the correct equality attributes when we maximise equality
    *********)

    local
      fun make_TE tyvar_list tycon =
      let
	val arity =
	  List.size tyvar_list

	val tyname =
	  StatObject.freshTyName {name = tycon, arity = arity, equality = false}

	val typeFcn =
	  StatObject.TyName_in_TypeFcn tyname

	val tystr =
	  Environments.mkTyStr(typeFcn, Environments.emptyCE)
      in
	Environments.singleTE(tycon, tystr)
      end
    in
      fun initial_TE (IG.DATBIND(_, tyvar_list, tycon, _, None)) =
	  make_TE tyvar_list tycon
	| initial_TE (IG.DATBIND(_, tyvar_list, tycon, _, Some datbind)) =
	  Environments.TE_plus_TE(make_TE tyvar_list tycon, initial_TE datbind)
    end

   (* addLabelInfo: given a RecType and a PATROW, populate the info fields
      with LAB_INFO specifying which label index (numbered canonically from
      0) is associated with each label. This is needed by the compiler. *)

    fun addLabelInfo(recType, patrow) =
      let
	val sortedLabs = StatObject.sortedLabsOfRecType recType

	fun f(OG.PATROW(i, lab, pat, patrow_opt)) =
	  (case GrammarInfo.getPostElabErrorInfo i of
	    None => 
	      let
		val index = where sortedLabs lab
	      in
		OG.PATROW(addTypeInfo_LAB(i, index),
			  lab, pat,
			  case patrow_opt
			    of Some patrow => Some(f patrow)
			     | None => None
			 )
	      end
	   | Some _ => OG.PATROW(i, lab, pat, patrow_opt))

	  | f(OG.DOTDOTDOT i) = OG.DOTDOTDOT i
      in
	f patrow
      end

    (********************************************************)
    (*      Elaboration (type checking)                     *)
    (********************************************************)


    (****** atomic expressions - Definition page 23 ******)

    fun elab_atexp (C : Environments.Context, atexp : IG.atexp) :
	(Substitution * StatObject.Type * OG.atexp) =

	case atexp of

	  (* special constants *)                               (* rule 1 *)
	  IG.SCONatexp(i, scon) =>
	    (StatObject.Id, StatObject.GetTypescon scon, OG.SCONatexp(okConv i,scon))

	  (* identifiers - variables or constructors *)     
	| IG.IDENTatexp(i, IG.OP_OPT(ident, withOp)) =>
	    (case Environments.Lookup_longid(C,ident) of

	       (* Variable *)                                   (* rule 2 *)
	      Some(Environments.LONGVAR sigma) =>
                 let 
                   val instance = StatObject.instance sigma
                 in  
	           (StatObject.Id, instance,
		    OG.IDENTatexp(
(* mikon#1.29 *)
				  addSimTrace (CoreTrace.TYPE instance)
(* end mikon#1.29 *)
		      (case (StatObject.getOverloadedTyVar instance) of
			   None => okConv i
			 | Some tv => 
			       preOverloadingConv(i, 
			          OverloadingInfo.UNRESOLVED 
				     (StatObject.mkTypeTyVar tv))),
		      OG.OP_OPT(R.LONGVAR(StatObject.mk_longvar ident), withOp)
		    )
  		   )
                 end

	      (* Constructor *)                                 (* rule 3 *)
	    | Some(Environments.LONGCON sigma) =>
		let
		  val tau = StatObject.instance sigma
		in
		  (StatObject.Id, tau,
		   let
		     val longcon = StatObject.mk_longcon ident
		   in
		     OG.IDENTatexp(
(* mikon#1.29 *)
				   addSimTrace (CoreTrace.TYPE tau)
(* end mikon#1.29 *)
				   (addTypeInfo_CON(okConv i, C,
						   StatObject.isTypeArrow tau,
						   ident
						  )),
				   OG.OP_OPT(R.LONGCON longcon, withOp)
				  )
		   end
		  )
		end

	     (* Exception constructor *)                            (* rule 4 *)
	   | Some(Environments.LONGEXCON tau) =>
		(StatObject.Id,
		 tau,
		 let
		   val excon = StatObject.mk_longexcon ident
		 in
		   OG.IDENTatexp(addTypeInfo_EXCON(okConv i, tau),
				 OG.OP_OPT(R.LONGEXCON excon, withOp)
				)
		 end)

	     (* Not found in current context *)
	   | None =>
	       (StatObject.bogus_Subst, bogus_Type(),
		 OG.IDENTatexp(lookupIdError(i, ident),
			       OG.OP_OPT(R.LONGVAR StatObject.bogusVar, withOp)
			      )
	       )
	  )

	  (* record expression *)                                  (* rule 5 *)
	| IG.RECORDatexp(i, None) =>
	    (StatObject.Id, StatObject.TypeUnit, OG.RECORDatexp(okConv i,None)) 

	  (* record expression *)
	| IG.RECORDatexp(i, Some exprow) =>
	    let
	      val (S, rho, out_exprow) = elab_exprow(C,exprow)
	    in
	      (S, StatObject.mkTypeRecType rho,
	       OG.RECORDatexp(okConv i,Some out_exprow)) 
	    end 

	  (* let expression *)                                      (* rule 6 *)
	| IG.LETatexp(i, dec, exp) => 
	    let
	      val (S1, E, out_dec)   = elab_dec(C,dec)
	      val (S2, tau, out_exp) = elab_exp((S1 onC C) C_cplus_E E, exp)
	    in

(* mikon#1.22 *)
		if Environments.is_tynameset_of_tau_in_C (tau, C) then 
		    (S2 oo S1, tau, OG.LETatexp(okConv i,out_dec,out_exp))
		else 
(* mju#1.6 *)
		    (S2 oo S1, tau, OG.LETatexp(errorConv(i, ErrorInfo.LOCAL_TYNAMES),out_dec,out_exp))
(* endmju#1.6 *)
	    end
(* end mikon#1.22 *)

	  (* parenthesised expression *)
	| IG.PARatexp(i, exp) =>                                     (* rule 7 *)
	    let val (S, tau, out_exp) = elab_exp(C,exp)
	    in (S, tau, OG.PARatexp(okConv i,out_exp)) end

(* mikon#1.5 *)

	  (* undefined value *)
	| IG.UNDEFatexp(i) =>                                     (* rule 7.1 *)
(* mikon#1.29 *)
		let
		    val tau = freshType() 
		in
		    (StatObject.Id, tau, 
		     OG.UNDEFatexp(addSimTrace (CoreTrace.CONTEXTxTYPE(C, tau)) (okConv i)))
		end
(* end mikon#1.29 *)
		
(* end mikon#1.5 *)

    (******** expression rows - Definition page 24 ********)

    and elab_exprow (C : Environments.Context, exprow : IG.exprow) :
	(Substitution * StatObject.RecType * OG.exprow) =

	case exprow of 

	  (* Expression row *)                                     (* rule 8 *)        
	  IG.EXPROW(i, lab, exp, None) =>
	    let
	      val (S, tau, out_exp) = elab_exp(C, exp)
	      val rho = StatObject.addField (lab,tau) StatObject.emptyRecType
	    in
	      (S, rho, OG.EXPROW(okConv i,lab,out_exp,None))
	    end

	  (* Expression row *)
	| IG.EXPROW(i, lab, exp, Some exprow) =>
	    let
	      val (S1, tau, out_exp   ) = elab_exp(C, exp)
	      val (S2, rho, out_exprow) = elab_exprow(S1 onC C,exprow)
	    in
	      if (List.member lab (StatObject.sortedLabsOfRecType rho)) then
		(S2, rho, 
		 OG.EXPROW(repeatedIdsError(i, [ErrorInfo.LAB_RID lab]),
			   lab, out_exp, Some out_exprow))
	      else
		(S2 oo S1, StatObject.addField (lab,S2 on tau) rho,
		 OG.EXPROW(okConv i,lab,out_exp,Some out_exprow))
	    end

    (******** expressions - Definition page 24 ********)

    and elab_exp(C : Environments.Context, exp : IG.exp) : 
	(Substitution * StatObject.Type * OG.exp) =
      let
        val _ =
	  if Flags.DEBUG_ELABDEC then
	    pr("elab_exp: ", PPInDecGrammar.layoutExp exp)
	  else ()

	val (S, ty, exp') = elab_exp'(C, exp)

	val _ =
	  if Flags.DEBUG_ELABDEC then
	    pr("giving:   ", StatObject.layoutType ty)
	  else ()
      in
	(S, ty, exp')
      end

    and elab_exp'(C, exp) =
      case exp

	   (* Atomic expression *)                                (* rule 9 *)
	of IG.ATEXPexp(i, atexp) =>
	     let
	       val (S, tau, out_atexp) = elab_atexp(C, atexp)
	     in
	       (S, tau, OG.ATEXPexp(okConv i, out_atexp))
	     end

	   (* Application expression *)                             (* rule 10 *)
	 | IG.APPexp(i, exp, atexp) => 
	     let
	       val (S1, tau1, out_exp)   = elab_exp(C, exp)
	       val (S2, tau2, out_atexp) = elab_atexp(S1 onC C, atexp)
	       val new   = freshType()
	       val arrow = StatObject.mkTypeArrow(tau2,new) 
	       val (S3, i') = Unify(arrow, S2 on tau1, i)
	     in
	       (S3 oo S2 oo S1, S3 on new, OG.APPexp(i', out_exp, out_atexp))
	     end

	   (* Typed expression *)                                   (* rule 11 *)
	 | IG.TYPEDexp(i, exp, ty) =>
	     let
	       val (S1, tau, out_exp) = elab_exp(C, exp)
	       val (tau', out_ty) = elab_ty(S1 onC C, ty)
	       val (S2, i') = Unify(tau, tau', i)
	     in
	       (S2 oo S1, S2 on tau', OG.TYPEDexp(i', out_exp, out_ty))
	     end

(* mikon#1.5 *)

	   (* Comparision expression *)                             (* rule 11.1 *)
	 | IG.COMPARexp(i, exp1, exp2) => 
	     let
(* mikon#1.20 *)
	       val U1 =
		 Environments.Unguarded_exp_TyVars(exp1)
	       val U2 =
		 Environments.Unguarded_exp_TyVars(exp2)
	       val U = 
		 Environments.TyVarSet_union(U1, U2)

	       val (S1, tau1, out_exp1)   = elab_exp(Environments.C_plus_U(C,U), exp1)
	       val (S2, tau2, out_exp2)   = elab_exp(S1 onC (Environments.C_plus_U(C,U)), exp2)
(* end mikon#1.20 *)
	       val (S3, i') = Unify(S2 on tau1, tau2, i)
(* mikon#1.32 *)
	       val sim = CoreTrace.CONTEXTxTYPE(C, tau1)
	       val scheme = CoreTrace.SCHEME_C(C)
	       val i'' = addScheme_and_Sim (scheme, sim) i'
	     in
	       (S3 oo S2 oo S1, StatObject.TypeBool, OG.COMPARexp(i'', out_exp1, out_exp2))
	     end
(* end mikon#1.32 *)  

	   (* Existential quantifier expression *)                   (* rule 11.2 *)
	 | IG.EXIST_QUANTexp(i, match) => 
	     let
(* mikon#1.20 *)
	       val U =
		 Environments.Unguarded_match_TyVars(match)

	       val (S1, tau1, out_match) = elab_match(Environments.C_plus_U(C,U), match)
(* end mikon#1.20 *)
	       val new   = freshType()
	       val arrow = StatObject.mkTypeArrow(new, StatObject.TypeBool) 
	       val (S2, i') = Unify(arrow, tau1, i)
(* mikon#1.32 *)
	       val sim = CoreTrace.CONTEXTxTYPE(C, tau1)
	       val scheme = CoreTrace.SCHEME_C(C)
	       val i'' = addScheme_and_Sim (scheme, sim) i'
	     in
	       (S2 oo S1, StatObject.TypeBool, OG.EXIST_QUANTexp(i'', out_match))
	     end
(* end mikon#1.32 *)

	   (* Universal quantifier expression *)                     (* rule 11.3 *)
	 | IG.UNIV_QUANTexp(i, match) => 
	     let
(* mikon#1.20 *)
	       val U =
		 Environments.Unguarded_match_TyVars(match)

	       val (S1, tau1, out_match) = elab_match(Environments.C_plus_U(C,U), match)
(* end mikon#1.20 *)
	       val new   = freshType()
	       val arrow = StatObject.mkTypeArrow(new, StatObject.TypeBool) 
	       val (S2, i') = Unify(arrow, tau1, i)
(* mikon#1.32 *)
	       val sim = CoreTrace.CONTEXTxTYPE(C, tau1)
	       val scheme = CoreTrace.SCHEME_C(C)
	       val i'' = addScheme_and_Sim (scheme, sim) i'
	     in
	       (S2 oo S1, StatObject.TypeBool, OG.UNIV_QUANTexp(i'', out_match))
	     end
(* end mikon#1.32 *)

	   (* Comvergence predicate expression *)                    (* rule 11.4 *)
	 | IG.CONVERexp(i, exp) => 
	     let
	       val (S, tau, out_exp) = elab_exp(C, exp)
	     in
	       (S, StatObject.TypeBool, OG.CONVERexp(okConv i, out_exp))
	     end

(* end mikon#1.5 *)

	   (* Handle exception *)                                   (* rule 12 *)
	 | IG.HANDLEexp(i, exp, match) =>
	     let
	       val (S1, tau1, out_exp)   = elab_exp(C, exp)
	       val (S2, tau2, out_match) = elab_match(S1 onC C, match)
	       val matchTy = StatObject.mkTypeArrow(StatObject.TypeExn, tau1)
	       val (S3, i') = Unify(matchTy, tau2, i)
	     in
	       (S3 oo S2 oo S1, (S3 oo S2) on tau1,
		OG.HANDLEexp(i', out_exp, out_match))
	     end

	   (* Raise exception *)                                    (* rule 13 *)
	 | IG.RAISEexp(i, exp) =>
	     let
	       val (S1, tau1, out_exp) = elab_exp(C, exp)
	       val exnType = StatObject.TypeExn
	       val (S2, i')    = Unify(exnType, S1 on tau1, i)
(* mikon#1.32 *)
	       val tau = freshType() 
	     in
	       (S2 oo S1, tau, OG.RAISEexp(addSimTrace (CoreTrace.TYPE tau) i', out_exp))
	     end
(* end mikon#1.32 *)

	   (* Function expression *)                                 (* rule 14 *)
	 | IG.FNexp(i, match) => 
	     let
	       val (S, tau, out_match) = elab_match(C, match)
	     in
	       (S, tau, OG.FNexp(
(* mikon#1.32 *)
				 addSimTrace (CoreTrace.CONTEXTxTYPE(C, tau))
(* end mikon#1.32 *)
				 (okConv i), out_match))
	     end

	 | IG.UNRES_INFIXexp _ =>
	     Crash.impossible "elab_exp(UNRES_INFIX)"


    (******** matches - Definition page 25 ********)

    and elab_match (C : Environments.Context, match : IG.match) :
	(Substitution * StatObject.Type * OG.match) =

	case match of

	  (* Match *)                                                (* rule 15 *)
	  IG.MATCH(i, mrule, None) =>
	    let val (S, tau, out_mrule) = elab_mrule(C,mrule)
	    in (S, tau, OG.MATCH(okConv i,out_mrule,None)) end

	  (* Match *)
	| IG.MATCH(i, mrule, Some match') =>
	    let
	      val (S ,tau ,out_mrule) = elab_mrule(C,mrule)
	      val (S',tau',out_match) = elab_match(S onC C,match')
	      val (S'', i') = Unify(tau',S' on tau,i)
	    in
	      (S'' oo S' oo S, S'' on tau',
	       OG.MATCH(i', out_mrule, Some out_match))
	    end

    (******** match rules - Definition page 25 ********)

    and elab_mrule (C : Environments.Context, mrule : IG.mrule) : 
	(Substitution * StatObject.Type * OG.mrule) =

	case mrule of

	  (* Match rule *)                                         (* rule 16 *)
	  IG.MRULE(i, pat, exp) =>
	    let
	      val (S, (VE,tau),out_pat) = elab_pat(C,pat)
(* mikon#1.23 *)
	      val (S',tau',  out_exp) = 
		     elab_exp(Environments.C_cplus_VE(S onC C,VE),exp)
(* end mikon#1.23 *)
	      val S'' = S' oo S
	    in
	      (S'', StatObject.mkTypeArrow(S'' on tau,tau'),
	       OG.MRULE(okConv i,out_pat,out_exp))
	    end

    (******** declarations - Definition page 25 ********)

    and elab_dec(C: Environments.Context, dec: IG.dec) :
	(Substitution * Environments.Env * OG.dec) =

	case dec of

	   (* Value declaration *)                                 (* rule 17 *)

	  IG.VALdec(i, valbind) =>
	     let
	       val U =
		 Environments.Scoped_TyVars(valbind, Environments.U_of_C C)

	       val (S, VE, out_valbind) =
		 elab_valbind(Environments.C_plus_U(C,U), valbind)

	       val VE' =
		 Environments.Clos(S onC C, valbind, VE)

	       val i' =
		 case Environments.intersect(Environments.tyvarsVE VE', U)
		   of nil => okConv i
		 | tyvars => errorConv(i, ErrorInfo.FREE_TYVARS tyvars)
	     in
	       (S, Environments.VE_in_E VE', OG.VALdec(
(* mikon#1.33 *)
						       addSchemeTrace (CoreTrace.SCHEME_C(C)) 
(* end mikon#1.33 *)
						       i', out_valbind)) 
	     end

	   (* `fun'-declaration *)
	 | IG.UNRES_FUNdec _ => Crash.impossible "elab_dec(UNRES_FUN)"

	   (* Type declaration *)                                   (* rule 18 *)
	 | IG.TYPEdec(i, typbind) =>
	     let
	       (* Note that no substitutions are produced *)
	       val (TE, out_typbind) = elab_typbind(C, typbind)
	     in
	       (StatObject.Id, Environments.TE_in_E TE,
		OG.TYPEdec(
(* mikon#1.33 *)
			   addSimTrace (CoreTrace.TYENV(TE))
(* end mikon#1.33 *)
			   (okConv i), out_typbind))
	     end

(* mikon#1.9 *)
	   (* Eqtype declaration *)                                 (* rule 18.1 *)
	 | IG.EQTYPEdec(i, typbind) =>
	     let
	       (* Note that no substitutions are produced *)
	       val (TE, out_typbind) = elab_typbind(C, typbind)
(* mju#1.6 *)
               fun f((tycon: IG.tycon, tystr: Environments.TyStr), tyfcnl)
		   : StatObject.TypeFcn list =
		   let val theta = Environments.Theta_of tystr
		       val side_condition = StatObject.admits_equality theta
		   in if side_condition 
		      then tyfcnl
		      else theta :: tyfcnl
		   end
	       val bad_thetas = Environments.TEFold f nil TE
             in
       		 if List.isEmpty bad_thetas
                 then 
	           (StatObject.Id, Environments.TE_in_E TE,
		    OG.EQTYPEdec(
(* mikon#1.33 *)
				 addSimTrace (CoreTrace.TYENV(TE))
(* end mikon#1.33 *)
				 (okConv i), out_typbind))
	         else
	           (StatObject.Id, Environments.TE_in_E TE,
		    OG.EQTYPEdec(
(* mikon#1.33 *)
				 addSimTrace (CoreTrace.TYENV(TE))
(* end mikon#1.33 *)
				 (errorConv(i, ErrorInfo.SHOULD_ADMIT_EQ(rev bad_thetas))), out_typbind))
(* end mju#1.6 *)
 	     end
(* end mikon#1.9 *)

	   (* Datatype declaration *)                               (* rule 19 *)
	 | IG.DATATYPEdec(i, datbind) =>
	     let
	       val TE = initial_TE datbind
	       val ((VE1, TE1), out_datbind) = elab_datbind(C C_cplus_TE TE, datbind)
	       val (VE2, TE2) = Environments.maximise_equality (VE1, TE1)
(* mikon#1.33 *)
	       val E = Environments.VE_and_TE_in_E(VE2, TE2)		
	     in
	       (StatObject.Id, E, OG.DATATYPEdec(addSimTrace (CoreTrace.ENV(E)) (okConv i), out_datbind))
	     end
(* end mikon#1.33 *)

	   (* Abstype declaration *)                                (* rule 20 *)
	 | IG.ABSTYPEdec(i, datbind, dec) =>
	     let
	       val TE = initial_TE datbind
	       val ((VE1, TE1), out_datbind) = elab_datbind(C C_cplus_TE TE, datbind)
	       val (VE2, TE2) = Environments.maximise_equality (VE1, TE1)
	       val (S, E, out_dec) = elab_dec(C C_cplus_VE_and_TE (VE2,TE2), dec)
(* mikon#1.33 *)
	       val (E', phi_Ty) = Environments.ABS_for_traces(TE2, E)
	     in
	       (S, E', OG.ABSTYPEdec(addSimTrace (CoreTrace.VARENVxTYREA(VE2, phi_Ty)) (okConv i), 
									 out_datbind, out_dec))
	     end
(* end mikon#1.33 *)

	   (* Exception declaration *)                              (* rule 21 *)
	 | IG.EXCEPTIONdec(i, exbind) =>
	     let
	       val (EE, out_exbind) = elab_exbind(C, exbind)
	       val VE = Environments.VE_of_EE EE
	     in
	       (StatObject.Id,
		Environments.VE_and_EE_in_E(VE, EE),
		OG.EXCEPTIONdec(okConv i, out_exbind))
	     end

(* mikon#2
           (* Axiom declaration *)                                (* rule 21.1 *)
         | IG.AXIOMdec(i, ax) =>
             let
               val (S, out_ax) = elab_ax(C, ax)
             in    
               (S, Environments.VE_in_E Environments.emptyVE, OG.AXIOMdec(okConv i, out_ax))
             end
   end mikon#2 & commented out by mikon#1.1 *)
 
	   (* Local declaration *)                                  (* rule 22 *)
	 | IG.LOCALdec(i, dec1, dec2) =>
	     let
	       val (S1, E1, out_dec1) = elab_dec(C,dec1)
	       val (S2, E2, out_dec2) = elab_dec((S1 onC C) C_cplus_E E1,dec2)
	     in
	       (S2 oo S1, E2, OG.LOCALdec(okConv i,out_dec1,out_dec2))
	     end

	   (* Open declaration *)                                   (* rule 23 *)
	 | IG.OPENdec(i, list) =>
	     let
	       fun process(E0, list)
		   : Env * Environments.longstrid OG.WithInfo list =
		 case list
		   of IG.WITH_INFO(i, longstrid) :: rest =>
		        (case Environments.Lookup_longstrid(C, longstrid)
			   of Some Str =>
			        let
				  val (_, E) = Environments.unStr Str
				  val (E', rest') = process(E0, rest)
				in
				  (Environments.E_plus_E(E, E'),
				   OG.WITH_INFO(okConv i, longstrid) :: rest'
				  )
				end

			    | None =>	(* Lookup failure: process rest of
					   list and build env regardless. *)
				let
				  val (E', rest') = process(E0, rest)
				  val ei = ErrorInfo.LOOKUP_LONGSTRID longstrid
				in
				  (E', OG.WITH_INFO(errorConv(i, ei), longstrid)
				       :: rest'
				  )
				end
			)

		    | nil => (Environments.emptyE, nil)

	       val (E', list') = process(Environments.emptyE, list)
	     in
	       (StatObject.Id, E', OG.OPENdec(okConv i, list'))
	     end

	 | IG.INFIXdec(i, prec, ids) =>    (* infix -- no rule in Definition *)
	     (StatObject.Id,
	      Environments.VE_in_E Environments.emptyVE,
	      OG.INFIXdec(okConv i,
			  case prec
			    of Some p => Some p
			     | None => None,
			  map StatObject.mk_var ids
			 )
	     )

	 | IG.INFIXRdec(i, prec, ids) =>   (* infixr -- no rule in Definition *)
	     (StatObject.Id,
	      Environments.VE_in_E Environments.emptyVE,
	      OG.INFIXRdec(okConv i,
			   case prec
			     of Some p => Some p
			      | None => None,
			   map StatObject.mk_var ids
			  )
	     )

	 | IG.NONFIXdec(i, ids) =>         (* nonfix -- no rule in Definition *)
	     (StatObject.Id,
	      Environments.VE_in_E Environments.emptyVE,
	      OG.NONFIXdec(okConv i, map StatObject.mk_var ids)
	     )

	   (* Empty declaration *)                                     (* rule 24 *)
	 | IG.EMPTYdec(i) =>
	     (StatObject.Id, Environments.VE_in_E Environments.emptyVE, OG.EMPTYdec(okConv i))

	   (* Sequential declaration *)                                (* rule 25 *)
	 | IG.SEQdec(i, dec1, dec2) =>
	     let
	       val (S1, E1, out_dec1) = elab_dec(C,dec1)
	       val (S2, E2, out_dec2) = elab_dec((S1 onC C) C_cplus_E E1,dec2)
	       val E1' = Environments.onE(S2,E1)
	     in
	       (S2 oo S1,
		Environments.E_plus_E(E1',E2),
		OG.SEQdec(okConv i,out_dec1,out_dec2)) 
	     end

    (****** value bindings - Definition page 26 ******)

    and elab_valbind(C: Environments.Context, valbind: IG.valbind)
          : (Substitution * Environments.VarEnv * OG.valbind) =

	case valbind of

	(* Simple value binding *)                                    (* rule 26 *)
        IG.PLAINvalbind(i, pat, exp, valbind_opt) =>
	  let
	    val (S0, (VE,tau), out_pat) = elab_pat(C, pat)
	    val (S1, tau1, out_exp) = elab_exp(S0 onC C, exp)
	    val (S2, i') = Unify((S1 oo S0) on tau, tau1, i)

	    val (S3, VE', valbind_opt') =
	      case valbind_opt
		of Some valbind =>
		     let
		       val (S, VE, vb) =
			 elab_valbind((S2 oo S1 oo S0) onC C, valbind)
		     in
		       (S, VE, Some vb)
		     end

		 | None =>
		     (StatObject.Id, Environments.emptyVE, None)
	    val intdom = EqSet.intersect (Environments.VEdom VE)
		                         (Environments.VEdom VE')
	  in 
	    if EqSet.isEmpty intdom then
	      (S3 oo S2 oo S1 oo S0, 
	       Environments.VE_plus_VE((S3 oo S2 oo S1 oo S0) onVE VE, VE'),
	       OG.PLAINvalbind(i', out_pat, out_exp, valbind_opt')
	       )
	    else
	      (S3, VE',
	       OG.PLAINvalbind((case (GrammarInfo.getPostElabErrorInfo i') of
				  None => 
				   GrammarInfo.addPostElabErrorInfo i' 
				    (ErrorInfo.REPEATED_IDS 
				     (map ErrorInfo.ID_RID (EqSet.list intdom)))
			        | Some _ => i'),
			       out_pat, out_exp, valbind_opt'))
	  end
                                                                   (* rule 27 *)	
				(* Recursive value binding. Rather tricky
				   because we have to plant error info after
				   the second pass. Make that `very tricky.' *)
      | IG.RECvalbind(i, valbind) => 
	  let
				(* Function to unify the occurrence of a
				   variable in two VE's. The result is a
				   substitution and an ErrorInfo tag. *)
	    fun processID(i, VE, VE', var): Substitution * PostElabGrammarInfo =
	      case
		(Environments.lookupVE(VE, Var.unVar var),
		 Environments.lookupVE(VE', Var.unVar var))
	      of
		(Some(Environments.LONGVAR sigma1),
		 Some(Environments.LONGVAR sigma2)) =>
		  let
		    val (_, tau1) = StatObject.unTypeScheme sigma1
		    val (_, tau2) = StatObject.unTypeScheme sigma2
		  in
		    case StatObject.unify(tau1, tau2) of
		      Some S =>
			(S, i)
		    | None =>
			(StatObject.bogus_Subst,
			 GrammarInfo.addPostElabErrorInfo i
			 (ErrorInfo.UNIFICATION(tau1, tau2)))
		  end

	       | _ => Crash.impossible "ElabDec.unifyVE"

				(* Traverse the out_valbind, doing a
				   unification (and adding ErrorInfo if reqd.)
				   and giving a subst. at each stage. The
				   ErrorInfo goes into the pattern...
				   ...somewhere... *)
	    fun traverseRecValbind(VE, VE', vb): Substitution * OG.valbind =
	      case vb
		of OG.PLAINvalbind(i, pat, exp, vb_opt) =>
		     let
		       val (S, pat') = traverseRecPat(VE, VE', pat)

		       val (S', vb_opt') =
			 case vb_opt
			   of Some vb =>
			        let
				  val (S', vb') =
				    traverseRecValbind
				    (S onVE VE, S onVE VE', vb)
				in
				  (S' oo S, Some vb')
				end

			    | None => (S, None)
		     in
		       (S', OG.PLAINvalbind(i, pat', exp, vb_opt'))
		     end

		 | OG.RECvalbind(i, vb) =>
		     let
		       val (S, vb') = traverseRecValbind(VE, VE', vb)
		     in
		       (S, OG.RECvalbind(i, vb'))
		     end

	    and traverseRecPat(VE, VE', pat): Substitution * OG.pat =
	      case pat
		of OG.ATPATpat(i, atpat) =>
		     let
		       val (S, atpat') = traverseRecAtpat(VE, VE', atpat)
		     in
		       (S, OG.ATPATpat(i, atpat'))
		     end

(*				(* We can reject a CONS node. *)
		 | OG.CONSpat(i, id, atpat) =>
		     (StatObject.bogus_Subst,
		      OG.CONSpat(
		        GrammarInfo.addPostElabErrorInfo
			  i ErrorInfo.NOTALLOWEDINREC,
			id, atpat
		      )
		     )
*)
                 | OG.CONSpat(i, id, atpat) =>
		     let 
		       val (S, atpat') = traverseRecAtpat(VE, VE', atpat)
		     in
		       (S, OG.CONSpat(i, id, atpat'))
		     end

		 | OG.TYPEDpat(i, pat, ty) =>
		     let
		       val (S, pat') = traverseRecPat(VE, VE', pat)
		     in
		       (S, OG.TYPEDpat(i, pat', ty))
		     end

		 | OG.LAYEREDpat(i, id as OG.OP_OPT(id', withOp),
				 ty_opt, pat
				) =>
		     let
		       val (S, pat') = traverseRecPat(VE, VE', pat)
		       val (S', i') = processID(i, S onVE VE, S onVE VE', id')
		     in
		       (S oo S', OG.LAYEREDpat(i', id, ty_opt, pat'))
		     end
	    
	         | OG.UNRES_INFIXpat _ =>
		     Crash.impossible "traverseRecPat(UNRES_INFIX)"

	    and traverseRecAtpat(VE, VE', atpat): Substitution * OG.atpat =
	      case atpat
		of OG.WILDCARDatpat _ =>
		     (StatObject.Id, atpat)

		 | OG.SCONatpat _ =>
		     (StatObject.Id, atpat)

		 | OG.LONGIDatpat(i, longid as OG.OP_OPT(longid', withOp)) =>
		     (case longid'
			of R.LONGVAR longvar =>
			     (case StatObject.decomposeLongVar longvar
			        of (nil, var) =>
				     let
				       val (S, i') = processID(i, VE, VE', var)
				     in
				       (S, OG.LONGIDatpat(i', longid))
				     end
				    
			         | _ =>
				     Crash.impossible "traverseRecAtpat(longid)"
			     )

	                 | _ =>		(* LONGCON, LONGEXCON allowed. *)
			     (StatObject.Id, atpat)
		     )

(*		 | OG.RECORDatpat(i, patrowOpt) =>
		     (StatObject.bogus_Subst,
		      OG.RECORDatpat(
			GrammarInfo.addPostElabErrorInfo
			  i ErrorInfo.NOTALLOWEDINREC,
			patrowOpt
		      )
		     )

*)
		 | OG.RECORDatpat(i, patrowOpt) =>
		     (case patrowOpt of 
			None => (StatObject.Id,atpat)
	              | Some patrow => 
			  let 
			    val (S, patrow') =
			      traverseRecPatrow (VE, VE', patrow)
			  in
			    (S, OG.RECORDatpat(i, Some patrow'))
			  end)

		 | OG.PARatpat(i, pat) =>
		     let
		       val (S, pat') = traverseRecPat(VE, VE', pat)
		     in
		       (S, OG.PARatpat(i, pat'))
		     end

	    and traverseRecPatrow(VE, VE', patrow): Substitution * OG.patrow =
	      case patrow of 
		OG.DOTDOTDOT i => (StatObject.Id, patrow)
	      | OG.PATROW(i, l, pat, patrowOpt) =>
		  let 
		    val (S, pat') = traverseRecPat(VE, VE', pat)
		    val (S', patrowOpt') =
		      (case patrowOpt of 
			None => (StatObject.Id, None)
		      | Some patrow => 
			  let 
			    val (S'', patrow') = traverseRecPatrow(VE, VE', patrow)
			  in
			    (S'', Some patrow')
			  end)
		  in
		    (S' oo S, OG.PATROW(i, l, pat', patrowOpt'))
		  end


				(* set up a value environment, VE,
				   for the recursively declared values *)

	    val domain_list = dom_vb(C, valbind)

	    fun freshTypeScheme() =
	      StatObject.Type_in_TypeScheme(freshType())

	    fun setup id VE =
	      Environments.VE_plus_VE(
		VE,
		Environments.singleVarVE(id, freshTypeScheme())
	      )

	    val VE = List.foldL setup Environments.emptyVE domain_list
				(* VE now maps each rec identifier to 'a. *)

				(* Proceed with type checking. The ErrorInfo
				   tags for the rec identifiers will be
				   untouched (I hope, since we might assign
				   them further on). *)
	    val (S, VE', valbind') =
	      elab_valbind(Environments.C_plus_VE(C, VE), valbind)

				(* Post-pass, to patch up the rec identifiers
				   and plant unification error tags: *)
	    val (S', valbind'') =
	      traverseRecValbind(S onVE VE, VE', valbind')

	    val VE'' = S' onVE VE'

	    val _ =
	      if Flags.DEBUG_ELABDEC then
		let
		  val t = PP.NODE{start="{", finish="}", indent=0,
				  children=[Environments.layoutVE VE''],
				  childsep=PP.NONE
				 }
		in
		  pr("RECvalbind: ", t)
		end
	      else ()
	  in
	    (S' oo S, VE'', OG.RECvalbind(okConv i, valbind''))
	  end

    (******* type bindings - Definition page 27 *******)

    and elab_typbind (C : Environments.Context, typbind : IG.typbind) :
      (Environments.TyEnv * OG.typbind) =

      case typbind of

	(* Type binding *)                                         (* rule 28 *)
	IG.TYPBIND(i, tyvar_list, tycon, ty, typbind_opt) =>
	  let
	    val TyVar_list =
	      map (fn tv => StatObject.mkExplicitTyVar tv) tyvar_list
	    val tyvarsRepeated = getRepeatedElements TyVar_list
	    val tyvarsNotInTyVarList =
	      List.all 
	        (fn tv => not (List.member tv tyvar_list)) 
		(IG.getExplicitTyVarsTy ty)
	    val (tau, out_ty) = elab_ty(C, ty)
	    val typeFcn = StatObject.mkTypeFcn(TyVar_list, tau)
	    val tystr = Environments.mkTyStr(typeFcn, Environments.emptyCE)

	    val (TE, out_typbind_opt) = elab_typbind_opt(C, typbind_opt)
	      
	  in
	    if tyvarsNotInTyVarList <> [] then
	      (Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE),
	       OG.TYPBIND(errorConv(i, 
                      ErrorInfo.TYVARS_NOT_IN_TYVARSEQ 
			    (map StatObject.mkExplicitTyVar tyvarsNotInTyVarList)),
			  tyvar_list, tycon, out_ty, out_typbind_opt))
	    else
	      if (List.member tycon (Environments.TEdom TE)) then 
		(Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE),
		 OG.TYPBIND(repeatedIdsError(i, [ErrorInfo.TYCON_RID tycon]),
			    tyvar_list, tycon, out_ty, out_typbind_opt))
	      else
		if (tyvarsRepeated <> []) then
		  (Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE),
		   OG.TYPBIND(repeatedIdsError(i,
				map ErrorInfo.TYVAR_RID tyvarsRepeated),
			      tyvar_list, tycon, out_ty, out_typbind_opt))
		else
		  (Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE),
		   OG.TYPBIND(okConv i, tyvar_list, tycon, out_ty, out_typbind_opt))
	  end

(* mikon#1.10 *)
	(* Qestion mark type binding *)                                         (* rule 28.1 *)
	| IG.QUEST_TYPBIND(i, tyvar_list, tycon, typbind_opt) =>
	  let
	    val TyVar_list =
	      map (fn tv => StatObject.mkExplicitTyVar tv) tyvar_list
	    val tyvarsRepeated = getRepeatedElements TyVar_list
(* end mikon#1.10 *)

(* mikon#1.13 *)
	val arity =
	  List.size tyvar_list

	val tyname =
	  StatObject.freshTyName {name = tycon, arity = arity, equality = false}

	val typeFcn =
	  StatObject.TyName_in_TypeFcn tyname
(* end mikon#1.13 *)

(* mikon#1.10 
	    val typeFcn = StatObject.mkTypeFcn(TyVar_list, freshType())
   end mikon#1.10 & commented out by mikon#1.13 *)

(* mikon#1.10 *)
	    val t = Environments.mkTyStr(typeFcn, Environments.emptyCE)
	    val (TE, out_typbind_opt) = elab_typbind_opt(C, typbind_opt)
	  in
	      if (List.member tycon (Environments.TEdom TE)) then 
		(Environments.TE_plus_TE(Environments.singleTE(tycon, t), TE),
		 OG.QUEST_TYPBIND(
(* mikon#1.40 *)
				  addSimTrace (CoreTrace.CONTEXTxTYNAME(C, tyname))
(* end mikon#1.40 *)
                                 (repeatedIdsError(i, [ErrorInfo.TYCON_RID tycon])),
			         tyvar_list, tycon, out_typbind_opt))
	      else
		if (tyvarsRepeated <> []) then
		  (Environments.TE_plus_TE(Environments.singleTE(tycon, t), TE),
		   OG.QUEST_TYPBIND(
(* mikon#1.40 *)
				    addSimTrace (CoreTrace.CONTEXTxTYNAME(C, tyname))
(* end mikon#1.40 *)
                                    (repeatedIdsError(i,
				map ErrorInfo.TYVAR_RID tyvarsRepeated)),
			      tyvar_list, tycon, out_typbind_opt))
		else
		  (Environments.TE_plus_TE(Environments.singleTE(tycon, t), TE),
		   OG.QUEST_TYPBIND(
(* mikon#1.40 *)
				    addSimTrace (CoreTrace.CONTEXTxTYNAME(C, tyname))
(* end mikon#1.40 *)
				    (okConv i), tyvar_list, tycon, out_typbind_opt))
	  end
(* end mikon#1.10 *)

    and elab_typbind_opt (C : Environments.Context, typbind_opt : IG.typbind Option) :
      (Environments.TyEnv * OG.typbind Option) =

      case typbind_opt of

	Some(typbind) =>
	  let
	    val (TE, out_typbind) = elab_typbind(C, typbind)
	  in
	    (TE, Some out_typbind)
	  end

      | None =>
	  (Environments.emptyTE, None)

    (******* datatype bindings - Definition page 27 *******)

    and elab_datbind (C : Environments.Context, datbind : IG.datbind)
      : ((Environments.VarEnv * Environments.TyEnv) * OG.datbind) =

      case datbind of

	(* Datatype binding *)                                  (* rule 29 *)
	IG.DATBIND(i, tyvar_list, tycon, conbind, datbind_opt) =>
	  let
	    val TyVar_list =
	      map (fn tv => StatObject.mkExplicitTyVar tv) tyvar_list
	    val tyvarsRepeated = getRepeatedElements TyVar_list
	    val tyvarsNotInTyVarList =
	      List.all 
	        (fn tv => not (List.member tv tyvar_list)) 
		(IG.getExplicitTyVarsConbind conbind)
	    val (typeFcn, _) = 
	      case Environments.Lookup_tycon(C, tycon) of
		Some(tystr) => Environments.unTyStr(tystr)
	      | None => Crash.impossible "ElabDec.datbind(1)"

	    val tyname =
	      case StatObject.unTyName_TypeFcn(typeFcn) of
		Some(tyname) => tyname
	      | None => Crash.impossible "ElabDec.datbind(2)"

	    val tau_list =
	      map StatObject.mkTypeTyVar TyVar_list
	    val tau =
	      StatObject.mkTypeConsType(StatObject.mkConsType(tau_list, tyname))

(***
	    val cons = allCons conbind
 ***)
	    val (CE, out_conbind) = elab_conbind(C, tau, conbind)

	    val ClosCE = Environments.ClosCE CE
	    val tystr = Environments.mkTyStr(StatObject.TyName_in_TypeFcn tyname, ClosCE)

	    val ((VE, TE), out_datbind_opt) = elab_datbind_opt(C, datbind_opt)
	    val CE_closed_to_VE = Environments.ClosCE_to_VE CE
	    val intdom = EqSet.intersect (Environments.VEdom VE) 
	                                 (Environments.VEdom CE_closed_to_VE)
	  in
	    if tyvarsNotInTyVarList <> [] then 
		((Environments.VE_plus_VE(CE_closed_to_VE, VE),
		  Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE)),
		 OG.DATBIND(errorConv(i,
		      ErrorInfo.TYVARS_NOT_IN_TYVARSEQ 
                       (map StatObject.mkExplicitTyVar tyvarsNotInTyVarList)),
			    tyvar_list, tycon, out_conbind, out_datbind_opt))
	    else
	      if tyvarsRepeated <> [] then
		((Environments.VE_plus_VE(CE_closed_to_VE, VE),
		  Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE)),
		 OG.DATBIND(repeatedIdsError(i,
			      map ErrorInfo.TYVAR_RID tyvarsRepeated),
			    tyvar_list, tycon, out_conbind, out_datbind_opt))
	      else
		case (EqSet.isEmpty intdom, List.member tycon (Environments.TEdom TE)) of
		  (true, false) => 
		    ((Environments.VE_plus_VE(CE_closed_to_VE, VE),
		      Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE)),
		     OG.DATBIND(okConv i, tyvar_list, tycon, out_conbind, out_datbind_opt))
		| (true, true) => 
		    ((Environments.VE_plus_VE(CE_closed_to_VE, VE),
		      Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE)),
		     OG.DATBIND(repeatedIdsError(i, [ErrorInfo.TYCON_RID tycon]),
				tyvar_list, tycon, out_conbind, out_datbind_opt))
		| (false, false) =>
		    ((Environments.VE_plus_VE(CE_closed_to_VE, VE),
		      Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE)),
		     OG.DATBIND(repeatedIdsError(i, 
			 map ErrorInfo.ID_RID (EqSet.list intdom)),
				tyvar_list, tycon, out_conbind, out_datbind_opt))
		| (false, true) => 
		    ((Environments.VE_plus_VE(CE_closed_to_VE, VE),
		      Environments.TE_plus_TE(Environments.singleTE(tycon, tystr), TE)),
		     OG.DATBIND(repeatedIdsError(i, [ErrorInfo.TYCON_RID tycon] @
			 map ErrorInfo.ID_RID (EqSet.list intdom)),
				tyvar_list, tycon, out_conbind, out_datbind_opt))
	  end

    and elab_datbind_opt (C : Environments.Context, datbind_opt : IG.datbind Option)
      : ((Environments.VarEnv * Environments.TyEnv) * OG.datbind Option) =

      case datbind_opt of

	Some(datbind) =>
	  let
	    val ((VE, TE), out_datbind) = elab_datbind(C, datbind)
	  in
	    ((VE, TE), Some out_datbind)
	  end

       | None =>
	  ((Environments.emptyVE, Environments.emptyTE), None)

    (****** constructor bindings - Definition page 27 *****)

    and elab_conbind(C: Environments.Context,
		     tau: StatObject.Type,
		     conbind: IG.conbind
		    ): (Environments.ConEnv * OG.conbind) =

      case conbind of

	(* Constructor binding *)                            (* rule 30 *)
	IG.CONBIND(i, IG.OP_OPT(con, withOp), Some ty, conbind_opt) =>
	  let
	    val (tau', out_ty) = elab_ty(C, ty)
(* mikon#1.34 *)
	    val tau'' = StatObject.mkTypeArrow(tau', tau)
	    val arrow = StatObject.Type_in_TypeScheme tau''

	    val (CE, out_conbind_opt) =
	      elab_conbind_opt(C, tau, conbind_opt)
	  in
	    if List.member con (Environments.domCE CE) then
	      (Environments.CE_plus_CE(Environments.singleCE(con, arrow), CE),
	       OG.CONBIND(addSimTrace (CoreTrace.TYPE tau'')
(* end mikon#1.34 *)
			  (repeatedIdsError(i, [ErrorInfo.CON_RID con])),
			  OG.OP_OPT(con, withOp),
			  Some out_ty, out_conbind_opt))
	    else
	      (Environments.CE_plus_CE(Environments.singleCE(con, arrow), CE),
	       OG.CONBIND(
(* mikon#1.34 *)
			  addSimTrace (CoreTrace.TYPE tau'')
(* end mikon#1.34 *)
			  (okConv i), OG.OP_OPT(con, withOp),
			  Some out_ty, out_conbind_opt))
	  end

	(* Constructor binding *)
      | IG.CONBIND(i, IG.OP_OPT(con, withOp), None, conbind_opt) =>
	  let
	    val (CE, out_conbind_opt) =
	      elab_conbind_opt(C, tau, conbind_opt)
	    val ts = StatObject.Type_in_TypeScheme tau
	  in
	    if List.member con (Environments.domCE CE) then
	      (Environments.CE_plus_CE(Environments.singleCE(con, ts), CE),
	       OG.CONBIND(
(* mikon#1.34 *)
			  addSimTrace (CoreTrace.TYPE tau)
(* end mikon#1.34 *)
			  (repeatedIdsError(i, [ErrorInfo.CON_RID con])),
			  OG.OP_OPT(con, withOp), None, out_conbind_opt))
	    else
	      (Environments.CE_plus_CE(Environments.singleCE(con, ts), CE),
	       OG.CONBIND(
(* mikon#1.34 *)
			  addSimTrace (CoreTrace.TYPE tau)
(* end mikon#1.34 *)
			  (okConv i), 
			  OG.OP_OPT(con, withOp), None, out_conbind_opt))
	  end

    and elab_conbind_opt(C: Environments.Context,
			 tau: StatObject.Type,
			 conbind_opt: IG.conbind Option
			): (Environments.ConEnv * OG.conbind Option) =

      case conbind_opt of

	Some(conbind) =>
	  let
	    val (CE, out_conbind) = elab_conbind(C, tau, conbind)
	  in
	    (CE, Some out_conbind)
	  end

      | None =>
	  (Environments.emptyCE, None)

    (****** exception bindings - Definition page 27 *****)

    and elab_exbind (C : Environments.Context, exbind : IG.exbind)
      : (Environments.ExConEnv * OG.exbind) =

      case exbind of

	(* Exception binding *)                              (* rule 31 *)
	IG.EXBIND(i, IG.OP_OPT(excon, withOp), Some ty, rest) =>
	  let
	    val (tau, out_ty) = elab_ty(C, ty)
	    val exnTy = StatObject.mkTypeArrow(tau, StatObject.TypeExn)
	    val EE_this = Environments.singleEE(excon, exnTy)
	    val (EE_rest, out_rest) = elab_exbind_opt(C, rest)
	    val intdom = EqSet.intersect (Environments.EEdom EE_this)
                                         (Environments.EEdom EE_rest)
	  in
	    if EqSet.isEmpty intdom then
	      if StatObject.isImperativeType tau then
		(Environments.EE_plus_EE(EE_this, EE_rest),
		 OG.EXBIND(
(* mikon#1.34 *)
			   addSimTrace (CoreTrace.TYPE tau)
(* end mikon#1.34 *)
			   (okConv i), OG.OP_OPT(excon, withOp), Some out_ty, out_rest))
	      else		  
		(EE_rest, 
		 OG.EXBIND(
(* mikon#1.34 *)
			   addSimTrace (CoreTrace.TYPE tau)
(* end mikon#1.34 *)
			   (errorConv (i, (ErrorInfo.NOTIMPERATIVE tau))),
			   OG.OP_OPT(excon, withOp), Some out_ty, out_rest))
	    else
              (Environments.EE_plus_EE(EE_this, EE_rest),
	       OG.EXBIND(
(* mikon#1.34 *)
			 addSimTrace (CoreTrace.TYPE tau)
(* end mikon#1.34 *)
			 (repeatedIdsError(i, [ErrorInfo.EXCON_RID excon])),
			 OG.OP_OPT(excon, withOp), Some out_ty, out_rest))
	  end

	(* Exception binding *)                            (* rule 32 *)
      | IG.EXBIND(i, IG.OP_OPT(excon, withOp), None, rest) =>
	  let
	    val EE_this = Environments.singleEE(excon, StatObject.TypeExn)
	    val (EE_rest, out_rest) = elab_exbind_opt(C, rest)
	    val intdom = EqSet.intersect (Environments.EEdom EE_this)
                                         (Environments.EEdom EE_rest)
	  in
	    if EqSet.isEmpty intdom then
	      (Environments.EE_plus_EE(EE_this, EE_rest),
	       OG.EXBIND(okConv i, OG.OP_OPT(excon, withOp), None, out_rest))
	    else
	      (Environments.EE_plus_EE(EE_this, EE_rest),
	       OG.EXBIND(repeatedIdsError(i, [ErrorInfo.EXCON_RID excon]),
			 OG.OP_OPT(excon, withOp), None, out_rest))
	  end

	(* Exception binding *)
      | IG.EXEQUAL(i, IG.OP_OPT(excon, exconOp),
		      IG.OP_OPT(longid, longidOp), rest
		  ) =>
	  (case Environments.Lookup_longid(C, longid) of

	     Some(Environments.LONGEXCON tau) =>
	       let
		 val EE_this = Environments.singleEE(excon, tau)
		 val (EE_rest, out_rest) = elab_exbind_opt(C, rest)
		 val longexcon = R.LONGEXCON(StatObject.mk_longexcon longid)
		 val intdom = EqSet.intersect (Environments.EEdom EE_this)
		                              (Environments.EEdom EE_rest)

	       in
		 if EqSet.isEmpty intdom then
		   (Environments.EE_plus_EE(EE_this, EE_rest),
		    OG.EXEQUAL(okConv i, OG.OP_OPT(excon, exconOp),
			       OG.OP_OPT(longexcon, longidOp), out_rest))
		 else
		   (Environments.EE_plus_EE(EE_this, EE_rest),
		    OG.EXEQUAL(repeatedIdsError(i, [ErrorInfo.EXCON_RID excon]),
			       OG.OP_OPT(excon, exconOp),
			       OG.OP_OPT(longexcon, longidOp), out_rest))
	       end

	   | _ =>
	       let
		 (* Carry on, building an error node. *)
		 val (EE_rest, out_rest) = elab_exbind_opt(C, rest)
		 val longexcon = R.LONGEXCON(StatObject.mk_longexcon longid)
	       in
		 (EE_rest, OG.EXEQUAL(
			     lookupIdError(i, longid),
			     OG.OP_OPT(excon, exconOp),
			     OG.OP_OPT(longexcon, longidOp),
			     out_rest
			   )
		 )
	       end)

    and elab_exbind_opt(C, Some exbind) =
	  let
	    val (EE, out_exbind) = elab_exbind(C, exbind)
	  in
	    (EE, Some out_exbind)
	  end

      | elab_exbind_opt(C, None) =
	  (Environments.emptyEE, None)

(* mikon#2 
    (****** axiom body  ******)
    and elab_ax (C : Environments.Context, ax : IG.ax) :
        (Substitution * OG.ax) =
 
        case ax of

          (* Expression *)                                    (* rule 32.1 *)
          IG.AX(i, exp)  =>
            let
              val (S1, tau, out_exp) = elab_exp(C, exp)
              val (S2, i') = Unify(tau, StatObject.TypeBool, i)           
            in
              (S2 oo S1, OG.AX(i', out_exp))
            end
   end mikon#2 & commented out by mikon#1.1 *)

    (****** atomic patterns - Definition page 28   ******)

    and elab_atpat (C : Environments.Context, atpat : IG.atpat) :
	(Substitution * (Environments.VarEnv * StatObject.Type) * OG.atpat) =

	case atpat of

	  (* Wildcard *)                                       (* rule 33 *)
	  IG.WILDCARDatpat i  =>
(* mikon#1.34 *)
          let
	      val tau = freshType()
	  in
	      (StatObject.Id, (Environments.emptyVE, tau),
	       OG.WILDCARDatpat(addSimTrace (CoreTrace.TYPE tau) (okConv i)))
	  end
(* end mikon#1.34 *)
	  (* Special constant *)                               (* rule 34 *)
	| IG.SCONatpat(i,scon) =>
	    (StatObject.Id,
	     (Environments.emptyVE, StatObject.GetTypescon scon),
	      OG.SCONatpat(okConv i,scon))

	  (* Long identifier *)
	| IG.LONGIDatpat(i, IG.OP_OPT(longid, withOp)) =>
	    (case Environments.Lookup_longid(C, longid)
	       of Some(Environments.LONGCON sigma) =>          (* rule 36 *)
		    let
		      fun isConsType tau =
			case StatObject.unTypeConsType tau
			  of Some _ => true
			   | None => false

		      val tau = StatObject.instance sigma
		      
		      val (tau', i') =
			if isConsType tau then
			  (tau, okConv i)
			else
			  (bogus_Type(),
			   errorConv(i, ErrorInfo.NOTCONSTYPE tau)
			  )
		    in
		      (StatObject.Id,
		       (Environments.emptyVE, tau'),
		       let
			 val longcon = StatObject.mk_longcon longid
		       in
			 OG.LONGIDatpat(
(* mikon#1.34 *)
					addSimTrace (CoreTrace.TYPE tau')
(* end mikon#1.34 *)
					(addTypeInfo_CON(i', C,
							 StatObject.isTypeArrow tau', longid
							 )),
					OG.OP_OPT(R.LONGCON longcon, withOp)
					)
		       end
		      )
		    end

	        | Some(Environments.LONGEXCON tau) =>            (* rule 37 *)
		    let
		      val longexcon = StatObject.mk_longexcon longid
		      val exnType = StatObject.TypeExn
		      val (_, i') = Unify(tau, exnType, i)
		    in
		      (StatObject.Id,
		       (Environments.emptyVE, exnType),
		       OG.LONGIDatpat(i',
				      OG.OP_OPT(R.LONGEXCON longexcon, withOp)
				     )
		      )
		    end

		| _ =>		(* make new variable environment *) (* unbound long identifier *)
		    let
		      val var = StatObject.mk_longvar longid
		      val tau = freshType()
		      val tau_scheme = StatObject.Type_in_TypeScheme tau
		    in
		      case StatObject.decomposeLongId longid
			of (nil, id) =>
			     (StatObject.Id,
			      (Environments.singleVarVE(id, tau_scheme), tau),
			      OG.LONGIDatpat(
(* mikon#1.34 *)
					     addSimTrace (CoreTrace.TYPE tau)
(* end mikon#1.34 *)
					     (okConv i),
					     OG.OP_OPT(R.LONGVAR var, withOp)
					    )
			     )

		         | (_, _) =>
			     (StatObject.bogus_Subst,
			      (Environments.bogus_VE, bogus_Type()),
			      OG.LONGIDatpat(
				errorConv(i, ErrorInfo.QUALIFIED_ID longid),
				OG.OP_OPT(R.LONGVAR var, withOp)
			      )
			     )
		    end
	    )

	  (* Record pattern *)                                  (* rule 38 *)
	| IG.RECORDatpat(i, row_opt as None) =>
	    (StatObject.Id,
	     (Environments.emptyVE, StatObject.TypeUnit),
	      OG.RECORDatpat(okConv i, None)
	     )

	  (* Record pattern *)
	| IG.RECORDatpat(i, row_opt as Some patrow) =>
	    let
	      val (S, (VE, rho), out_patrow) = elab_patrow(C, patrow)
	    in
	      (S, 
	       (VE,StatObject.mkTypeRecType rho),
		OG.RECORDatpat(okConv i, Some(addLabelInfo(rho, out_patrow)))
	      ) 
	    end

	  (* Parenthesised pattern *)
	| IG.PARatpat(i, pat) =>
	    let val (S, (VE,tau), out_pat) = elab_pat(C, pat)
	    in (S, (VE,tau), OG.PARatpat(okConv i,out_pat)) end

    (****** pattern rows - Definition page 28 ******)

    and elab_patrow(C: Environments.Context, patrow: IG.patrow)
	  : (Substitution * (Environments.VarEnv * StatObject.RecType) * OG.patrow) =
      case patrow of

	   (* Pattern row *)                                  (* rule 41 *)
	   IG.PATROW(i, lab, pat, None) =>
	     let
	       val (S, (VE, tau), out_pat) = elab_pat(C, pat)
	     in
	       (S, (VE, StatObject.addField (lab, tau) StatObject.emptyRecType),
		OG.PATROW(okConv i, lab, out_pat, None)
	       )
	     end

	 | IG.PATROW(i, lab, pat, Some patrow) =>
	     let
	       val (S, (VE, tau), out_pat) = elab_pat(C, pat)
	       val (S', (VE', rho), out_patrow) = elab_patrow(C, patrow)
	       val intdom = EqSet.intersect (Environments.VEdom VE)
		                              (Environments.VEdom VE')
	     in
	       case (EqSet.isEmpty intdom, 
		     List.member lab (StatObject.sortedLabsOfRecType rho)) of
		 (true, false) =>
		   (S' oo S,
		    (Environments.VE_plus_VE(VE, VE'),
		     StatObject.addField (lab, tau) rho
		     ), OG.PATROW(okConv i, lab, out_pat, Some out_patrow)
		    )
	       | (true, true) => 
		   (StatObject.bogus_Subst,
		    (VE', rho),
		    OG.PATROW(repeatedIdsError(i,[ErrorInfo.LAB_RID lab]),
			      lab, out_pat, Some out_patrow))
	       | (false, false) =>
		   (StatObject.bogus_Subst,
		    (VE', rho),
		    OG.PATROW(repeatedIdsError(i, 
				  map ErrorInfo.ID_RID (EqSet.list intdom)),
			      lab, out_pat, Some out_patrow))
	       | (false, true) => 
		   (StatObject.bogus_Subst,
		    (VE', rho),
		    OG.PATROW(repeatedIdsError(i, 
			       (map ErrorInfo.ID_RID (EqSet.list intdom)) @
			       [ErrorInfo.LAB_RID lab]),
			      lab, out_pat, Some out_patrow))
	     end

	| IG.DOTDOTDOT i => (* Flexible record treatment... *) (* rule 40 *)
	    let 
	      val rho = StatObject.emptyFlexRecType()
	    in
	      (StatObject.Id,
	       (Environments.emptyVE, rho),
	       OG.DOTDOTDOT(
(* mikon#1.34 *)
			    addSimTrace (CoreTrace.TYPE(StatObject.mkTypeRecType rho))
(* end mikon#1.34 *)
			    (preOverloadingConv(i, OverloadingInfo.UNRESOLVED (StatObject.mkTypeRecType rho))))
	       )
	    end

    (****** patterns - Definition page 29 ******)

    and elab_pat (C : Environments.Context, pat : IG.pat) :
	(Substitution * (Environments.VarEnv * StatObject.Type) * OG.pat) =
      let
	val _ =
	  if Flags.DEBUG_ELABDEC then
	    pr("elab_pat: ", PPInDecGrammar.layoutPat pat)
	  else ()

	val (S, (VE, ty), pat') = elab_pat'(C, pat)

	val _ =
	  if Flags.DEBUG_ELABDEC then
	    let 
	      val t = PP.NODE{start="{", finish="}", indent=0,
			      children=[Environments.layoutVE VE,
					StatObject.layoutType ty
				       ],
			      childsep=PP.RIGHT "; "
			     }
	    in
	      pr("giving:   ", t)
	    end
	      else ()
      in
	(S, (VE, ty), pat')
      end

    and elab_pat'(C, pat) =
	case pat of

	  (* Atomic pattern *)                             (* rule 42 *)
	  IG.ATPATpat(i, atpat) =>
	    let val (S, (VE,tau), out_atpat) = elab_atpat(C, atpat)
	    in (S, (VE,tau), OG.ATPATpat(okConv i,out_atpat)) end

	  (* Constructed pattern *)
	| IG.CONSpat(i, IG.OP_OPT(longid, withOp), atpat) =>
	    let
	      val (S, (VE,tau'), out_atpat) = elab_atpat(C, atpat)
	    in
	      case Environments.Lookup_longid(C, longid) of

		Some(Environments.LONGCON sigma) =>       (* rule 43 *)
		  let
		    val new = freshType()
		    val arrow = StatObject.mkTypeArrow(tau', new) 
		    val tau1 = StatObject.instance sigma
		    val (S1, i') = Unify(arrow, tau1, i)
		    val longcon = StatObject.mk_longcon longid
		  in
		    (S1 oo S, (S1 onVE VE, S1 on new),
		     OG.CONSpat(
(* mikon#1.34 *)
				addSimTrace (CoreTrace.TYPE tau1)
(* end mikon#1.34 *)
				(addTypeInfo_CON(i', C, true, longid)),
				OG.OP_OPT(R.LONGCON longcon, withOp),
				out_atpat
			       )
		    )
		  end

	      | Some(Environments.LONGEXCON tau) =>       (* rule 44 *)
		  let
		    val arrow = StatObject.mkTypeArrow(tau',StatObject.TypeExn)
		    val (S1, i') = Unify(arrow, tau, i)
		    val longexcon = StatObject.mk_longexcon longid
		  in
		    (S1 oo S,
		     (S1 onVE VE,StatObject.TypeExn),
		     OG.CONSpat(i', OG.OP_OPT(R.LONGEXCON longexcon, withOp),
				    out_atpat
			       )
		    )
		  end

(***KEVIN's idea for expressions as patterns:

	      | Some(Environments.LONGVAR sigma) =>
		  let
		    val new = freshType()
		    val arrow = StatObject.mkTypeArrow(new, tau')
		    val tau = StatObject.instance sigma
		    val (S1, i') = Unify(arrow,tau,i)
		    val longvar = StatObject.mk_longvar longid
		  in
		    (S1 oo S,
		    (S1 onVE VE, S1 on new),
		     OG.CONSpat(i', OG.OP_OPT(R.LONGVAR longvar, withOp),
				    out_atpat
			       )
		    )
		  end

***)

	      | _ => (* Mark the error. *)
		  (StatObject.bogus_Subst,
		   (VE, bogus_Type()),
		   OG.CONSpat(lookupIdError(i, longid),
			      OG.OP_OPT(R.LONGCON StatObject.bogusCon, withOp),
			      out_atpat
			     )
		  )
	    end

	  (* Typed pattern *)                                (* rule 45 *)
	| IG.TYPEDpat(i, pat, ty) =>
	    let
	      val (S, (VE,tau), out_pat) = elab_pat(C, pat)
	      val (tau', out_ty) = elab_ty(C, ty)
	      val (S', i') = Unify(tau, tau', i)
	      val S'' = S' oo S
	    in
	      (S'',
	       (S'' onVE VE, S'' on tau), OG.TYPEDpat(i', out_pat, out_ty))
	    end

	  (* Layered pattern *)                              (* rule 46 *)
	| IG.LAYEREDpat(i, IG.OP_OPT(id, withOp), None, pat) =>
	    let
	      val (S, (VE1, tau), out_pat) = elab_pat(C, pat)
	      val VE2 = Environments.singleVarVE(id, StatObject.Type_in_TypeScheme tau)
	      val intdom = EqSet.intersect (Environments.VEdom VE1)
		                           (Environments.VEdom VE2)
	      val VE3 = Environments.VE_plus_VE(VE1, VE2)
	      val var = StatObject.mk_var id
	    in
	      if EqSet.isEmpty intdom then
		(S, (VE3, tau), OG.LAYEREDpat(okConv i, OG.OP_OPT(var, withOp),
					   None, out_pat
					   )
		 )
	      else
		(S, (VE3, tau),
		 OG.LAYEREDpat(
			       repeatedIdsError(i, map ErrorInfo.ID_RID (EqSet.list intdom)),
			       OG.OP_OPT(var, withOp),
			       None, out_pat))
	    end

	| IG.LAYEREDpat(i, IG.OP_OPT(id, withOp), Some ty, pat) =>
	    let
	      val (S, (VE1, tau), out_pat) = elab_pat(C, pat)
	      val (tau', out_ty) = elab_ty(C, ty)
	      val (S', i') = Unify(tau, tau', i)
	      val S'' = S' oo S
	      val VE2 = Environments.singleVarVE(id, StatObject.Type_in_TypeScheme tau)
	      val intdom = EqSet.intersect (Environments.VEdom VE1)
		                           (Environments.VEdom VE2)
	      val var = StatObject.mk_var id
	      val VE3 = Environments.VE_plus_VE(VE1, VE2)
	    in
	      if EqSet.isEmpty intdom then
		(S'',
		 (S'' onVE VE3, S'' on tau),
		 OG.LAYEREDpat(i', OG.OP_OPT(var, withOp), Some out_ty, out_pat)
		 )
	      else
		(S'', 
		 (S'' onVE VE3, S'' on tau),
		 OG.LAYEREDpat(repeatedIdsError(i, map ErrorInfo.ID_RID (EqSet.list intdom)),
			       OG.OP_OPT(var, withOp), Some out_ty, out_pat)
		 )
	    end

	| IG.UNRES_INFIXpat _ =>
	    Crash.impossible "elab_pat(UNRES_INFIX)"

    (****** types - Definition page 29 ******)

    and elab_ty (C :  Environments.Context, ty : IG.ty) :
	(StatObject.Type * OG.ty) =

	case ty of

	  (* Explicit type variable *)                      (* rule 47 *)
	  IG.TYVARty(i, tyvar) =>
	    (StatObject.mkTypeTyVar(StatObject.mkExplicitTyVar tyvar),
	     OG.TYVARty(okConv i,tyvar))

	  (* Record type *)                                 (* rule 48 *)
	| IG.RECORDty(i, None) =>
	    (StatObject.TypeUnit, OG.RECORDty(okConv i,None))

	  (* Record type *)
	| IG.RECORDty(i, Some tyrow) =>
	    let
	      val (rho, out_tyrow) = elab_tyrow(C, tyrow)
	    in
	      (StatObject.mkTypeRecType rho,
	       OG.RECORDty(okConv i, Some out_tyrow))
	    end

	(* Constructed type *)                              (* rule 49 *)
	| IG.CONty(i, ty_list, longtycon) =>
	    let
	      val res_list    = map (fn ty => elab_ty (C, ty)) ty_list
	      val tau_list    = map #1 res_list
	      val out_ty_list = map #2 res_list
	    in
	      case Environments.Lookup_longtycon(C, longtycon)
		of Some tystr =>
		     let
		       val (typeFcn, _) = Environments.unTyStr tystr
		       val expectedArity = StatObject.arity_TypeFcn typeFcn
		       val actualArity = List.size tau_list
		     in
		       if expectedArity = actualArity then
			 (StatObject.applyTypeFcn(typeFcn, tau_list),
			  OG.CONty(okConv i, out_ty_list, longtycon)
			 )
		       else
			 (bogus_Type(),
			  OG.CONty(errorConv(i, ErrorInfo.WRONG_ARITY{
						  expected=expectedArity,
						  actual=actualArity
						}
					    ),
				   out_ty_list, longtycon
				  )
			 )
		     end

	         | None =>
		       (bogus_Type(),
			OG.CONty(
			  lookupTyConError(i, longtycon),
			  out_ty_list, longtycon
			)
		       )
	    end

	  (* Function type *)                            (* rule 50 *)
	| IG.FNty(i, ty, ty') =>
	    let
	      val (tau , out_ty ) = elab_ty(C, ty )
	      val (tau', out_ty') = elab_ty(C, ty')
	    in
	      (StatObject.mkTypeArrow(tau, tau'),
	       OG.FNty(okConv i, out_ty, out_ty'))
	    end

	  (* Parenthesised type *)                       (* rule 51 *)
	| IG.PARty(i, ty) =>
	    let
	      val (tau, out_ty) = elab_ty(C, ty)
	    in
	      (tau, OG.PARty(okConv i, out_ty))
	    end

    (****** type rows - Definition page 30 ******)

    and elab_tyrow (C :  Environments.Context, tyrow : IG.tyrow) :
	(StatObject.RecType * OG.tyrow) =

	case tyrow of

	  (* Type row *)                                 (* rule 52 *) 
	  IG.TYROW(i, lab, ty, None) =>
	    let
	      val (tau, out_ty) = elab_ty(C, ty)
	      val rho = StatObject.addField (lab,tau) StatObject.emptyRecType
	    in
	      (rho, OG.TYROW(okConv i, lab, out_ty, None))
	    end

	  (* Type row *)
	| IG.TYROW(i, lab, ty, Some tyrow) =>
	    let
	      val (tau, out_ty) = elab_ty(C, ty)
	      val (rho, out_tyrow) = elab_tyrow(C, tyrow)
	    in
	      if (List.member lab (StatObject.sortedLabsOfRecType rho)) then
		(rho,
		 OG.TYROW(repeatedIdsError(i, [ErrorInfo.LAB_RID lab]),
			  lab, out_ty, Some out_tyrow))
	      else
		(StatObject.addField (lab,tau) rho,
		 OG.TYROW(okConv i, lab, out_ty, Some out_tyrow))
	    end
	  
(**** Overloading resolution ****)  

(* mikon#4 *)
datatype dec_or_exp = DEC of OG.dec | EXP of OG.exp

fun resolve (S : Substitution, d_o_e : dec_or_exp): dec_or_exp =

    (* resolves overloading in dec or exp, by applying S on every recorded
       overloaded type variable --- if repeated application of S 
       yields int or real, we record this information in the info
       field; otherwise overloading cannot be resolved
       and error-info is inserted in the info-field. *)
(* end mikon#4 *)

let
  open OG 

  exception NotResolved

  fun res (typ : Type) : OverloadingInfo.info =
    let 
      val typ' = (S on typ)
    in
      if Flags.DEBUG_ELABDEC then
	(pr("res: tv is: ", StatObject.layoutType typ);
	 pr("res:  S on tv yields type: ", StatObject.layoutType typ'))
      else ();
	
      case (StatObject.unTypeTyVar typ') of
      None => 
        (if typ' = StatObject.TypeInt then  OverloadingInfo.RESOLVED_INT
         else if typ' =  StatObject.TypeReal then  OverloadingInfo.RESOLVED_REAL
	 else raise NotResolved
        )
      | Some tv' =>
	if (typ' = typ) then raise NotResolved
        else res typ'    (* Repeat application of S *)
    end
	    
  datatype flexresResult = FLEX_RESOLVED | FLEX_NOTRESOLVED
  fun flexrecres(typ : Type) : flexresResult =
    let
      fun loop typ = 
	let
	  val typ' = S on typ
	in
	  if Flags.DEBUG_FLEXRECORDS then 
	    (pr("flexrecres: typ = ", StatObject.layoutType typ);
	     pr("flexrecres: typ' = ", StatObject.layoutType typ'))
	  else 
	    ();

	  if typ' = typ then typ
	  else loop typ'
	end
    in
      if StatObject.existsRecVarsType (loop typ) then FLEX_NOTRESOLVED
      else FLEX_RESOLVED
    end

  fun resolve_atexp (atexp : atexp) : atexp =
      case atexp of
	  SCONatexp _ => atexp
	| IDENTatexp(i, op_opt) =>
	      (case (GrammarInfo.getPostElabOverloadingInfo i) of 
		   None => atexp
		 | Some (OverloadingInfo.UNRESOLVED typ) =>
		       (IDENTatexp(postOverloadingConv(i, res typ), op_opt)
		       handle NotResolved =>
			   IDENTatexp(overloadingError i, op_opt))
		 | Some _ => Crash.impossible "ElabDec.resolve_atexp"
	      )
	| RECORDatexp(i, None) => atexp
	| RECORDatexp(i, Some exprow) =>
	      RECORDatexp(i, Some (resolve_exprow exprow))
	| LETatexp(i, dec, exp) =>
	      LETatexp(i, resolve_dec dec, resolve_exp exp)
	| PARatexp(i, exp) =>
	      PARatexp(i, resolve_exp exp)
(* mikon#1.8 *)
        | UNDEFatexp(_) => atexp 	
(* end mikon#1.8 *)
	      
  and resolve_exprow (exprow: exprow) : exprow =
      case exprow of 
	  EXPROW(i, l, exp, None) =>
	      EXPROW(i, l, resolve_exp exp, None)
	| EXPROW(i, l, exp, Some exprow) =>
	      EXPROW(i, l, resolve_exp exp, Some (resolve_exprow exprow))
	      
  and resolve_exp (exp: exp) : exp =
      case exp of
	  ATEXPexp(i, atexp) => 
	      ATEXPexp(i, resolve_atexp atexp)
	| APPexp(i, exp, atexp) => 
	      APPexp(i, resolve_exp exp, resolve_atexp atexp)
	| TYPEDexp(i, exp, ty) =>
	      TYPEDexp(i, resolve_exp exp, ty)
(* mikon#1.8 *)
     
        |  COMPARexp(i, exp1, exp2) => COMPARexp(i, resolve_exp exp1, resolve_exp exp2) 
        |  EXIST_QUANTexp(i, match) => EXIST_QUANTexp(i, resolve_match match)
        |  UNIV_QUANTexp(i, match) => UNIV_QUANTexp(i, resolve_match match)
        |  CONVERexp(i, exp) => CONVERexp(i, resolve_exp exp)
	
(* end mikon#1.8 *)
	| HANDLEexp(i, exp, match) =>
	      HANDLEexp(i, resolve_exp exp, resolve_match match)
	| RAISEexp(i, exp) => 
	      RAISEexp(i, resolve_exp exp)
	| FNexp(i, match) =>
	      FNexp(i, resolve_match match)
	| UNRES_INFIXexp _ =>
	      Crash.impossible "resolve_exp(UNRES_INFIX)"

  and resolve_match (match: match) : match =
      case match of 
          MATCH(i, mrule, None) => 
	      MATCH(i, resolve_mrule mrule, None)
        | MATCH(i, mrule, Some match) =>
	      MATCH(i, resolve_mrule mrule, Some (resolve_match match))

  and resolve_mrule (MRULE(i, pat, exp) : mrule) : mrule =
      MRULE(i, resolve_pat pat, resolve_exp exp)
      
  and resolve_dec (dec : dec) : dec =
      case dec of 
	  VALdec(i, valbind) => VALdec(i, resolve_valbind valbind)
	| UNRES_FUNdec _ =>
	      Crash.impossible "resolve_dec(UNRES_FUNdec)"
	| TYPEdec _ => dec
(* mikon#1.8 *)
        | EQTYPEdec _ => dec 	
(* end mikon#1.8 *)
	| DATATYPEdec _ => dec
	| ABSTYPEdec(i, datbind, dec) =>
	      ABSTYPEdec(i, datbind, resolve_dec dec)
	| EXCEPTIONdec _ => dec
(* mikon#6 
        | AXIOMdec(i, ax) => AXIOMdec(i, resolve_ax ax)
   end mikon#6 & commented out by mikon#1.1 *)

        | LOCALdec(i, dec1, dec2) =>
	      LOCALdec(i, resolve_dec dec1, resolve_dec dec2)
        | OPENdec _ => dec
	| SEQdec(i, dec1, dec2) =>
	      SEQdec(i, resolve_dec dec1, resolve_dec dec2)
        | INFIXdec _ => dec
        | INFIXRdec _ => dec
        | NONFIXdec _ => dec
        | EMPTYdec _ => dec

  and resolve_valbind (valbind : valbind) : valbind =
      case valbind of
	  PLAINvalbind(i, pat, exp, None) =>
	      PLAINvalbind(i, pat, resolve_exp exp, None)
        | PLAINvalbind(i, pat, exp, Some valbind) =>
	      PLAINvalbind(i, pat, resolve_exp exp, Some (resolve_valbind valbind))
	| RECvalbind(i, valbind) =>
	      RECvalbind(i, resolve_valbind valbind)
      
(* mikon#6
  and resolve_ax (ax : ax) : ax =
      let val (AX(i, exp)) = ax 
      in AX(i, resolve_exp exp)
      end
   mikon#6 end & commented out by mikon#1.1 *)

  and resolve_atpat (atpat : atpat) : atpat =
    case atpat of
      WILDCARDatpat _ => atpat
    | SCONatpat _ => atpat
    | LONGIDatpat _ => atpat
    | RECORDatpat(i, None) => atpat
    | RECORDatpat(i, Some patrow) =>
	RECORDatpat(i, Some (resolve_patrow patrow))
    | PARatpat(i, pat) =>
	PARatpat(i, resolve_pat pat)

  and resolve_patrow (patrow : patrow): patrow  =
    case patrow of
      DOTDOTDOT(i) => 
	(case (GrammarInfo.getPostElabOverloadingInfo i) of 
	   None => patrow
	 | Some (OverloadingInfo.UNRESOLVED typ) =>
	     (case flexrecres typ of
		FLEX_RESOLVED => 
		  DOTDOTDOT(GrammarInfo.removePostElabOverloadingInfo i)
	      | FLEX_NOTRESOLVED =>
		  DOTDOTDOT(
		      GrammarInfo.addPostElabErrorInfo 
			    i ErrorInfo.FLEX_REC_NOT_RESOLVED))
	 | Some _ => Crash.impossible "ElabDec.resolve_patrow")
    | PATROW(i, lab, pat, None) => 
	PATROW(i, lab, resolve_pat pat, None)
    | PATROW(i, lab, pat, Some patrow) =>
	PATROW(i, lab, resolve_pat pat, Some (resolve_patrow patrow))

  and resolve_pat (pat : pat) : pat =
    case pat of
      ATPATpat(i, atpat) =>
	ATPATpat(i, resolve_atpat atpat)
    | CONSpat(i, longidopt, atpat) =>
	CONSpat(i, longidopt, resolve_atpat atpat)
    | TYPEDpat(i, pat, ty) =>
	TYPEDpat(i, resolve_pat pat, ty)
    | LAYEREDpat(i, idopt, tyopt, pat) =>
	LAYEREDpat(i, idopt, tyopt, resolve_pat pat)
    | UNRES_INFIXpat _ =>
	Crash.impossible "resolve_pat(UNRES_INFIX)"


in
  case d_o_e of 
    DEC(dec) => DEC(resolve_dec dec)
  | EXP(exp) => EXP(resolve_exp exp) 
end (* let *)


(* mikon#1.37 *)
	
fun process_trace (S : Substitution, d_o_e : dec_or_exp): dec_or_exp =

    (* pocesses trace represented by d_o_e, by 
       1. applying S on every trace component stored 
          in the info field of d_o_e or it's subexpressions, 
       2. for every subexpression of d_o_e:
          - computing the list of the free variables of the trace 
            represented by this subexpression, 
          - if the trace should be closed, closing it using 
            the list of the free variables and the stored context *)

let

  open OG 

  type TyVar = Environments.TyVar

  local

    fun pair (f1, f2) (a1, a2) = (f1 a1, f2 a2)
    fun Id x = x

  in

    fun process_info_and_tvs (i, tvs) = 
	case (GrammarInfo.getPostElabTrace i, tvs)
	  of (Some(trace), tvs) => 
	      pair (GrammarInfo.addPostElabTrace i, Id) 
	      (case (Trace.unTraceCoreTrace trace, tvs)
		 of (Some(core_trace), tvs) => 
		     pair (Trace.TraceCOR_in_Trace, Id)
		     (CoreTrace.process_core_trace_and_tvs (core_trace, tvs) S)
		  | (None, _) => Crash.impossible "ElabDec.process_info_and_tvs(unTrace)")
	   | (None, l) => (i, l)

  end

  fun process_info i = process_info_and_tvs (i, [])

  fun process_atexp (atexp : atexp) : atexp * TyVar list =
      case atexp 
	of SCONatexp (i, s) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (SCONatexp(i', s), tvs)
	      end
	| IDENTatexp(i, id) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (IDENTatexp(i', id), tvs)
	      end
	| RECORDatexp(i, None) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (RECORDatexp(i', None), tvs)
	      end
	| RECORDatexp(i, Some exprow) =>
	      let
		  val (proc_exprow, tvs) = process_exprow exprow 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (RECORDatexp(i', Some proc_exprow), tvs')
	      end
	| LETatexp(i, dec, exp) =>
	      let
		  val (proc_dec, tvs1) = process_dec dec
		  val (proc_exp, tvs2) = process_exp exp
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (LETatexp(i', proc_dec, proc_exp), tvs')
	      end
	| PARatexp(i, exp) =>
	      let
		  val (proc_exp, tvs) = process_exp exp 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (PARatexp(i', proc_exp), tvs')
	      end
        | UNDEFatexp(i) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (UNDEFatexp(i'), tvs)
	      end
	      
  and process_exprow (exprow: exprow) : exprow * TyVar list =
      case exprow of 
	  EXPROW(i, l, exp, None) =>
	      let
		  val (proc_exp, tvs) = process_exp exp 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (EXPROW(i', l, proc_exp, None), tvs')
	      end
	| EXPROW(i, l, exp, Some exprow) =>
	      let
		  val (proc_exp, tvs1) = process_exp exp
		  val (proc_exprow, tvs2) = process_exprow exprow
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (EXPROW(i', l, proc_exp, Some(proc_exprow)), tvs')
	      end
	      
  and process_exp (exp: exp) : exp * TyVar list =
      case exp of
	  ATEXPexp(i, atexp) =>
	      let
		  val (proc_atexp, tvs) = process_atexp atexp 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (ATEXPexp(i', proc_atexp), tvs')
	      end
	| APPexp(i, exp, atexp) => 
	      let
		  val (proc_exp, tvs1) = process_exp exp
		  val (proc_atexp, tvs2) = process_atexp atexp
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (APPexp(i', proc_exp, proc_atexp), tvs')
	      end 
	| TYPEDexp(i, exp, ty) =>
	      let
		  val (proc_exp, tvs) = process_exp exp 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (TYPEDexp(i', proc_exp, ty), tvs')
	      end
	|  COMPARexp(i, exp1, exp2) => 
	      let
		  val (proc_exp1, tvs1) = process_exp exp1
		  val (proc_exp2, tvs2) = process_exp exp2
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (COMPARexp(i', proc_exp1, proc_exp2), tvs')
	      end 
        |  EXIST_QUANTexp(i, match) => 
	      let
		  val (proc_match, tvs) = process_match match 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (EXIST_QUANTexp(i', proc_match), tvs')
	      end
        |  UNIV_QUANTexp(i, match) => 
	      let
		  val (proc_match, tvs) = process_match match 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (UNIV_QUANTexp(i', proc_match), tvs')
	      end
        |  CONVERexp(i, exp) => 
	      let
		  val (proc_exp, tvs) = process_exp exp 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (CONVERexp(i', proc_exp), tvs')
	      end
	| HANDLEexp(i, exp, match) =>
	      let
		  val (proc_exp, tvs1) = process_exp exp
		  val (proc_match, tvs2) = process_match match
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (HANDLEexp(i', proc_exp, proc_match), tvs')
	      end
	| RAISEexp(i, exp) => 
	      let
		  val (proc_exp, tvs) = process_exp exp 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (RAISEexp(i', proc_exp), tvs')
	      end 
	| FNexp(i, match) =>
	      let
		  val (proc_match, tvs) = process_match match 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (FNexp(i', proc_match), tvs')
	      end
	| UNRES_INFIXexp _ =>
	      Crash.impossible "process_exp(UNRES_INFIX)"

  and process_match (match: match) : match * TyVar list =
      case match of 
          MATCH(i, mrule, None) => 
	      let
		  val (proc_mrule, tvs) = process_mrule mrule 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (MATCH(i', proc_mrule, None), tvs')
	      end 
        | MATCH(i, mrule, Some match) =>
	      let
		  val (proc_mrule, tvs1) = process_mrule mrule
		  val (proc_match, tvs2) = process_match match
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (MATCH(i', proc_mrule, Some(proc_match)), tvs')
	      end

  and process_mrule (MRULE(i, pat, exp) : mrule) : mrule * TyVar list =
      let
	  val (proc_pat, tvs1) = process_pat pat
	  val (proc_exp, tvs2) = process_exp exp
	  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
      in
	  (MRULE(i', proc_pat, proc_exp), tvs')
      end
      
  and process_dec (dec : dec) : dec * TyVar list =
      case dec of 
	  VALdec(i, valbind) => 
	      let
		  val (proc_valbind, tvs) = process_valbind valbind 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (VALdec(i', proc_valbind), tvs')
	      end
	| UNRES_FUNdec _ =>
	      Crash.impossible "process_dec(UNRES_FUNdec)"
	| TYPEdec(i, typbind) => 
	      let
		  val (proc_typbind, tvs) = process_typbind typbind 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (TYPEdec(i', proc_typbind), tvs')
	      end
        | EQTYPEdec(i, typbind) => 
	      let
		  val (proc_typbind, tvs) = process_typbind typbind 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (EQTYPEdec(i', proc_typbind), tvs')
	      end
	| DATATYPEdec(i, datbind) => 
	      let
		  val (proc_datbind, tvs) = process_datbind datbind
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (DATATYPEdec(i', proc_datbind), tvs')
	      end
	| ABSTYPEdec(i, datbind, dec) =>
	      let
		  val (proc_datbind, tvs1) = process_datbind datbind
		  val (proc_dec, tvs2) = process_dec dec
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (ABSTYPEdec(i', proc_datbind, proc_dec), tvs')
	      end 
	| EXCEPTIONdec(i, exbind) => 
	      let
		  val (proc_exbind, tvs) = process_exbind exbind
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (EXCEPTIONdec(i', proc_exbind), tvs')
	      end
        | LOCALdec(i, dec1, dec2) =>
	      let
		  val (proc_dec1, tvs1) = process_dec dec1
		  val (proc_dec2, tvs2) = process_dec dec2
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (LOCALdec(i', proc_dec1, proc_dec2), tvs')
	      end 
        | OPENdec(i, long_list) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (OPENdec(i', long_list), tvs)
	      end
	| SEQdec(i, dec1, dec2) =>
	      let
		  val (proc_dec1, tvs1) = process_dec dec1
		  val (proc_dec2, tvs2) = process_dec dec2
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (SEQdec(i', proc_dec1, proc_dec2), tvs')
	      end
        | INFIXdec _ => (dec, [])
        | INFIXRdec _ => (dec, [])
        | NONFIXdec _ => (dec, [])
        | EMPTYdec(i) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (EMPTYdec(i'), tvs)
	      end 

  and process_valbind (valbind : valbind) : valbind * TyVar list =
      case valbind of
	  PLAINvalbind(i, pat, exp, None) =>
	      let
		  val (proc_pat, tvs1) = process_pat pat
		  val (proc_exp, tvs2) = process_exp exp
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (PLAINvalbind(i', proc_pat, proc_exp, None), tvs')
	      end 
        | PLAINvalbind(i, pat, exp, Some valbind) =>
	      let
		  val (proc_pat, tvs1) = process_pat pat
		  val (proc_exp, tvs2) = process_exp exp
		  val (proc_valbind, tvs3) = process_valbind valbind 
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2 @ tvs3)
	      in
		  (PLAINvalbind(i', proc_pat, proc_exp, Some(proc_valbind)), tvs')
	      end 
	| RECvalbind(i, valbind) =>
	      let
		  val (proc_valbind, tvs) = process_valbind valbind 
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (RECvalbind(i', proc_valbind), tvs')
	      end 

  and process_typbind (typbind : typbind) : typbind * TyVar list =
      case typbind of
	  TYPBIND(i, tyvar_list, tycon, ty, None) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (TYPBIND(i', tyvar_list, tycon, ty, None), tvs)
	      end
        | TYPBIND(i, tyvar_list, tycon, ty, Some(typbind)) =>
	      let
		  val (proc_typbind, tvs) = process_typbind typbind		  
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (TYPBIND(i', tyvar_list, tycon, ty, Some(proc_typbind)), tvs')
	      end 
	| QUEST_TYPBIND(i, tyvar_list, tycon,  None) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (QUEST_TYPBIND(i', tyvar_list, tycon,  None), tvs)
	      end
	| QUEST_TYPBIND(i, tyvar_list, tycon, Some(typbind)) =>
	      let
		  val (proc_typbind, tvs) = process_typbind typbind		  
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (QUEST_TYPBIND(i', tyvar_list, tycon, Some(proc_typbind)), tvs')
	      end 

  and process_datbind (datbind : datbind) : datbind * TyVar list =
      case datbind of
	  DATBIND(i, tyvar_list, tycon, conbind, None) => 
	      let
		  val (proc_conbind, tvs) = process_conbind conbind
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (DATBIND(i', tyvar_list, tycon, proc_conbind, None), tvs')
	      end
        | DATBIND(i, tyvar_list, tycon, conbind, Some(datbind)) => 
	      let
		  val (proc_conbind, tvs1) = process_conbind conbind
		  val (proc_datbind, tvs2) = process_datbind datbind
		  val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	      in
		  (DATBIND(i', tyvar_list, tycon, proc_conbind, Some(proc_datbind)), tvs')
	      end

  and process_conbind (conbind : conbind) : conbind * TyVar list =
      case conbind of
	  CONBIND(i, con_op_opt, ty_opt, None) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (CONBIND(i', con_op_opt, ty_opt, None), tvs)
	      end
        | CONBIND(i, con_op_opt, ty_opt, Some(conbind)) => 
	      let
		  val (proc_conbind, tvs) = process_conbind conbind
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (CONBIND(i', con_op_opt, ty_opt, Some(proc_conbind)), tvs')
	      end

  and process_exbind (exbind : exbind) : exbind * TyVar list =
      case exbind of
	  EXBIND(i, excon_op_opt, ty_opt, None) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (EXBIND(i', excon_op_opt, ty_opt, None), tvs)
	      end
        | EXBIND(i, excon_op_opt, ty_opt, Some(exbind)) => 
	      let
		  val (proc_exbind, tvs) = process_exbind exbind
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (EXBIND(i', excon_op_opt, ty_opt, Some(proc_exbind)), tvs')
	      end
	| EXEQUAL(i, excon_op_opt, longid_op_opt, None) => 
	      let
		  val (i', tvs) = process_info i
	      in
		  (EXEQUAL(i', excon_op_opt, longid_op_opt, None), tvs)
	      end
        | EXEQUAL(i, excon_op_opt, longid_op_opt, Some(exbind)) => 
	      let
		  val (proc_exbind, tvs) = process_exbind exbind
		  val (i', tvs') = process_info_and_tvs (i, tvs)
	      in
		  (EXEQUAL(i', excon_op_opt, longid_op_opt, Some(proc_exbind)), tvs')
	      end

  and process_atpat (atpat : atpat) : atpat * TyVar list =
      case atpat 
	of WILDCARDatpat(i) => 
	    let
		val (i', tvs) = process_info i
	    in
		(WILDCARDatpat(i'), tvs)
	    end
	 | SCONatpat(i, s) => 
	    let
		val (i', tvs) = process_info i
	    in
		(SCONatpat(i', s), tvs)
	    end
	 | LONGIDatpat(i, longid_op_opt) =>
	    let
		val (i', tvs) = process_info i
	    in
		(LONGIDatpat(i', longid_op_opt), tvs)
	    end
	 | RECORDatpat(i, None) =>
	    let
		val (i', tvs) = process_info i
	    in
		(RECORDatpat(i', None), tvs)
	    end
	 | RECORDatpat(i, Some(patrow)) =>
	    let
		val (proc_patrow, tvs) = process_patrow patrow
		val (i', tvs') = process_info_and_tvs (i, tvs)
	    in
		(RECORDatpat(i', Some(proc_patrow)), tvs')
	    end
	 | PARatpat(i, pat) =>
	    let
		val (proc_pat, tvs) = process_pat pat
		val (i', tvs') = process_info_and_tvs (i, tvs)
	    in
		(PARatpat(i', proc_pat), tvs')
	    end

  and process_patrow (patrow : patrow): patrow * TyVar list =
      case patrow 
	of DOTDOTDOT(i) => 
	    let
		val (i', tvs) = process_info i
	    in
		(DOTDOTDOT(i'), tvs)
	    end
	 | PATROW(i, lab, pat, None) =>
	    let
		val (proc_pat, tvs) = process_pat pat
		val (i', tvs') = process_info_and_tvs (i, tvs)
	    in
		(PATROW(i', lab, proc_pat, None), tvs')
	    end
	 | PATROW(i, lab, pat, Some(patrow)) =>
	    let
		val (proc_pat, tvs1) = process_pat pat
		val (proc_patrow, tvs2) = process_patrow patrow
		val (i', tvs') = process_info_and_tvs (i, tvs1 @ tvs2)
	    in
		(PATROW(i', lab, proc_pat, Some(proc_patrow)), tvs')
	    end

  and process_pat (pat : pat) : pat * TyVar list =
      case pat 
	of ATPATpat(i, atpat) =>
	    let
		val (proc_atpat, tvs) = process_atpat atpat
		val (i', tvs') = process_info_and_tvs (i, tvs)
	    in
		(ATPATpat(i', proc_atpat), tvs')
	    end
	 | CONSpat(i, longidopt, atpat) =>
	    let
		val (proc_atpat, tvs) = process_atpat atpat
		val (i', tvs') = process_info_and_tvs (i, tvs)
	    in
		(CONSpat(i', longidopt, proc_atpat), tvs')
	    end
	 | TYPEDpat(i, pat, ty) =>
	    let
		val (proc_pat, tvs) = process_pat pat
		val (i', tvs') = process_info_and_tvs (i, tvs)
	    in
		(TYPEDpat(i', proc_pat, ty), tvs')
	    end
	 | LAYEREDpat(i, idopt, tyopt, pat) =>
	    let
		val (proc_pat, tvs) = process_pat pat
		val (i', tvs') = process_info_and_tvs (i, tvs)
	    in
		(LAYEREDpat(i', idopt, tyopt, proc_pat), tvs')
	    end
	 | UNRES_INFIXpat _ =>
	    Crash.impossible "process_pat(UNRES_INFIX)"


in
  case d_o_e of 
    DEC(dec) => DEC (#1(process_dec dec))
  | EXP(exp) => EXP (#1(process_exp exp)) 
end (* let *)

(* end mikon#1.37 *)

    (****** Elaborate a declaration and resolve overloading ******)

    val elab_dec : (Environments.Context * IG.dec) ->
      (Environments.Env * OG.dec) =

      fn (C, dec) =>
      (let
	   val (S, E, out_dec) = elab_dec(C, dec)
       in
(* mikon#4 
          let
            val DEC(res_dec) = resolve(S, DEC(out_dec))
          in
            (E, res_dec)
          end
   end mikon#4 & commented out by mikon#1.18 *)

(* mikon#1.37 *)
	   (E, case resolve(S, DEC(out_dec))
		 of DEC(res_dec) => (case (process_trace (S, DEC(res_dec)))
				       of DEC(pro_dec) => pro_dec
					| EXP(_) => Crash.impossible "elab_dec(EXP inner)")
		  | EXP(_) => Crash.impossible "elab_dec(EXP outer)")
(* end mikon#1.37 *)
		 
       end)

(* mikon#4 
    (****** Elaborate axiomspec body an resolve overloading  ******)
    (****** Needed by ElabTopdec *****)   
     fun elab_axiom (i : PreElabGrammarInfo, C : Environments.Context, exp : IG.exp) 
      : (PostElabGrammarInfo * OG.exp) =
 
            let
              val (S1, tau, out_exp) = elab_exp(C, exp)
              val (S2, i') = Unify(tau, StatObject.TypeBool, i)           
            in
              let
                val EXP(res_exp) = resolve(S2 oo S1, EXP(out_exp)) 
              in
                (i', res_exp)
              end
            end
   end mikon#4 & commented out by mikon#1.12 *)

(* mikon#1.38 *)
    (****** Elaborate an exp, resolve overloading, close  ******)
    (******  and process trace contained in the out_exp   ******)
    (******             Needed by ElabTopdec              ******)   
    fun elab_resolve_close_and_process_exp (C : Environments.Context, exp : IG.exp) 
	: (StatObject.Type * OG.exp) =
	let

	    val close_info = changeSchemeTrace (CoreTrace.SCHEME_C(Environments.E_in_C Environments.emptyE))

	    open OG 

	    fun close_exp_trace (exp : exp) =
		case exp 
		  of ATEXPexp(i, atexp) => ATEXPexp(close_info i, atexp)
		   | APPexp(i, exp, atexp) => APPexp(close_info i, exp, atexp)
		   | TYPEDexp(i, exp, ty) => TYPEDexp(close_info i, exp, ty)
		   | COMPARexp(i, exp1, exp2) => COMPARexp(close_info i, exp1, exp2)
		   | EXIST_QUANTexp(i, match) => EXIST_QUANTexp(close_info i, match)
		   | UNIV_QUANTexp(i, match) => UNIV_QUANTexp(close_info i, match)
		   | CONVERexp(i, exp) => CONVERexp(close_info i, exp)
		   | HANDLEexp(i, exp, match) => HANDLEexp(close_info i, exp, match)
		   | RAISEexp(i, exp) => RAISEexp(close_info i, exp)
		   | FNexp(i, match) => FNexp(close_info i, match)
		   | UNRES_INFIXexp _ => Crash.impossible "close_exp_trace(UNRES_INFIX)"

	    val (S1, tau, out_exp) = elab_exp(C, exp)

	in

	    (tau, case resolve(S1, EXP(out_exp))
		    of EXP(res_exp) => (case (process_trace (S1, EXP(close_exp_trace res_exp)))
					  of EXP(pro_exp) => pro_exp
					   | DEC(_) => Crash.impossible "elab_and_resolve_exp(DEC inner)")
		     | DEC(_) => Crash.impossible "elab_and_resolve_exp(DEC outer)")

	end
(* end mikon#1.38 *)

    (* mikon#24 *)
     (***** Check whether the set of unguarded explicit type *****)
     (***** variables of exp is empty. Needed by ElabTopdec. *****)
      fun U_of_exp_empty (exp : IG.exp) =
        Environments.TyVarSet_empty(Environments.Unguarded_exp_TyVars(exp))
(* end mikon#1.24 *)

end; 
