(*
$File: Parsing/Infixing.sml $
$Date: 1992/04/07 14:58:37 $
$Revision: 1.12 $
$Locker:  $
*)

(*$Infixing:
	INFIX_BASIS GRAMMAR_UTILS IDENT DF_INFO
	PPDECGRAMMAR PRETTYPRINT CRASH InfixStack INFIXING
 *)
functor Infixing(structure InfixBasis: INFIX_BASIS
		 structure GrammarUtils: GRAMMAR_UTILS

		 structure Ident: IDENT
		   sharing type GrammarUtils.C.id = InfixBasis.id = Ident.id
		       and type GrammarUtils.C.longid = Ident.longid

		 structure DFInfo: DF_INFO

		 structure PPDecGrammar: PPDECGRAMMAR
		   sharing PPDecGrammar.G = GrammarUtils.C

		 structure PP: PRETTYPRINT
		   sharing type PPDecGrammar.StringTree = PP.StringTree

		 structure Crash: CRASH
		): INFIXING =
  struct
    open GrammarUtils.M GrammarUtils.C

    val i = GrammarUtils.i

   (* Two operator precedence stack modules - one for expressions and
      another for patterns. *)

    structure ExpStack =
      InfixStack(structure InfixBasis = InfixBasis

		 type FullObject = exp
		 type AtomObject = atexp

		 type id = InfixBasis.id
		 val pr_id = Ident.pr_id

		 fun atomToFull atexp = ATEXPexp(i, atexp)
		 fun fullToAtom exp = PARatexp(i, exp)

		 fun pair(exp1, exp2) = GrammarUtils.tupleAtExp [exp1, exp2]

		 fun asId atexp =
		   case atexp
		     of IDENTatexp(_, OP_OPT(longid, withOp)) =>
		          if Ident.unqualified longid andalso not withOp then
			    Some(Ident.decompose0 longid)
			  else
			    None

		      | _ => None

		 fun applyId(id, atexp) =
		   let
		     val identExp =
		       ATEXPexp(
			 i,
			 IDENTatexp(i, OP_OPT(Ident.idToLongId id, false))
		       )
		   in
		     APPexp(i, identExp, atexp)
		   end

		 fun applyObj(exp, atexp) = APPexp(i, exp, atexp)

		 structure Crash = Crash
		)

    structure PatStack =
      InfixStack(structure InfixBasis = InfixBasis

		 type FullObject = pat
		 type AtomObject = atpat

		 type id = InfixBasis.id
		 val pr_id = Ident.pr_id

		 fun atomToFull atpat = ATPATpat(i, atpat)
		 fun fullToAtom pat = PARatpat(i, pat)

		 fun pair(pat1, pat2) = GrammarUtils.tupleAtPat [pat1, pat2]

		 fun asId atpat =
		   case atpat
		     of LONGIDatpat(_, OP_OPT(longid, withOp)) =>
		          if Ident.unqualified longid andalso not withOp then
			    Some(Ident.decompose0 longid)
			  else
			    None

		      | _ => None

		 fun applyId(id, atpat) =
		   CONSpat(i, OP_OPT(Ident.idToLongId id, false), atpat)

		 fun applyObj(pat, atpat) =
		   case pat
		     of ATPATpat(_, LONGIDatpat(_, OP_OPT(longid, withOp))) =>
		          CONSpat(i, OP_OPT(longid, withOp), atpat)

		      | _ =>
			  Crash.unimplemented "error printing(bad pattern)"

		 structure Crash = Crash
		)

    open InfixBasis
    type InfixBasis = Basis

    infix ++
    val op ++ = compose

    fun isInfix(iBas, id) =
      let
	open InfixBasis
      in
	case lookup iBas id
	  of NONFIX => false
	   | INFIX _ => true
	   | INFIXR _ => true
      end

    fun checkNoInfixes iBas atpat =
      case atpat
	of LONGIDatpat(_, OP_OPT(longid, false)) =>
	     let
	       open InfixBasis
	     in
	       case Ident.decompose longid
		 of (nil, id) =>
		      (case lookup iBas id
			 of NONFIX => ()
			  | _ =>
			      Crash.unimplemented("checkNoInfixes: reject("
						  ^ Ident.pr_id id ^ ")"
						 )
		      )

		  | _ => ()
	     end

         | _ => ()

   (* `fun' bindings are a pain in the posterior. The definition (V4, Apdx. B,
      Fig. 20) gives the syntax rules as a footnote (sigh). I've formalised
      them as below. The parser delivers a FUN binding as a sequence (>= 1)
      of atomic patterns, followed by an optional `: <ty>', and `=' and so on.
      Of that general syntax, I permit the following:

	(1) "fun" NonfixID NonfixAP+ (":" Ty)? "=" ...

	(2) "fun" "op" ID NonfixAP+ (":" Ty)? "=" ...

	(3) "fun" "(" NonfixAP InfixID NonfixAP ")" NonfixAP* (":" Ty)? "=" ...

	(4) "fun" NonfixAP InfixID NonfixAP (":" Ty)? "=" ...

      NonfixID is any identifier which isn't an infix. InfixID is an identifier
      with infix status. NonfixAP is any atomic pattern other than an isolated
      identifier which has infix status (that's legal in our parser). ID is
      any identifier (except "=" - look to Topdec_GRAM.src for the gory details
      of what we actually consider an identifier).
    *)

   (* resolveFClauseArgs - takes an infix basis and a list of atomic patterns,
			   and returns the identifier plus its atpat arguments.
			   (We're throwing away the infix status at present,
			   since each branch might have different infix use
			   anyway...) Note that
			   it isn't responsible for resolving the individual
			   atpats. resolveFClauseArgs can fail for an
			   ineligible list of patterns. *)

    fun resolveFClauseArgs(iBas, atpats): id * atpat list =
      let
	fun reject msg =
	  let
	    val tree =
	      PP.NODE{start=msg ^ ": ",
		      finish="",
		      indent=0,
		      children=map PPDecGrammar.layoutAtpat atpats,
		      childsep=PP.RIGHT " "
		     }
	  in
	    Crash.unimplemented("resolveFClauseArgs.reject:\n"
				^ PP.flatten(PP.format(1000, tree))
			       )
	  end

	datatype Category = INFIXED of id | OTHER

	fun categorise atpat: Category =
	  let
	    open InfixBasis
	  in
	    case atpat
	      of LONGIDatpat(_, OP_OPT(_, true)) => OTHER	(* has `op'. *)
	       | LONGIDatpat(_, OP_OPT(longid, false)) =>
		   (case Ident.decompose longid
		      of (nil, id) => if isInfix(iBas, id)
				      then INFIXED id else OTHER
		       | _ => OTHER			(* qualified. *)
		   )

	       | _ => OTHER				(* complex pattern. *)
	  end

	fun mustDecompose longid =
	  case Ident.decompose longid
	    of (nil, id) => id
	     | _ => reject("LongId: " ^ Ident.pr_longid longid)

	fun pair(ap1, ap2) =
	  GrammarUtils.tupleAtPat [ATPATpat(i, ap1), ATPATpat(i, ap2)]
      in
	case map categorise atpats
	  of [OTHER, INFIXED id, OTHER] =>	(* SUCCESS: matches (4). *)
	       (case atpats
		  of [ap1, _, ap2] => (id, [pair(ap1, ap2)])

		   | _ => Crash.impossible "resolveClauseArgs"
	       )

	   | OTHER :: _ =>			(* Try for case (1)/(2)/(3): *)
	       (case atpats
		  of PARatpat(_, UNRES_INFIXpat(_, [ap1, ap2, ap3])) :: rest =>
						(* Try for case (3)... *)
		       (case categorise ap2
			  of INFIXED id =>	(* SUCCESS: matches (3). *)
			       (id, pair(ap1, ap3) :: rest)

			   | OTHER =>		(* `fun (<ap1> <junk> <ap2>)' *)
			       reject "Bad infixed function identifier"
		       )

		   | fst :: snd :: rest =>	(* Try for case (1)/(2)... *)
		       (case fst
			  of LONGIDatpat(_, OP_OPT(longid, withOp)) =>
			       (mustDecompose longid, snd :: rest)
						(* `longid' can't be an infix
						   because it matches OTHER. *)

			   | _ =>		(* `fun <junk> <junk> ...' *)
			       reject "Expecting function declaration"
		       )

		   | _ =>			(* `fun <ap> = ...' *)
		       reject "Expecting function arguments"
	      )

	   | _ =>				(* `fun +' or something. *)
	       reject "Expecting function identifier or infix pattern"
      end

   (* resolveFClause - turn an FClause into an id plus list of argument
		       lists, RHS's and type constraints. All the id's must
		       agree, of course. It's here that we also run through
		       the arguments, resolving them as well, and resolve the
		       RHS exp. *)

    and resolveFClause(iBas, fclause)
        : id * (atpat list * exp * ty Option) list =
      case fclause
	of FCLAUSE(_, atpats, ty_opt, exp, fclause_opt) =>
	     let
	       val (id, args) = resolveFClauseArgs(iBas, atpats)
	       val _ = map (checkNoInfixes iBas) args
	       val args' = map (fn a => resolveAtpat(iBas, a)) args
	       val exp' = resolveExp(iBas, exp)
	     in
	       case fclause_opt
		 of Some fclause' =>
		      let
			val (id', rest) = resolveFClause(iBas, fclause')
		      in
			if id = id' then
			  (id, (args', exp', ty_opt) :: rest)
			else
			  Crash.unimplemented "resolveFClause: ID mismatch"
		      end

		  | None =>
		      (id, [(args', exp', ty_opt)])
	     end


   (* resolveFValBind - turn the form `FUN <FValBind>' into a dec. There's
			an enclosing `VAL REC', and then each FValBind
			gives rise to a single fn, whose branches are built from
			FClauses (which must have identical identifiers). *)

    and resolveFValBind(iBas, fvalbind) =
      let
	fun probableTupleAtExp atexps =
	  case atexps
	    of [a] => a
	     | _ => GrammarUtils.tupleAtExp(map GrammarUtils.expOfAtexp atexps)

	fun probableTuplePat atpats =
	  GrammarUtils.patOfAtpat(
	    case atpats
	      of [a] => a
	       | _ =>
		   GrammarUtils.tupleAtPat(map GrammarUtils.patOfAtpat atpats)
	  )

	fun fvalbindToValbind fvalbind =
	  case fvalbind
	    of FVALBIND(_, fclause, _) =>
	      let
		val (id, rhsList: (atpat list * exp * ty Option) list) = 
		  resolveFClause(iBas, fclause)

		val numArgs =
		  case rhsList
		    of (atpats, _, _) :: _ => List.size atpats
		     | _ => Crash.impossible "Infixing.fvalbindToValbind"

		fun inventVars 0 = nil
		  | inventVars n = GrammarUtils.inventId() :: inventVars(n - 1)

		val vars = inventVars numArgs

		val varTuple =
		  probableTupleAtExp(map GrammarUtils.atexpOfIdent vars)

		fun mkMatch((atpats, exp, ty_opt) :: rest) =
		      let
			val exp' =
			  case ty_opt
			    of Some ty => TYPEDexp(i, exp, ty)
			     | None => exp
		      in
			Some(MATCH(i,
				   MRULE(i, probableTuplePat atpats, exp'),
				   mkMatch rest
				  )
			    )
		      end
		  | mkMatch nil = None

		val innerApp =
		  case mkMatch rhsList
		    of Some m => APPexp(i, FNexp(i, m), varTuple)
		     | None => Crash.impossible "fvalbindToValbind(innerApp)"

		fun curry id exp =
		  FNexp(i,
			MATCH(i,
			      MRULE(i, GrammarUtils.patOfIdent(id, false),
				    exp
				   ),
			      None
			     )
		       )

		val curriedFn =
		  List.foldL curry innerApp (rev vars)
	      in
		PLAINvalbind(i,
			     GrammarUtils.patOfIdent(id, isInfix(iBas, id)),
			     curriedFn,
			     None
			    )
	      end

	fun resolveAll fvalbind =
	  case fvalbind
	    of FVALBIND(_, _, rest) =>
	      case fvalbindToValbind fvalbind
		of PLAINvalbind(i, id, exp, None) =>
		     PLAINvalbind(i, id, exp,
				  (case rest
				     of Some fvalbind' =>
				          Some(resolveAll fvalbind')
				      | None => None
				  )
				 )

		| _ =>
		    Crash.impossible "resolveFValBind.resolveAll"
      in
	VALdec(i, RECvalbind(i, resolveAll fvalbind))
      end

   (* resolveTopdec - walk over a topdec, resolving infix patterns, expressions
      and `FUN'-bindings. Infix/nonfix declarations are analysed and scoped as
      well. The result of resolveInfixes contains the new infix basis built
      from any infix/nonfix declarations at this scope. Note: the iBas argument
      to each resolveXXX function is the entire basis, but the iBas result
      returned by any resolveXXX function contains the fixity declarations
      established by that phrase only. Some resolveXXX functions don't make
      any alterations to the infix basis - in this case, we only return the
      re-written phrase. *)

    and resolveTopdec(iBas, topdec): Basis * topdec =
      case topdec
	of STRtopdec(i, strdec) =>
	     let
	       val (iBas', strdec') = resolveStrdec(iBas, strdec)
	     in
	       (iBas', STRtopdec(i, strdec'))
	     end

  (* robmar#4 *)
  (*
	 | SIGtopdec(i, sigdec) =>
	     (emptyB, SIGtopdec(i, sigdec))
  *)

         | SIGtopdec(i, sigdec) =>
             (emptyB, SIGtopdec(i, resolveSigdec(iBas, sigdec)))

         | FUNtopdec(i, fundec) =>
	     (emptyB, FUNtopdec(i, resolveFundec(iBas, fundec)))

    and resolveStrdec(iBas, strdec) =
      case strdec
	of DECstrdec(i, dec) =>
	     let
	       val (iBas', dec') = resolveDec(iBas, dec)
	     in
	       (iBas', DECstrdec(i, dec'))
	     end
(* alx#1.29*)
         | AXIOMstrdec(i, ax) =>
	     (emptyB, AXIOMstrdec(i, resolveAX(iBas, ax)))
(* alx#1.29#end*)

	 | STRUCTUREstrdec(i, strbind) =>
	     (emptyB, STRUCTUREstrdec(i, resolveStrbind(iBas, strbind)))

	 | LOCALstrdec(i, strdec1, strdec2) =>
	     let
	       val (iBas', strdec1') = resolveStrdec(iBas, strdec1)
	       val (iBas'', strdec2') = resolveStrdec(iBas ++ iBas', strdec2)
	     in
	       (iBas'', LOCALstrdec(i, strdec1', strdec2'))
	     end

	 | EMPTYstrdec i =>
	     (emptyB, EMPTYstrdec i)

	 | SEQstrdec(i, strdec1, strdec2) =>
	     let
	       val (iBas', strdec1') = resolveStrdec(iBas, strdec1)
	       val (iBas'', strdec2') = resolveStrdec(iBas ++ iBas', strdec2)
	     in
	       (iBas' ++ iBas'', SEQstrdec(i, strdec1', strdec2'))
	     end

(* alx#1.29*)
    and resolveAX(iBas, ax) =
          (case ax of
                AXIOMax( i, AXIOM_EXPaxexp( i1, expr), Some ax' ) => 
                    AXIOMax( i, 
                          AXIOM_EXPaxexp( i1,
                            resolveExp(iBas, expr)), 
                              Some(resolveAX(iBas, ax'))) |
                 AXIOMax( i, AXIOM_EXPaxexp( i1, expr), None ) =>
                    AXIOMax( i, 
                          AXIOM_EXPaxexp( i1,
                            resolveExp(iBas, expr)), 
                              None))
(* alx#1.29#end*)

    (* robmar#4 *)
    and resolveSigdec(iBas, sigdec) =
      case sigdec
        of SIGNATUREsigdec(i, sigbind) => 
             SIGNATUREsigdec(i, resolveSigbind(iBas, sigbind))
         | EMPTYsigdec(i) => EMPTYsigdec i
         | SEQsigdec(i, sigdec1, sigdec2) =>
             SEQsigdec(i, resolveSigdec(iBas, sigdec1),
                          resolveSigdec(iBas, sigdec2)
                      )    

    and resolveSigbind(iBas, sigbind) =
      case sigbind
        of SIGBIND(i, sigid, sigexp, sigbind_opt) =>
             SIGBIND(i, sigid, resolveSigexp(iBas, sigexp),
		     case sigbind_opt
		       of Some sigbind => Some(resolveSigbind(iBas, sigbind))
			| None => None
                    )
(*alx#1.28*)                              
    and resolveSigexp (iBas, psigexp) =
      case psigexp
        of PRINCIPpsigexp( i, SIGsigexp(i1, spec) )=>
             PRINCIPpsigexp( i, SIGsigexp(i1, resolveSpec(iBas, spec)))
         | PRINCIPpsigexp( i, SIGIDsigexp(i1, sigid)) =>
             PRINCIPpsigexp( i, SIGIDsigexp(i1, sigid) )
(*alx#1.28#end*)
         
    and resolveSpec(iBas, spec) =
      case spec
        of
(*alx#1.18*)
(* this is ruled out because of AXIOM1spec that is not present anymore
	  AXIOM1spec(i, exp) =>
             AXIOM1spec(i, resolveExp(iBas, exp))
         | SEQspec(i, spec1, spec2) =>
             SEQspec(i, resolveSpec(iBas, spec1),
                        resolveSpec(iBas, spec2)
                      )    
*)
          SEQspec(i, spec1, spec2) =>
             SEQspec(i, resolveSpec(iBas, spec1),
                        resolveSpec(iBas, spec2)
                      )    
(*alx#1.18#end*)
         | LOCALspec(i, spec1, spec2) =>
             LOCALspec(i, resolveSpec(iBas, spec1),
                          resolveSpec(iBas, spec2)
                      )    
(* alx#1.30*)
         | AXIOMspec(i, AxDesc) => AXIOMspec(i, resolveAxDesc(iBas, AxDesc))
(* alx#1.30#end*)
(* alx#1.32*)
         | STRUCTUREspec(i, StrDesc ) => STRUCTUREspec(i, resolveStrDesc(iBas, StrDesc))
(* alx#1.32#end*)
         | spec' => spec'
       
    (* robmar#4 koniec *)
(* alx#1.30*) 

(* alx#1.32*)
    and  resolveStrDesc( iBas, StrDesc ) =
      (case StrDesc of
	STRDESC( i, StrId, SIGsigexp(i1, Spec), None ) => 
          STRDESC( i, StrId, 
            SIGsigexp(i1, resolveSpec(iBas, Spec)), 
            None)
      | STRDESC( i, StrId, SIGsigexp(i1, Spec), (Some StrDesc1) ) => 
          STRDESC( i, StrId, 
             SIGsigexp(i1, resolveSpec(iBas, Spec)), 
             Some( resolveStrDesc(iBas, StrDesc1)))
      | StrDesc' => StrDesc')
(* alx#1.32#end*)


    and  resolveAxDesc(iBas, AxDesc) =
      (case AxDesc
        of AXDESC(i, specexp, Some( Axdesc1 )) =>
             AXDESC(i, resolveSpecExp(iBas, specexp ), 
                       Some( resolveAxDesc(iBas, Axdesc1 )))
         | AXDESC(i, specexp, None ) =>
             AXDESC(i, resolveSpecExp(iBas, specexp ), 
                       None))

    and resolveSpecExp(iBas, specexp) =
       (case specexp
         of SPECEXP(i, strdec, axexp ) =>
	     let
	       val (iBas', strdec') = resolveStrdec(iBas, strdec)
	     in
               SPECEXP(i, strdec', resolveAxExp(iBas ++ iBas', axexp ))
	     end)

    and resolveAxExp(iBas, axexp) =
       (case axexp
         of AXIOM_EXPaxexp(i, exp_ ) =>
              AXIOM_EXPaxexp(i, resolveExp(iBas, exp_ )))
(* alx#1.30#end*)

    and resolveFundec(iBas, fundec) =
      case fundec
	of FUNCTORfundec(i, funbind) =>
	     FUNCTORfundec(i, resolveFunbind(iBas, funbind))

	 | EMPTYfundec i =>
	     EMPTYfundec i

	 | SEQfundec(i, fundec1, fundec2) =>
	     SEQfundec(i, resolveFundec(iBas, fundec1),
		       	  resolveFundec(iBas, fundec2)
		      )

(* robmar#4 : dodalem resolveSigexp *)
(* alx#1.27*)
    and resolveFunbind(iBas, funbind) =
      case funbind
	of FUNBINDfunbind(i, funid, strid, sigexp, sigexp1, strexp, funbind_opt) =>
	     FUNBINDfunbind(i, funid, strid, resolveSigexp(iBas,sigexp), 
                     resolveSigexp(iBas, sigexp1),
		     resolveStrexp(iBas, strexp),
		     case funbind_opt
		       of Some funbind => Some(resolveFunbind(iBas, funbind))
			| None => None
		    ) |
            UNDEFfunbind(i, funid, strid, sigexp, sigexp1, funbind_opt) =>
	     UNDEFfunbind(i, funid, strid, resolveSigexp(iBas,sigexp), 
                     resolveSigexp(iBas, sigexp1),
		     case funbind_opt
		       of Some funbind => Some(resolveFunbind(iBas, funbind))
			| None => None
		    )
(* alx#1.27#end*)
(* robmar#4 : dodalem resolveSigexp *)
(* alx#1.28*)
    and resolveStrbind(iBas, strbind) =
      case strbind
	of STRBIND(i, sglstrbind, strbind_opt) =>
	     STRBIND(i, resolveSglstrbind(iBas, sglstrbind), 
		     case strbind_opt
		       of Some strbind => Some(resolveStrbind(iBas, strbind))
			| None => None
		    )
    and resolveSglstrbind(iBas, sglstrbind) =
      case sglstrbind
	of 
          SINGLEsglstrbind( i, strid, psigexp, strexp ) =>
            SINGLEsglstrbind( i, strid, 
                      resolveSigexp( iBas, psigexp ), 
                      resolveStrexp( iBas, strexp )) |
          UNDEFsglstrbind( i, strid, psigexp ) =>
            UNDEFsglstrbind( i, strid, 
                      resolveSigexp( iBas, psigexp )) |
          UNGUARDsglstrbind( i, strid, strexp ) =>
            UNGUARDsglstrbind( i, strid, 
                      resolveStrexp( iBas, strexp ))
(* alx#1.28#end*)


    and resolveStrexp(iBas, strexp) =
      case strexp
	of STRUCTstrexp(i, strdec) =>
	     let
	       val (_, strdec') = resolveStrdec(iBas, strdec)
	     in
	       STRUCTstrexp(i, strdec')
	     end

	 | LONGSTRIDstrexp(i, longstrid) =>
	     LONGSTRIDstrexp(i, longstrid)

	 | APPstrexp(i, funid, strexp) =>
	     APPstrexp(i, funid, resolveStrexp(iBas, strexp))

	 | LETstrexp(i, strdec, strexp) =>
	     let
	       val (iBas', strdec') = resolveStrdec(iBas, strdec)
	     in
	       LETstrexp(i, strdec', resolveStrexp(iBas ++ iBas', strexp))
	     end


   (* Core level *)

    and resolveDec(iBas, dec): Basis * dec =
      case dec
	of VALdec(i, valbind) =>
	     (emptyB, VALdec(i, resolveValbind(iBas, valbind)))

	 | UNRES_FUNdec(i, fvalbind) =>
	     (emptyB, resolveFValBind(iBas, fvalbind))

	 | TYPEdec(i, typbind) =>
	     (emptyB, TYPEdec(i, typbind))
				(* typbinds don't need resolving -
				   they don't know or care about infixing. *)

	 | DATATYPEdec(i, datbind) =>
	     (emptyB, DATATYPEdec(i, resolveDatbind(iBas, datbind)))

	 | ABSTYPEdec(i, datbind, dec) =>
	     let
	       val datbind' = resolveDatbind(iBas, datbind)
	       val (iBas', dec') = resolveDec(iBas, dec)
	     in
	       (iBas', ABSTYPEdec(i, datbind', dec'))
	     end

(* alx#1.9 *)
	 | EQTYPEdec(i, typbind) =>
	     (emptyB, EQTYPEdec(i, typbind))
				(* typbinds don't need resolving -
				   they don't know or care about infixing. *)
(* alx#1.9#end *)

	 | EXCEPTIONdec(i, exbind) =>
	     (emptyB, EXCEPTIONdec(i, resolveExbind(iBas, exbind)))
(* alx#1 
         | AXIOMdec( i, axio ) =>
             (emptyB, AXIOMdec(i, resolveax( iBas, axio )))
 alx#1 koniec alx#1.3 commented out*)

	 | LOCALdec(i, dec1, dec2) =>
	     let
	       val (iBas', dec1') = resolveDec(iBas, dec1)
	       val (iBas'', dec2') = resolveDec(iBas ++ iBas', dec2)
	     in
	       (iBas'', LOCALdec(i, dec1', dec2'))
	     end

	 | OPENdec(i, list) =>
	     (emptyB, OPENdec(i, list))

	 | SEQdec(i, dec1, dec2) =>
	     let
	       val (iBas', dec1') = resolveDec(iBas, dec1)
	       val (iBas'', dec2') = resolveDec(iBas ++ iBas', dec2)
	     in
	       (iBas' ++ iBas'', SEQdec(i, dec1', dec2'))
	     end

	 | INFIXdec(i, int_opt, ids) =>
	     let
	       val newBas =
		 new(ids, INFIX(case int_opt of Some n => n | None => 0))
	     in
	       (newBas, INFIXdec(i, int_opt, ids))
	     end

	 | INFIXRdec(i, int_opt, ids) =>
	     let
	       val newBas =
		 new(ids, INFIXR(case int_opt of Some n => n | None => 0))
	     in
	       (newBas, INFIXRdec(i, int_opt, ids))
	     end

	 | NONFIXdec(i, ids) =>
	     let
	       val newBas = new(ids, NONFIX)
	     in
	       (newBas, NONFIXdec(i, ids))
	     end

	 | EMPTYdec i =>
	     (emptyB, EMPTYdec i)
(* alx#1 
    and resolveax( iBas, axio ) =
     ( case axio
        of AX( i, expr ) => AX( i, resolveExp( iBas, expr ) ) )
 alx#1 alx#1.3 commented out*)
 
    and resolveValbind(iBas, valbind) =
      case valbind
	of PLAINvalbind(i, pat, exp, valbind_opt) =>
	     PLAINvalbind(i, resolvePat(iBas, pat),
			     resolveExp(iBas, exp),
			     case valbind_opt
			       of Some valbind =>
				    Some(resolveValbind(iBas, valbind))
				| None => None
			 )

	 | RECvalbind(i, valbind) =>
	     RECvalbind(i, resolveValbind(iBas, valbind))

   (* resolveDatbind - we don't actually need to resolve it at all - it's
      unambiguous - but we are required to check occurrences of `op' against
      infixed operators. *)

    and resolveDatbind(iBas, datbind) =
      datbind				(* MEMO: incomplete (no checks yet) *)

   (* resolveExbind - same treatment as resolveDatbind. *)

    and resolveExbind(iBas, exbind) =
      exbind				(* MEMO: incomplete (no checks yet) *)

    and resolveExp(iBas, exp) =
      case exp
	of ATEXPexp(i, atexp) =>
	     ATEXPexp(i, resolveAtexp(iBas, atexp))

	 | APPexp(i,  exp, atexp) =>
	     APPexp(i, resolveExp(iBas, exp), resolveAtexp(iBas, atexp))

	 | TYPEDexp(i, exp_, ty) =>
	     TYPEDexp(i, resolveExp(iBas, exp_), ty)

(* alx#1.8 *)
	 | COMPARexp( i, exp1,  exp2) =>
        COMPARexp( i, resolveExp(iBas, exp1),  resolveExp(iBas, exp2))

    | EXIST_QUANTexp( i, match ) =>
        EXIST_QUANTexp( i, resolveMatch(iBas, match) )

    | UNIV_QUANTexp( i, match ) =>
        UNIV_QUANTexp( i, resolveMatch(iBas, match) )

	 | CONVERexp( i, exp ) =>
        CONVERexp( i, resolveExp(iBas, exp) )
(* alx#1.8#end *)

	 | HANDLEexp(i, exp, match) =>
	     HANDLEexp(i, resolveExp(iBas, exp), resolveMatch(iBas, match))

	 | RAISEexp(i, exp) =>
	     RAISEexp(i, resolveExp(iBas, exp))

	 | FNexp(i, match) =>
	     FNexp(i, resolveMatch(iBas, match))

	 | UNRES_INFIXexp(i, atexps) =>
	     ExpStack.resolveInfix(iBas,
				   map (fn a => resolveAtexp(iBas, a))
				       atexps
				  )

    and resolveMatch(iBas, match) =
      case match
	of MATCH(i, mrule, match_opt) =>
	     MATCH(i, resolveMrule(iBas, mrule),
		      case match_opt
			of Some match => Some(resolveMatch(iBas, match))
			 | None => None
		  )

    and resolveMrule(iBas, mrule) =
      case mrule
	of MRULE(i, pat, exp) =>
	     MRULE(i, resolvePat(iBas, pat), resolveExp(iBas, exp))

    and resolvePat(iBas, pat) =
      case pat
	of ATPATpat(i, atpat) =>
	     ATPATpat(i, resolveAtpat(iBas, atpat))

	 | CONSpat(i, longidOp_opt, atpat) =>
	     CONSpat(i, longidOp_opt, resolveAtpat(iBas, atpat))

	 | TYPEDpat(i, pat, ty) =>
	     TYPEDpat(i, resolvePat(iBas, pat), ty)

	 | LAYEREDpat(i, idOp_opt, ty_opt, pat) =>
	     LAYEREDpat(i, idOp_opt, ty_opt, resolvePat(iBas, pat))

	 | UNRES_INFIXpat(i, atpats) =>
	     PatStack.resolveInfix(iBas,
				   map (fn a => resolveAtpat(iBas, a))
			     	       atpats
				  )

    and resolveAtexp(iBas, atexp) =
      case atexp
	of SCONatexp _ => atexp
	 | IDENTatexp _ => atexp

	 | RECORDatexp(i, exprow_opt) =>
	     RECORDatexp(i,
			 case exprow_opt
			   of Some exprow => Some(resolveExprow(iBas, exprow))
			    | None => None
			)

	 | LETatexp(i, dec, exp) =>
	     let
	       val (iBas', dec') = resolveDec(iBas, dec)
	     in
	       LETatexp(i, dec', resolveExp(iBas ++ iBas', exp))
	     end

	 | PARatexp(i, exp) =>
	     PARatexp(i, resolveExp(iBas, exp))

(* alx#1.8 *)
    | UNDEFatexp _ => atexp
(* alx#1.8#end *)

    and resolveExprow(iBas, exprow) =
      case exprow
	of EXPROW(i, lab, exp, exprow_opt) =>
	     EXPROW(i, lab, resolveExp(iBas, exp),
		    case exprow_opt
		      of Some exprow => Some(resolveExprow(iBas, exprow))
		       | None => None
		   )

    and resolveAtpat(iBas, atpat) =
      case atpat
	of WILDCARDatpat _ => atpat
	 | SCONatpat _ => atpat
         | LONGIDatpat _ => atpat

	 | RECORDatpat(i, patrow_opt) =>
	     RECORDatpat(i,
			 case patrow_opt
			   of Some patrow => Some(resolvePatrow(iBas, patrow))
			    | None => None
			)

	 | PARatpat(i, pat) =>
	     PARatpat(i, resolvePat(iBas, pat))

    and resolvePatrow(iBas, patrow) =
      case patrow
	of DOTDOTDOT _ =>
	     patrow

	 | PATROW(i, lab, pat, patrow_opt) =>
	     PATROW(i, lab, resolvePat(iBas, pat),
		    case patrow_opt
		      of Some patrow => Some(resolvePatrow(iBas, patrow))
		       | None => None
		   )
  end;
