(* Error information *)

(*
$File: Common/ErrorInfo.sml $
$Date: 1993/03/05 14:38:28 $
$Revision: 1.26 $
$Locker: birkedal $
*)

(*$ErrorInfo:
	STATOBJECT_PROP EXCON TYCON SIGID STRID FUNID REPORT
	ERROR_INFO
 *)

functor ErrorInfo(structure StatObjectProp: STATOBJECT_PROP
		  structure Lab:   LAB
		  structure Con:   CON
		  structure Excon: EXCON
		  structure TyCon: TYCON
		  structure SigId: SIGID
		  structure StrId: STRID
		  structure FunId: FUNID
		  structure Report: REPORT
		 ): ERROR_INFO =
  struct
    type Type = StatObjectProp.Type
     and TyVar = StatObjectProp.TyVar
     and TyName = StatObjectProp.TyName
     and TypeFcn = StatObjectProp.TypeFcn
     and id = StatObjectProp.id
     and longid = StatObjectProp.longid
     and tycon = TyCon.tycon
     and longtycon = TyCon.longtycon
     and lab = Lab.lab
     and con = Con.con
     and excon = Excon.excon
     and sigid = SigId.sigid
     and strid = StrId.strid
     and longstrid = StrId.longstrid
     and funid = FunId.funid

    datatype Tail = TYCON of tycon	(* Tail end for Covers errors. *)
      		  | STRID of strid

    datatype RepeatedId = ID_RID of id      (* Repeated identifier, syntax *)
			| LAB_RID of lab    (* errors *)
			| TYCON_RID of tycon
			| EXCON_RID of excon
			| CON_RID of con
			| TYVAR_RID of TyVar
			| STRID_RID of strid
			| SIGID_RID of sigid
			| FUNID_RID of funid

    datatype info =
     (* Core errors: *)
	UNIFICATION of Type * Type
      | LOOKUP_LONGID of longid
      | LOOKUP_LONGTYCON of longtycon
      | NOTCONSTYPE of Type
      | QUALIFIED_ID of longid
      | FREE_TYVARS of TyVar list
      | WRONG_ARITY of {expected: int, actual: int}
      | NOTRESOLVED
      | NOTIMPERATIVE of Type
      | FLEX_REC_NOT_RESOLVED 
      | REPEATED_IDS of RepeatedId list
      | TYVARS_NOT_IN_TYVARSEQ of TyVar list
(* mju#1.6 *)
      | SHOULD_ADMIT_EQ of TypeFcn list
      | LOCAL_TYNAMES
(* endmju#1.6 *)

     (* General module errors: *)
      | LOOKUP_SIGID of sigid
      | LOOKUP_LONGSTRID of longstrid
      | LOOKUP_FUNID of funid
      | ASSEMBLY_COVER of strid list * Tail
      | NOTTYPEEXPLICIT
      | NOTWELLFORMEDSIG
      | NOTEQPRINCIPAL of longtycon list
      | EXDESC_SIDECONDITION
(* mju#1.6 *)
      | AXEXP_SHOULD_BE_BOOL
      | UNGUARD_EXPLICIT_TV_IN_AXEXP
(* endmju#1.6 *)

     (* Signature matching errors: *)
      | S_RIGIDSTRCLASH of longstrid Option
      | MISSINGSTR  of longstrid
      | MISSINGTYPE of longtycon
      | S_CONFLICTINGARITY of longtycon * (TyName * TypeFcn)
      | CONFLICTINGEQUALITY of longtycon * (TyName * TypeFcn)
      | MISSINGVAR of strid list * id
      | MISSINGEXC of strid list * excon
      | S_RIGIDTYCLASH of longtycon
      | S_CONFLICTING_DOMCE of longtycon
      | NOTYENRICHMENT of strid list * id
      | EXCNOTEQUAL of strid list * excon * (Type * Type)

     (* Module unification errors: *)
      | U_RIGIDSTRCLASH of longstrid * longstrid
      | CYCLE of longstrid
      | U_RIGIDTYCLASH of longtycon * longtycon
      | TYPESTRILLFORMEDNESS of longtycon * longtycon
      | U_CONFLICTING_DOMCE of longtycon * longtycon
      | U_CONFLICTINGARITY of longtycon * longtycon
      | RIGIDTYFUNEQERROR of longtycon * longtycon

    type Report = Report.Report
    val line = Report.line
    infix //
    val op // = Report.//

    fun prStrIds strids =
      case List.foldR
	     (fn strid => fn str =>
		case str
		  of "" => StrId.pr_StrId strid
		   | _ => StrId.pr_StrId strid ^ "." ^ str
	     ) "" strids
	of "" => ""
	 | x => x ^ "."

   (* A lot of the module unification errors carry the same argument types: *)

    fun pr_UnifyArgs(longtycon1, longtycon2) =
      TyCon.pr_LongTyCon longtycon1 ^ ", " ^ TyCon.pr_LongTyCon longtycon2

    fun reportInfo info =
      case info
	of UNIFICATION(ty1, ty2) =>
	     let
	       val names = StatObjectProp.newTVNames()
	       val pr = StatObjectProp.pr_TypePRETTY names
	     in
	          line "Type mismatch,"
	       // line("   expecting: " ^ pr ty1)
	       // line("   found:     " ^ pr ty2)
	     end

	 | LOOKUP_LONGID longid =>
	     line("Unbound identifier: " ^ StatObjectProp.pr_longid longid)

	 | LOOKUP_LONGTYCON longtycon =>
	     line("Unbound type constructor: " ^ TyCon.pr_LongTyCon longtycon)

	 | NOTCONSTYPE ty =>
	     line("Not a constructed type: " ^ StatObjectProp.pr_Type ty)

	 | QUALIFIED_ID longid =>
	     line("Qualified identifier not allowed: "
		  ^ StatObjectProp.pr_longid longid
		 )

	 | FREE_TYVARS tvs =>
	     line("Free type variables not allowed: "
		  ^ List.foldR
		      (fn tv => fn str => StatObjectProp.pr_TyVar tv ^ " " ^ str)
		      "" tvs
		 )

	 | WRONG_ARITY{expected, actual} =>
	     line("Wrong arity (expected " ^ Int.string expected
		  ^ ", actual " ^ Int.string actual ^ ")"
		 )

	 | NOTRESOLVED =>
             line "Overloading not resolved"

	 | NOTIMPERATIVE tau =>
	     line("Illegal type in exception binding ("
		  ^ StatObjectProp.pr_Type tau ^" is not imperative)"
		  )

         | FLEX_REC_NOT_RESOLVED =>
	     line "Overloading not resolved in flex record"

         | REPEATED_IDS ids =>
	     let 
	       fun pr_repeatedId rid =
		 case rid of
		   ID_RID id => (StatObjectProp.pr_id id) (* ^ "ID_RID" *)
		 | LAB_RID lab => (Lab.pr_Lab lab) (* ^ "LAB_RID" *)
		 | TYCON_RID tycon => (TyCon.pr_TyCon tycon) (* ^ "TYCON_RID" *)
		 | EXCON_RID excon => (Excon.pr_excon excon) (* ^ "EXCON_RID" *)
		 | CON_RID con => (Con.pr_con con) (*  ^ "CON_RID" *)
		 | TYVAR_RID tyvar => (StatObjectProp.pr_TyVar tyvar) (* ^ "TYVAR_RID" *)
                 | STRID_RID strid => (StrId.pr_StrId strid)
		 | SIGID_RID sigid => (SigId.pr_SigId sigid)
		 | FUNID_RID funid => (FunId.pr_FunId funid)

	       val sids = List.foldL 
		          (fn rid => fn sids => (pr_repeatedId rid) ^ sids)
			  ""
			  ids
	     in
	       line("Repeated identifier(s): " ^ sids)
	     end

         | TYVARS_NOT_IN_TYVARSEQ tyvars =>
	     line("The tyvar(s) " 
		  ^ (List.foldL (fn tv => fn s => 
			      (StatObjectProp.pr_TyVar tv) ^ s) "" tyvars)
		  ^ " should occur in tyvarseq")

(* mju#1.6 *)
         | SHOULD_ADMIT_EQ typefcns =>
	     let
		 val tfcns = map StatObjectProp.pr_TypeFcn typefcns
	     in
		 line("Type function(s):") //
		 line("   " ^ (List.foldR (fn s1 => fn s2 => 
					   (if s2 = "" 
					    then s1
					    else s1 ^ ", " ^ s2))
			                   ""
					   tfcns)) //
		 line("should admit equality")
	     end
       | LOCAL_TYNAMES =>
	     line("Local tyname(s) escape from a let expression")
(* endmju#1.6 *)

	 | LOOKUP_SIGID sigid =>
	     line("Unbound signature identifier: " ^ SigId.pr_SigId sigid)

	 | LOOKUP_LONGSTRID longstrid =>
	     line("Unbound structure identifier: "
		  ^ StrId.pr_LongStrId longstrid
		 )

	 | LOOKUP_FUNID funid =>
	     line("Unbound functor identifier: " ^ FunId.pr_FunId funid)

	 | ASSEMBLY_COVER(strids, tail) =>
	     let
	       fun prTail(TYCON tycon) = TyCon.pr_TyCon tycon
		 | prTail(STRID strid) = StrId.pr_StrId strid
	     in
	       line("Some totally obscure assembly coverage error: "
		    ^ prStrIds strids ^ prTail tail
		   )
	     end

	 | NOTTYPEEXPLICIT =>
	     line "Signature not type explicit"

         | NOTWELLFORMEDSIG =>
	     line "Signature not well-formed"

	 | NOTEQPRINCIPAL longtycons =>
	     line("Not equality principal in: "
		  ^ List.foldR
		      (fn tc => fn str => TyCon.pr_LongTyCon tc ^ " " ^ str)
		      "" longtycons
		 )

         | EXDESC_SIDECONDITION =>
	     line "Type variables not allowed in type expression in exception description"

(* mju#1.6 *)
         | AXEXP_SHOULD_BE_BOOL => 
	     line "Axiomatic expressions must be of type bool"

         | UNGUARD_EXPLICIT_TV_IN_AXEXP =>
	     line "Unguarded explicit type variable(s) in axiomatic expression"
(* endmju#1.6 *)
	     
	 | S_RIGIDSTRCLASH longstrid_opt =>
	     line("Rigid structure clash: "
		  ^ (case longstrid_opt of Some x => StrId.pr_LongStrId x
		       			 | None => "?" (* MEMO: what is this? *)
		    )
		 )

	 | MISSINGSTR longstrid =>
	     line("Missing structure: " ^ StrId.pr_LongStrId longstrid)

	 | MISSINGTYPE longtycon =>
	     line("Missing type: " ^ TyCon.pr_LongTyCon longtycon)

	 | S_CONFLICTINGARITY(longtycon, _) =>
	     line("S/Conflicting arity: " ^ TyCon.pr_LongTyCon longtycon)

	 | CONFLICTINGEQUALITY(longtycon, _) =>
	     line("Conflicting equality attributes: "
		  ^ TyCon.pr_LongTyCon longtycon
		 )

	 | MISSINGVAR(strids, id) =>
	     line("Missing variable: " ^ prStrIds strids
		  ^ StatObjectProp.pr_id id
		 )

	 | MISSINGEXC(strids, excon) =>
	     line("Missing exception: " ^ prStrIds strids
		  ^ Excon.pr_excon excon
		 )

	 | S_RIGIDTYCLASH longtycon =>
	     line("Rigid type clash for: " ^ TyCon.pr_LongTyCon longtycon)

	 | S_CONFLICTING_DOMCE longtycon =>
	     line("Conflicting CE domains for: " ^ TyCon.pr_LongTyCon longtycon)

	 | NOTYENRICHMENT(strids, id) =>
	     line("No type enrichment: " ^ prStrIds strids
		  ^ StatObjectProp.pr_id id
		 )

	 | EXCNOTEQUAL(strids, excon, (ty1, ty2)) =>
	     line("Types for exception not equal: "
		  ^ prStrIds strids ^ Excon.pr_excon excon
		  ^ ", " ^ StatObjectProp.pr_Type ty1
		  ^ ", " ^ StatObjectProp.pr_Type ty2
		 )

	 | U_RIGIDSTRCLASH(longstrid1, longstrid2) =>
	     line("Rigid structure clash: " ^ StrId.pr_LongStrId longstrid1
		  ^ ", " ^ StrId.pr_LongStrId longstrid2
		 )

	 | CYCLE longstrid =>
	     line("Unification cycle: " ^ StrId.pr_LongStrId longstrid)

	 | U_RIGIDTYCLASH args =>
	     line("Rigid type clash: " ^ pr_UnifyArgs args)

	 | TYPESTRILLFORMEDNESS args =>
	     line("Ill-formed type structure: " ^ pr_UnifyArgs args)

	 | U_CONFLICTING_DOMCE args =>
	     line("Conflicting CE domains: " ^ pr_UnifyArgs args)

	 | U_CONFLICTINGARITY args =>
	     line("U/Conflicting arity: " ^ pr_UnifyArgs args)

	 | RIGIDTYFUNEQERROR args =>
	     line("Equality attribute differs: " ^ pr_UnifyArgs args)
  end;
