(* Copyright (C) 2003 Mikolaj Konarski
 *
 * This file is part of the Dule compiler.
 * The Dule compiler is released under the GNU General Public License (GPL).
 * Please see the file Dule-LICENSE for license information.
 *
 * $Id: mod_back.ml,v 1.42 2003/12/20 23:57:58 mikon Exp $
 *) 

open Middle_middle open Error_rep open Tools open Core_back open Core_front

module type ToolWSign = (* W-Dule --- module system with specialization *)
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Cat : T
    module Funct : T
    val singlePp : Funct.t -> Cat.t * Cat.t IList.t * Cat.t IList.t
    val footPp : Cat.t  IList.t -> IdIndex.t * Funct.t -> Funct.t
  end

module ToolWSign' 
    (IdIndex : IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Cat : SemFCat
    with module IdIndex = IdIndex 
    with module IList = IList)
    (Funct : ConPFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    : (ToolWSign
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with module Funct = Funct) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Cat = Cat
    module Funct = Funct
    open IList
    open Cat
    open Funct
    open SrcFCore

    let singlePp s =
      let c = src s in
      let lc = unPP c in
      let lcdu = ifilter IdIndex.is_dule lc in
      let lty = diff lc lcdu in
      let lf = unpp s in
      (* these are shared with the other signatures: *)
      let ld = ifilter (fun i -> not_in i lf) lcdu in
      (* if [s] was [S_Pp] (maybe in [S_Ww]) *)
      (* then [lcom] (its [cat] components) *)
      (* are as numerous as [lf] else they are [nil]: *)
      let lcom = diff lcdu ld in
      let lb = lcom @@ lty in
      (c, lb, ld)

    let footPp lc (i, s) =
      let lb = unPP (find i lc) in
      let f_PR_lc_i = f_PR lc i in
      let pib = imap (fun j -> 
	f_COMP f_PR_lc_i (f_PR lb j)) lb in
      let le = unPP (src s) in
      let ld = diff le lb in
      let pid = imap (f_PR lc) ld in
      let c = c_PP lc in
      f_RECORD c (pib @@ pid)
  end

module ToolWSign = ToolWSign' (IdIndex) (IList) (SemFCat) (ConPFunct) (SrcFCore)


module type ToolWDule =
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Cat : T
    module Funct : T
    module Trans : T
    val pack : Funct.t * Trans.t -> Trans.t
    val unpack : Trans.t -> Funct.t * Trans.t
    val att_dule : Trans.t -> Funct.t
    val atv_dule : Trans.t -> Funct.t
  end

module ToolWDule' 
    (IdIndex : IdIndex)
    (AtIndex : AtIndex with module IdIndex = IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Cat : SemFCat
    with module IdIndex = IdIndex 
    with module IList = IList)
    (Funct : ConPFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (Trans : ConPTrans
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    (DomFCore : DomFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    : (ToolWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with module Funct = Funct
    with module Trans = Trans) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Cat' = Cat
    module Funct = Funct
    module Trans = Trans
    open IList
    open Funct
    open Trans
    module Cat = Cat'

    let pack (f, t) = 
      t_RECORD (SrcFCore.src f) 
	(cons (AtIndex.att, t_id f)
           (cons (AtIndex.atv, t) nil))

    let unpack m =
      let le = Cat.unPP (SrcFCore.trg (DomFCore.dom m)) in
      (DomFCore.dom (t_TF m (f_PR le AtIndex.att)), 
       t_TF m (f_PR le AtIndex.atv))

    let att_dule m = 
      let f = DomFCore.dom m in
      let le = Cat.unPP (SrcFCore.trg f) in
      f_COMP f (f_PR le AtIndex.att)

    let atv_dule m = 
      let f = DomFCore.dom m in
      let le = Cat.unPP (SrcFCore.trg f) in
      f_COMP f (f_PR le AtIndex.atv)
  end

module ToolWDule = ToolWDule' (IdIndex) (AtIndex) (IList) (SemFCat)
    (ConPFunct) (ConPTrans) (SrcFCore) (DomFCore)


module type OkWSign =
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Cat : T
    module Funct : T
    val legPp : Cat.t IList.t -> IdIndex.t * Funct.t -> Funct.t
    val lsrcPp : Funct.t IList.t -> [`OK of Cat.t IList.t|`Error of string]
    val lsrcWw : Funct.t -> Funct.t ->
      [`OK of Cat.t IList.t * Cat.t IList.t * Cat.t IList.t 
	  * Cat.t IList.t * Cat.t IList.t
      |`Error of string]
  end

module OkWSign' 
    (IdIndex : IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Cat : SemFCat with module IdIndex = IdIndex and module IList = IList)
    (Funct : ConPFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    (EqFCat : EqFCat with type Cat.t = Cat.t)
    (ToolWSign : ToolWSign
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    : (OkWSign
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with module Funct = Funct) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Cat = Cat
    module Funct = Funct
    open IList
    open Cat
    open Funct
    open SrcFCore

    let append_cat l k = append_eq EqFCat.eq l k

    let legPp lc (i, s) = 
      f_COMP (ToolWSign.footPp lc (i, s)) s

    let lsrcPp ls =
      let single_src (i, s) r =
	let (c, lb, ld) = ToolWSign.singlePp s in
	let l1 = cons (i, c_PP lb) nil in
	(match append_cat l1 ld with
	|`OK la ->
	    append_cat la r
	|`Error er -> `Error er)
      in
      bfold1ok nil single_src ls

    let lsrcWw f1 s2 = (* f1 = att_dule m1, m1 : r1 -> s1=r2 *)
      let lc1 = unPP (src f1) in (* src r1 *)
      let le1 = unPP (trg f1) in (* src s1=r2 *)
      let le2 = unPP (src s2) in
      let ld = diff le2 le1 in
      let le12 = diff le2 ld in
      (match append_cat ld lc1 with
      |`OK la -> 
	  `OK (lc1, le1, le12, ld, la)
      |`Error er -> `Error er)
  end

module OkWSign = OkWSign' (IdIndex) (IList) (SemFCat) (ConPFunct)
    (SrcFCore) (EqFCat) (ToolWSign) 


module type SemWSign =
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Location : Location
    module ErrorRepLib : ErrorRepLib
    with module Location = Location
    module Cat : T
    module Funct : T
    module Trans : T
    val s_Pp : Funct.t IList.t -> [`OK of Funct.t|`Error of string]
    val s_Bb : Funct.t -> Cat.t IList.t -> Funct.t IList.t -> 
      [`OK of Funct.t|`Error of string]
    val sc_Bb : Funct.t -> Cat.t IList.t -> 
      (Cat.t -> [`OK of Funct.t|`Error of ErrorRepLib.error]) IList.t -> 
	[`OK of Funct.t|`Error of ErrorRepLib.error]
    val s_Ww : Trans.t -> Funct.t -> 
      [`OK of Funct.t|`Error of string]
  end

module SemWSign'
    (IdIndex : IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Location : Location)
    (ErrorRepLib : ErrorRepLib
    with module Location = Location)
    (Cat : SemFCat with module IdIndex = IdIndex and module IList = IList)
    (Funct : ConPFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (Trans : T)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    (OkError : OkError
    with module IdIndex = IdIndex
    with module IList = IList)
    (ToolWDule : ToolWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (OkWSign : OkWSign
    with module IdIndex = IdIndex
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    (EqFCat : EqFCat with type Cat.t = Cat.t)
    : (SemWSign
    with module IdIndex = IdIndex
    with module IList = IList
    with module Location = Location
    with module ErrorRepLib = ErrorRepLib
    with module Cat = Cat
    with module Funct = Funct
    with module Trans = Trans) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Location = Location
    module ErrorRepLib = ErrorRepLib
    module Cat = Cat
    module Funct = Funct
    module Trans = Trans
    open IList
    open Cat
    open Funct
(*
   Functor s is called a signature iff
   s = f_pp (c_PP lc) lf, for some lc and lf
*)
    let s_Pp ls = 
      (match OkWSign.lsrcPp ls with
      |`OK lc -> 
	  let legs = bmap (OkWSign.legPp lc) ls in
	  let c = c_PP lc in
	  let body = f_pp c legs in
	  `OK body
      |`Error er -> `Error er)

    let s_Bb r lc lf =
      let lb = unPP (SrcFCore.src r) in
      (* we merge local and context parameters: *)
      let c = c_PP (lc @@ subtract lb lc) in 
      let la = vmap SrcFCore.src lf in
      if vforall (EqFCat.eq c) la then
	let le = vmap SrcFCore.trg lf in
	if vforall (EqFCat.eq c_BB) le then
	  `OK (f_pp c lf)
	else `Error "types are of compund kinds in s_Bb"
      else `Error "types depend on wrong modules in s_Bb"

    let sc_Bb r lc l_c2f =
      let lb = unPP (SrcFCore.src r) in
      (* we merge local and context parameters: *)
      let c = c_PP (lc @@ lb) in 
      (match OkError.vmap1ok (fun c2f -> c2f c) l_c2f with
      |`OK lf ->
	  `OK (f_pp c lf)
      |`Error er -> `Error er)		

    let s_Ww m1 s2 =
      let f1 = ToolWDule.att_dule m1 in
      (match OkWSign.lsrcWw f1 s2 with
      |`OK (lc1, le1, le12, ld, la) -> 
	  let a = c_PP la in
	  let pic1 = imap (f_PR la) lc1 in 
	  let f = f_COMP (f_RECORD a pic1) f1 in
	  let pife1 = imap (fun i -> f_COMP f (f_PR le1 i)) le12 in
	  let pid = imap (f_PR la) ld in 
	  let re = f_RECORD a (pife1 @@ pid) in
	  `OK (f_COMP re s2)
      |`Error er -> `Error er)
  end

module SemWSign = SemWSign' (IdIndex) (IList) (Location) (ErrorRepLib)
    (SemFCat) (ConPFunct) (ATrans) (SrcFCore) (OkError) (ToolWDule) (OkWSign)
    (EqFCat)


module type OkWDule =
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Cat : T
    module Funct : T
    module Trans : T
    val mayRecord : Funct.t IList.t -> Funct.t IList.t -> 
      [`OK of Funct.t IList.t|`Error of string]
    val mayBase : Cat.t -> Cat.t -> Funct.t IList.t -> 
      [`OK of Funct.t|`Error of string]
    val mayInst : Funct.t -> Funct.t -> [`OK of Funct.t|`Error of string]
    val mayTrim_f : Cat.t -> Cat.t -> [`OK of Funct.t|`Error of string]
    val mayTrim_t : Funct.t -> Funct.t -> [`OK of Trans.t|`Error of string]
  end

module OkWDule' 
    (IdIndex : IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Cat : SemFCat with module IdIndex = IdIndex and module IList = IList)
    (Funct : ConPFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (Trans : ConPTrans
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    (PpFCat : PpFCat with type Cat.t = Cat.t)
    (PpFFunct : PpFFunct
    with type Funct.t = Funct.t)
    (EqFCat : EqFCat with type Cat.t = Cat.t)
    (EqFFunct : EqFFunct with type Funct.t = Funct.t)
    (OkError : OkError
    with module IdIndex = IdIndex
    with module IList = IList)
    (ToolWSign : ToolWSign
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    : (OkWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with module Funct = Funct
    with module Trans = Trans) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Cat = Cat
    module Funct = Funct
    module Trans = Trans
    open IList
    open Cat
    open Funct
    open Trans
    open SrcFCore

    let append_funct l k = append_eq EqFFunct.eq l k

    let mayRecord lf ls = 
      let cut_at_i (i, f) r =
	let s = find i ls in
	let (c, lb, ld) = ToolWSign.singlePp s in
	let lc = unPP c in
	let lfid = imap (fun i -> f_COMP f (f_PR lc i)) ld in
	let pib = imap (f_PR lc) lb in
	let fib = f_COMP f (f_RECORD c pib) in
	let l1 = cons (i, fib) nil in
	(match append_funct l1 lfid with
	|`OK lg ->
	    append_funct lg r
	|`Error er -> `Error er)
      in
      bfold1ok nil cut_at_i lf

    let mayBase c e lg =
      let la = vmap SrcFCore.src lg in
      if vforall (EqFCat.eq c) la then
	let lc = unPP c in
	let le = unPP e in
	let ld = inter lc le in (* context parameters *)
	let pild = imap (f_PR lc) ld in 
	let lf = lg @@ subtract pild lg in (* some are curtailed *)
	let f = f_RECORD c lf in
	let d = trg f in
	if EqFCat.eq d e then
  	  `OK f
	else `Error ("mayBase: type definitions do not agree with signature"
		     ^ "\nd(" ^ PpFCat.pp_c d ^ ") <> \ne("
		     ^ PpFCat.pp_c e ^ ")")
      else `Error "mayBase: type definitions depend on wrong modules"

    let mayInst f1 f2 = 
      let c1 = src f1 in
      let lc1 = unPP c1 in (* src r1 *)
      let le1 = unPP (trg f1) in (* src s1=r2 *)
      let le2 = unPP (trg f2) in (* src s2 *)
      let ld = diff le2 le1 in
      let f12 = f_COMP f1 f2 in
      let pid = imap (fun i -> f_COMP f12 (f_PR le2 i)) ld in 
      let pic1 = imap (f_PR lc1) lc1 in 
      (match append_funct pid pic1 with
      |`OK lf ->
	  `OK (f_RECORD c1 lf)
      |`Error er -> `Error er)

    let rec mayTrim_f e c = 
      if EqFCat.eq e c then 
	`OK (f_ID c) 
      else
      (match unPPok c with
      |`OK lc ->
	  (match unPPok e with
	  |`OK le ->
	      let fsu (i, c) =
		(match find_ok i le with
		|`OK e ->
		    (match mayTrim_f e c with
		    |`OK sf ->
			`OK (f_COMP (f_PR le i) sf)
		    |`Error er -> `Error er)
		|`Error er -> 
		    `Error ("mayTrim_f: " ^ IdIndex.t2string i ^ " not in le"))
	      in
	      (match OkError.bmap1ok fsu lc with
	      |`OK lf ->
		  `OK (f_RECORD e lf)
	      |`Error er -> `Error er)
	  |`Error er -> `Error "mayTrim_f: e not PP")
      |`Error er -> `Error "mayTrim_f: e <> c")

    let rec mayTrim_t h f = 
      if EqFFunct.eq h f then 
	`OK (t_id f) 
      else
      (match unpp_ok f with
      |`OK lf ->
	  (match unpp_ok h with
	  |`OK lh ->
	      let fsu (i, f) =
		(match find_ok i lh with
		|`OK h ->
		    (match mayTrim_t h f with
		    |`OK sf ->
			`OK (t_comp (t_pr lh i) sf)
		    |`Error er -> `Error er)
		|`Error er ->
		    `Error ("mayTrim_t: " ^ IdIndex.t2string i ^ " not in lh"))
	      in
	      (match OkError.bmap1ok fsu lf with
	      |`OK lt ->
		  `OK (t_record h lt)
	      |`Error er -> `Error er)
	  |`Error er -> `Error "mayTrim_t: h not pp")
      |`Error er -> `Error ("mayTrim_t: \nh(" 
			    ^ PpFFunct.pp_f h ^ ") <> \nf("
			    ^ PpFFunct.pp_f f ^ ")"))
  end

module OkWDule = OkWDule' (IdIndex) (IList) (SemFCat) (ConPFunct) (ConPTrans)
    (SrcFCore) (PpFCat) (PpFFunct) (EqFCat) (EqFFunct) (OkError) (ToolWSign)


module type SemWDule =
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Location : Location
    module ErrorRepLib : ErrorRepLib
    with module Location = Location
    module Cat : T
    module Funct : T
    module Trans : T

    val m_Id : Funct.t -> Trans.t
    val m_Comp : Trans.t -> Trans.t -> Trans.t
    val m_Pr : Funct.t IList.t -> IdIndex.t -> 
      [`OK of Trans.t|`Error of string]
    val m_Record : Funct.t -> Trans.t IList.t -> Funct.t IList.t ->
      [`OK of Trans.t|`Error of string]
    val m_Base : Funct.t -> Funct.t ->
      Funct.t IList.t -> Trans.t IList.t ->
	[`OK of Trans.t|`Error of string]
    val mc_Base : Funct.t -> Funct.t -> 
      (Cat.t -> [`OK of Funct.t|`Error of ErrorRepLib.error]) IList.t ->
	(Funct.t -> Funct.t -> 
	  [`OK of Trans.t|`Error of ErrorRepLib.error]) IList.t ->
	    Location.t ->
	      [`OK of Trans.t|`Error of ErrorRepLib.error]
    val m_Inst : Trans.t -> Trans.t -> [`OK of Trans.t|`Error of string]
    val m_Trim : Trans.t -> Funct.t -> [`OK of Trans.t|`Error of string]
  end

module SemWDule'
    (IdIndex : IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Location : Location)
    (ErrorRepLib : ErrorRepLib
    with module Location = Location)
    (Cat : ConPCat with module IdIndex = IdIndex and module IList = IList)
    (Funct : ConPFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (Trans : ConPTrans
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    (DomFCore : DomFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (EqFFunct : EqFFunct with type Funct.t = Funct.t)
    (PpFFunct : PpFFunct 
    with type Funct.t = Funct.t
    with module Funct.IdIndex = IdIndex
    with module Funct.IList = IList)
    (OkError : OkError
    with module IdIndex = IdIndex
    with module IList = IList)
    (ToolWSign : ToolWSign
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    (ToolWDule : ToolWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (OkWSign : OkWSign
    with module IdIndex = IdIndex
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    (OkWDule : OkWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    : (SemWDule
    with module IdIndex = IdIndex
    with module IList = IList
    with module Location = Location
    with module ErrorRepLib = ErrorRepLib
    with module Cat = Cat
    with module Funct = Funct
    with module Trans = Trans) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Location = Location
    module ErrorRepLib = ErrorRepLib
    module Cat = Cat
    module Funct = Funct
    module Trans = Trans
    open IList
    open Cat
    open Funct
    open Trans
    open SrcFCore
    open ErrorRepLib
(*
   Transformation m is called a module, 
   iff there is a functor f, transformation t,
   signature r and signature s, such that
   1. m = pack (f, t)
   2. f : src r -> src s
   3. t : r -> f_COMP f s

   Invariant: if m : r -> s type-correct then m is a module with r, s.
*)
    let m_Id s = (* : s -> s *)
      let c = src s in
      ToolWDule.pack (f_ID c, t_id s)

    let m_Comp m1 m2 = (* : r1 -> s2 *)
      let (f1, t1) = ToolWDule.unpack m1 in (* : r1 -> s1 *)
      let (f2, t2) = ToolWDule.unpack m2 in (* : r2 -> s2 *)
      let f = f_COMP f1 f2 in (* s1 = r2 *)
      let it2 = t_FT f1 t2 in
      let t = t_comp t1 it2 in
      ToolWDule.pack (f, t)
(*
The following drawing shows the domains and codomains
of the transformations appearing in the definition of [m_Comp].

       t1    s1         t2    s2         t    s2
  r1 ------> --    r2 ------> --    r1 -----> -- 
             f1               f2              f2
                                              --
                                              f1

The drawing below illustrates the value part, 
called [t], of the result of module composition.
Horizontal composition of transformations is here represented by 
placing the first transformation below the second.
Vertical composition is represented by sharing a common domain/codomain.

                  t2    s2
             s1 ------> --       s1 = r2
       t1               f2
  r1 ------> --         --
             f1 ------> f1
                 t_id
*)

    let m_Pr lr i = (* : S_Pp lr -> s *)
      (match find_ok i lr with
      |`OK s ->	
	  (match OkWSign.lsrcPp lr with
	  |`OK lc -> 
	      let foot_i = ToolWSign.footPp lc (i, s) in
	      let legs = bmap (OkWSign.legPp lc) lr in
	      let t = t_pr legs i in
	      `OK (ToolWDule.pack (foot_i, t))
	  |`Error er -> `Error er)
      |`Error er -> `Error er)

    let m_Record r lm ls = (* : r -> S_Pp ls *)
      let lft = vmap ToolWDule.unpack lm in (* : r -> s_i *)
      let lf = vmap (fun (f, t) -> f) lft in
      let lt = vmap (fun (f, t) -> t) lft in
      (match OkWDule.mayRecord lf ls with
      |`OK lf ->
	  let c = src r in
	  let f = f_RECORD c lf in 
	  let t = t_record r lt in
	  `OK (ToolWDule.pack (f, t))
      |`Error er -> `Error er)

    let m_Base r s lg lt = (* : r -> s *)
      let c = src r in
      let e = src s in
      (match OkWDule.mayBase c e lg with
      |`OK f ->
	  let lr = vmap DomFCore.dom lt in
	  if vforall (EqFFunct.eq r) lr then
	    let t = t_record r lt in
	    let ts = DomFCore.cod t in
	    let fs = f_COMP f s in
	    if EqFFunct.eq ts fs then
	      `OK (ToolWDule.pack (f, t))
	    else `Error 
              "value definitions do not agree with signature in m_Base"
	  else `Error "values depend on wrong modules in m_Base"
      |`Error er -> `Error er)

    (* f in l_c2g && f c = OK g => src g = c *)
    (* t in l_fh2t && t f h = OK u => dom u = f && cod u = h *)
    let mc_Base r s l_c2g l_fh2t l = (* : r -> s *)
      let c = src r in
      let e = src s in
      (match OkError.vmap1ok (fun c2f -> c2f c) l_c2g with
      |`OK lg ->
	  (match OkWDule.mayBase c e lg with
	  |`OK f ->
	      let lh = unpp (f_COMP f s) in
	      let fih (i, fh2t) = 
		(match find_ok i lh with
		|`OK h -> fh2t r h
		|`Error er ->
		    `Error 
		      (modBackError#instance 
			 [Loc l; 
			  Msg (er ^ " not found in specification")]))
	      in
	      (match OkError.bmap1ok fih l_fh2t with
	      |`OK lt -> 
		  let lht = vmap DomFCore.cod lt in
		  let not_defined = diff lh lht in
		  if is_nil not_defined then
		    let t = t_record r lt in
		    `OK (ToolWDule.pack (f, t))
		  else
		    `Error 
		      (modBackError#instance 
			 [Loc l; 
			  Msg ("values specified but not defined:\n" 
			       ^ PpFFunct.pp_lf not_defined)])
	      |`Error er -> `Error er)
	  |`Error er ->
	      `Error 
		(modBackError#instance 
		   [Loc l; 
		    Msg er]))
      |`Error er -> `Error er)

    let m_Inst m1 m2 = (* : r1 -> S_Ww (m1, s2) *)
      let (f1, t1) = ToolWDule.unpack m1 in (* : r1 -> s1=r2 *)
      let (f2, t2) = ToolWDule.unpack m2 in (* : s1=r2 -> s2 *)
      (match OkWDule.mayInst f1 f2 with
      |`OK f -> 
	  let it2 = t_FT f1 t2 in
	  let t = t_comp t1 it2 in
	  `OK (ToolWDule.pack (f, t))
      |`Error er -> `Error er)

    let m_Trim m1 r2 = (* : r1 -> r2 *)
      let (f1, t1) = ToolWDule.unpack m1 in (* : r1 -> s1 *)
      let e1 = trg f1 in (* [src s1] *)
      let c2 = src r2 in
      (match OkWDule.mayTrim_f e1 c2 with
      |`OK scf ->
	  let f = f_COMP f1 scf in
	  let fcr2 = f_COMP f r2 in
	  let f1s1 = DomFCore.cod t1 in (* = [f_COMP f1 s1] *)
	  (match OkWDule.mayTrim_t f1s1 fcr2 with (* weaker than s1 > r2 *)
	  |`OK sct ->
	      let t = t_comp t1 sct in
	      `OK (ToolWDule.pack (f, t))
	  |`Error er -> `Error er)
      |`Error er -> `Error er)
  end

module SemWDule = SemWDule' (IdIndex) (IList) (Location) (ErrorRepLib)
    (ConPCat) (ConPFunct) (ConPTrans) (SrcFCore) (DomFCore) (EqFFunct)
    (PpFFunct) (OkError) (ToolWSign) (ToolWDule) (OkWSign) (OkWDule)


module type SemLDule = (* L-Dule --- module system with linking *)
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Cat : T
    module Funct : T
    module Trans : T
    val m_Accord : Funct.t IList.t -> Trans.t IList.t -> Funct.t IList.t ->
      [`OK of Trans.t|`Error of string]
    val m_Concord : Funct.t IList.t -> Trans.t IList.t -> Funct.t IList.t ->
      [`OK of Trans.t|`Error of string]
    val m_Link : Funct.t IList.t -> Trans.t IList.t -> Funct.t IList.t ->
      [`OK of Trans.t|`Error of string]
    val m_Link_ordered : 
	Funct.t IList.t -> Trans.t IList.t -> Funct.t IList.t ->
	  [`OK of Trans.t|`Error of string]
  end

module SemLDule'
    (IdIndex : IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Cat : T)
    (Funct : ConPFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (Trans : T)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    (OkError : OkError
    with module IdIndex = IdIndex
    with module IList = IList)
    (SemWSign : SemWSign
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (SemWDule : SemWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (ToolWDule : ToolWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    : (SemLDule
    with module IdIndex = IdIndex
    with module IList = IList
    with module Cat = Cat
    with module Funct = Funct
    with module Trans = Trans) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Cat = Cat
    module Funct' = Funct
    module Trans = Trans
    open IList
    open SrcFCore
    open OkError
    module Funct = Funct'

    let m_Accord lr lm ls = (* : S_Pp lr -> S_Pp ls *)
      (match SemWSign.s_Pp lr with
      |`OK r ->
	  (match imap1ok (SemWDule.m_Pr lr) lr with
	  |`OK lpr ->
	      let prm m =
		let lf = Funct.unpp (ToolWDule.atv_dule m) in
		let lrf = imap (fun i -> find i lr) lf in
		let lmf = imap (fun i -> find i lpr) lf in
		(match SemWDule.m_Record r lmf lrf with
		|`OK re ->
		    `OK (SemWDule.m_Comp re m)
		|`Error er -> `Error er)
	      in
	      (match vmap1ok prm lm with
	      |`OK lm ->
		  SemWDule.m_Record r lm ls
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      |`Error er -> `Error er)

    let m_Concord lr lm ls = (* : S_Pp lr -> S_Pp (ls @@ diff lr ls) *)
      (match SemWSign.s_Pp lr with
      |`OK r ->
	  (match imap1ok (SemWDule.m_Pr lr) lr with
	  |`OK lpr ->
	      let prm m =
		let lf = Funct.unpp (ToolWDule.atv_dule m) in
		let lrf = imap (fun i -> find i lr) lf in
		let lmf = imap (fun i -> find i lpr) lf in
		(match SemWDule.m_Record r lmf lrf with
		|`OK re ->
		    `OK (SemWDule.m_Comp re m)
		|`Error er -> `Error er)
	      in
	      (match vmap1ok prm lm with
	      |`OK lm ->
		  let lm = lm @@ subtract lpr lm in
		  let ls = ls @@ diff lr ls in
		  SemWDule.m_Record r lm ls
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      |`Error er -> `Error er)

    (* here order of lm does't matter, but no circularity allowed: *)
    let m_Link lr lm ls = (* : S_Pp lr -> Pp ls *)
      (match SemWSign.s_Pp lr with
      |`OK r ->
	  (match imap1ok (SemWDule.m_Pr lr) lr with
	  |`OK lpr ->
	      let lrls = lr @@ ls in
	      let rec rlink i = 
		(match find_ok i lpr with
		|`OK pr -> `OK pr
		|`Error er ->
		    let m = find i lm in
		    let lf = Funct.unpp (ToolWDule.atv_dule m) in
		    let lrf = imap (fun i -> find i lrls) lf in
		    (match imap1ok rlink lf with
		    |`OK lmf ->
			(match SemWDule.m_Record r lmf lrf with
			|`OK re ->
			    `OK (SemWDule.m_Comp re m)
			|`Error er -> `Error er)
		    |`Error er -> `Error er))
	      in
	      (match imap1ok rlink lm with
	      |`OK lm ->
		  SemWDule.m_Record r lm ls
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      |`Error er -> `Error er)

    (* here we assume a module depends only on the previous ones in lm: *)
    let m_Link_ordered lr lm ls = (* : S_Pp lr -> S_Pp ls *)
      (match SemWSign.s_Pp lr with
      |`OK r ->
	  (match imap1ok (SemWDule.m_Pr lr) lr with
	  |`OK lpr -> pri "(";
	      let lrls = lr @@ ls in
	      let pro = fun (i, m) lt ->
		pri (" L:" ^ IdIndex.t2s i ^ " ");   
		let lf = Funct.unpp (ToolWDule.atv_dule m) in
		let lrf = imap (fun i -> find i lrls) lf in
		(match imap1ok (fun i -> find_ok i lt) lf with
		|`OK lmf ->
		    (match SemWDule.m_Record r lmf lrf with
		    |`OK re ->
			let t = SemWDule.m_Comp re m in
			`OK (cons (i, t) lt)
		    |`Error er -> `Error er)
		|`Error er -> 
		    `Error (IdIndex.t2string i 
			    ^ " depends on an unknown "
			    ^ er))
	      in
	      (match bfold1ok lpr pro lm  with
	      |`OK lprlm -> pri ")";
		  let lm = subtract lprlm lpr in
		  SemWDule.m_Record r lm ls
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      |`Error er -> `Error er)
  end

module SemLDule = SemLDule' (IdIndex) (IList) (ACat) (ConFFunct) (ATrans)
    (SrcFCore) (OkError) (SemWSign) (SemWDule) (ToolWDule)


module type ToolIDule = (* I-Dule --- module system with inductive modules *)
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Cat : T
    module Funct : T
    module Trans : T
    val addcon : Funct.t -> Funct.t -> Trans.t
  end

module ToolIDule' 
    (IdIndex : IdIndex)
    (AtIndex : AtIndex with module IdIndex = IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Cat : SemFCat with module IdIndex = IdIndex and module IList = IList)
    (Funct : SemFFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (Trans : SemFTrans
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    (DomFCore : DomFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    : (ToolIDule
    with module IdIndex = IdIndex
    with module IList = IList
    with module Cat = Cat
    with module Funct = Funct
    with module Trans = Trans) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Cat = Cat
    module Funct = Funct
    module Trans = Trans
    open IList
    open Cat
    open Funct
    open Trans
    open SrcFCore

    let addcon f g =
      let le2 = unPP (src g) in
      let d = src f in
      let ld = unPP d in
      let pid = imap (fun i -> t_id (f_PR ld i)) (inter ld le2) in
      let le = unPP (trg f) in
      let pie = imap (fun i -> 
	t_id (f_COMP f (f_PR le i))) (inter le le2) in
      let fcon = t_con (unii f) in
      let pifc = imap (fun i -> t_TF fcon (f_PR le i)) le in
      let pifc2 = rmap AtIndex.dule2sp pifc in
      let tc = t_RECORD d (pid @@ pie @@ pifc2) in
      (* if t_TF was correct (for ee it is not) then here this would suffice:
         t_TF tc g, but now we have to use complicated t_TF_coco below: *)
      let fde = t_de (unii f) in
      let pifd = imap (fun i -> t_TF fde (f_PR le i)) le in
      let pifd2 = rmap AtIndex.dule2sp pifd in
      let td = t_RECORD d (pid @@ pie @@ pifd2) in
      t_TF_coco d (tc, td) g
  end

module ToolIDule = ToolIDule' (IdIndex) (AtIndex) (IList) (SemFCat) (SemFFunct)
    (ConFTrans) (SrcFCore) (DomFCore)


module type SemIDule =
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Cat : T
    module Funct : T
    module Trans : T
    val m_Ind : 
	Funct.t IList.t -> Trans.t IList.t -> Funct.t IList.t ->
	  [`OK of Trans.t|`Error of string]
    val m_CoInd : 
	Funct.t IList.t -> Trans.t IList.t -> Funct.t IList.t ->
	  [`OK of Trans.t|`Error of string]
  end

module SemIDule'
    (IdIndex : IdIndex)
    (AtIndex : AtIndex with module IdIndex = IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Cat : SemFCat with module IdIndex = IdIndex and module IList = IList)
    (Funct : SemFFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (Trans : ConFTrans
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    (DomFCore : DomFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (SemWSign : SemWSign
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (ToolWDule : ToolWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (SemLDule : SemLDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (ToolIDule : ToolIDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    : (SemIDule
    with module IdIndex = IdIndex
    with module IList = IList
    with module Cat = Cat
    with module Funct = Funct
    with module Trans = Trans) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Cat' = Cat
    module Funct = Funct
    module Trans = Trans
    open IList
    open Cat
    open Funct
    open Trans
    open SrcFCore
    open DomFCore
    module Cat = Cat'

    let rec m_Ind lr lm ls = (* : S_Pp lr -> S_Pp ls *)
      pri "QS ";
      let lrs = lr @@ ls in
      pri "NS ";
      let ls2 = rmap AtIndex.dule2sp ls in
      let lm2 = rmap AtIndex.dule2sp lm in
      (match SemLDule.m_Accord lrs lm2 ls2 with (* : S_Pp lrs -> S_Pp ls2 *)
      |`OK m2 ->
	  let (f2, t2) = ToolWDule.unpack m2 in
(* cutting and renaming f2: *)
	  let e2 = trg f2 in
	  let le2 = unPP e2 in
(*	  let pie2 = imap (f_PR le2) (ifilter (fun i -> is_in i ls2) le2) in *)
	  let pie2 = imap (f_PR le2) ls2 in
	  let pie = rmap AtIndex.sp2dule pie2 in
	  let f = f_COMP f2 (f_RECORD e2 pie) in
(* f_ii f: *)
	  let lc = unPP (src f) in
	  let e = trg f in
	  let le = unPP e in
	  let ld = diff lc le in
	  let d = c_PP ld in
	  let led = Cat.coi e d in
	  let lepr = imap (f_PR le) le in
	  let peti = f_PR led AtIndex.atj in
	  let pii = vmap (fun epr -> f_COMP peti epr) lepr in
	  let ldpr = imap (f_PR ld) ld in
	  let petj = f_PR led AtIndex.atk in
	  let pij = vmap (fun dpr -> f_COMP petj dpr) ldpr in
	  let pas = f_RECORD (c_PP led) (pii @@ pij) in
	  let fi = f_COMP pas f in
	  let f = f_ii fi in
(* closing f: *)
	  let pie = vmap (fun epr -> f_COMP f epr) lepr in
	  let cf = f_RECORD d (ldpr @@ pie) in
(* repairing t2: *)
	  let ct2 = t_FT cf t2 in
	  pri "(*Q ";
	  (match SemWSign.s_Pp ls2 with
	  |`OK s2 ->
	      pri " Q "; flush stderr; 
	      let adc = ToolIDule.addcon f s2 in 
	      pri "Q*) "; flush stderr; 
	      let at2 = t_comp ct2 adc in
(* renaming at2: *)
	      let g2 = cod at2 in
	      let lg2 = unpp g2 in
	      let pig2 = imap (t_pr lg2) lg2 in
	      let pig = rmap AtIndex.sp2dule pig2 in
	      let t = t_comp at2 (t_record g2 pig) in
(* fix t: *)
	      let lf = unpp (dom t) in
	      let h = cod t in
	      let lh = unpp h in
	      let lg = diff lf lh in
	      let g = f_pp d lg in
	      let lhg = cof h g in
	      let petf = t_pr lhg AtIndex.atd in
	      let pii = imap (fun i -> t_comp petf (t_pr lh i)) lh in
	      let petg = t_pr lhg AtIndex.ate in
	      let pij = imap (fun i -> t_comp petg (t_pr lg i)) lg in
	      let pas = t_record (f_pp d lhg) (pii @@ pij) in
	      let tf = t_comp pas t in
	      let t = tl_fix tf in
(* cutting cf to ls: *)
	      let pid = imap (fun i -> find i ldpr) (inter ld le2) in
	      let sf = f_RECORD d (pid @@ pie) in
	      let m = ToolWDule.pack (sf, t) in 
	      pri "QE ";
	      `OK m
	  |`Error er -> `Error er)
      |`Error er -> `Error er)

    let m_CoInd = m_Ind (* a hack --- however, this is really dual *)
  end

module SemIDule = SemIDule' (IdIndex) (AtIndex) (IList) (SemFCat) (SemFFunct)
    (ConFTrans) (SrcFCore) (DomFCore) (SemWSign) (ToolWDule) (SemLDule)
    (ToolIDule)


module type IDule = (* module system with inductive modules *)
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module VarStamp : Stamp
    module Cat : T
    module Funct : T
    module Trans : T
    module Location : Location
    module BCore : BCore 
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Location = Location

    type sign = Location.t * sign'
    and sign' =
      |	S_Pp of sign IList.t
      |	S_Bb of sign * Cat.t IList.t * Funct.t IList.t
      |	SC_Bb of sign * BCore.kind IList.t * BCore.typ IList.t
      | S_Ww of dule * sign
      | S_Mm of string * sign
      | S_Ff of sign
      | S_Var of VarStamp.t
    and dule = Location.t * dule'
    and dule' =
      | M_Id of sign
      | M_Comp of dule * dule
      | M_Pr of sign IList.t * IdIndex.t
      | M_Record of sign * dule IList.t * sign IList.t 
      |	M_Base of sign * sign * Funct.t IList.t * Trans.t IList.t
      |	MC_Base of sign * sign * BCore.typ IList.t * BCore.valu IList.t
      | M_Inst of dule * dule
      | M_Trim of dule * sign
      | M_Accord of sign IList.t * dule IList.t * sign IList.t 
      | M_Concord of sign IList.t * dule IList.t * sign IList.t 
      | M_Link of sign IList.t * dule IList.t * sign IList.t 
      | M_Ind of sign IList.t * dule IList.t * sign IList.t 
      | M_CoInd of sign IList.t * dule IList.t * sign IList.t 
      | M_Memo of string * dule
      | M_Finish of Trans.t

    val term_sign : sign -> sign'
    val loc_sign : sign -> Location.t
    val term_dule : dule -> dule'
    val loc_dule : dule -> Location.t
  end

module IDule'
    (IdIndex : IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (VarStamp : Stamp)
    (Cat : T)
    (Funct : T)
    (Trans : T)
    (Location : Location)
    (BCore : BCore 
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Location = Location)
    : (IDule 
    with module IdIndex = IdIndex
    with module IList = IList
    with module VarStamp = VarStamp
    with module Cat = Cat
    with module Funct = Funct
    with module Trans = Trans
    with module Location = Location
    with module BCore = BCore) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module VarStamp = VarStamp
    module Cat = Cat
    module Funct = Funct
    module Trans = Trans
    module BCore = BCore
    module Location = Location

    type sign = Location.t * sign'
    and sign' =
      |	S_Pp of sign IList.t
      |	S_Bb of sign * Cat.t IList.t * Funct.t IList.t
      |	SC_Bb of sign * BCore.kind IList.t * BCore.typ IList.t
      | S_Ww of dule * sign
      | S_Mm of string * sign
      | S_Ff of sign
      | S_Var of VarStamp.t
    and dule = Location.t * dule'
    and dule' =
      | M_Id of sign
      | M_Comp of dule * dule
      | M_Pr of sign IList.t * IdIndex.t
      | M_Record of sign * dule IList.t * sign IList.t 
      |	M_Base of sign * sign * Funct.t IList.t * Trans.t IList.t
      |	MC_Base of sign * sign * BCore.typ IList.t * BCore.valu IList.t
      | M_Inst of dule * dule
      | M_Trim of dule * sign
      | M_Accord of sign IList.t * dule IList.t * sign IList.t 
      | M_Concord of sign IList.t * dule IList.t * sign IList.t 
      | M_Link of sign IList.t * dule IList.t * sign IList.t 
      | M_Ind of sign IList.t * dule IList.t * sign IList.t 
      | M_CoInd of sign IList.t * dule IList.t * sign IList.t 
      | M_Memo of string * dule
      | M_Finish of Trans.t

    let term_sign (_, s) = s
    let loc_sign (l, _) = l
    let term_dule (_, m) = m
    let loc_dule (l, _) = l
  end

module IDule = IDule' (IdIndex) (IList) (Stamp) (ACat) (AFunct) (ATrans)
    (Location) (BCore)


module type PpIDule =
  sig
    module IdIndex : IdIndex
    module IList : IList 
    with type Index.t = IdIndex.t
    module IDule : IDule
    with module IdIndex = IdIndex 
    with module IList = IList

    val pp_s : IDule.sign -> string
    val pp_ls : IDule.sign IList.t -> string
    val pp_m : IDule.dule -> string
    val pp_lm : IDule.dule IList.t -> string
  end

module PpIDule'
    (IdIndex : IdIndex)
    (IList : IList 
    with type Index.t = IdIndex.t)
    (IDule : IDule
    with module IdIndex = IdIndex 
    with module IList = IList) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module IDule = IDule
    open IList
    open IDule

    let pp_lfi lf =
      let s = bfold ""
	  (fun (i, f) s -> 
	    " " ^ IdIndex.t2s i ^ ";" ^ s) lf
      in
      (if s = "" then s else
      String.sub s 1 (String.length s - 2))

    let rec pp_s s =
      match term_sign s with
      | S_Pp ls -> "{" ^ pp_ls ls ^ "}"
      | S_Bb (r, lc, lf) -> "S_Bb(" ^ pp_s r ^ ", " ^ pp_lfi lf ^ ")"
      | SC_Bb (r, lc, lf) -> "SC_Bb(" ^ pp_s r ^ ", " ^ pp_lfi lf ^ ")"
      | S_Ww (m1, s2) -> "S_Ww(" ^ pp_m m1 ^ ", " ^ pp_s s2 ^ ")"
      | S_Mm (n, r) -> "S_Mm(" ^ n ^ ")"
      | S_Ff r -> "S_Ff(" ^ pp_s r ^ ")"
      | S_Var var_num -> "S_Var(" ^ VarStamp.t2string var_num ^ ")"
    and pp_ls ls =
      let s = bfold ""
	  (fun (i, f) s -> 
	    " " ^ IdIndex.t2s i ^ " : " ^ pp_s f ^ ";" ^ s) ls
      in
      IListBasic.pp_stamp ls ^ " " ^ (if s = "" then s else
      String.sub s 1 (String.length s - 2))
    and pp_m m =
      match term_dule m with
      | IDule.M_Id s -> 
	  "M_Id(" ^ pp_s s ^ ")"
      | IDule.M_Comp (m1, m2) -> 
	  "M_Comp(" ^ pp_m m1 ^ ", " ^ pp_m m2 ^ ")" 
      | IDule.M_Pr (lr, i) -> 
	  "M_Pr(" ^ pp_ls lr ^ ", " ^ IdIndex.t2string i ^ ")"
      | IDule.M_Record (r, lm, ls) -> 
	  "M_Record(" ^ pp_s r ^ ", " ^ pp_lm lm ^ " , " ^ pp_ls ls ^ ")"
      | IDule.M_Base (r, s, lg, lt) -> 
	  "M_Base(" ^ pp_s r ^ ", " ^ pp_s s ^ ")"
      | IDule.MC_Base (r, s, lg, lt) ->
	  "MC_Base(" ^ pp_s r ^ ", " ^ pp_s s ^ ")"
      | IDule.M_Inst (m1, m2) -> 
	  "M_Inst(" ^ pp_m m1 ^ ", " ^ pp_m m2 ^ ")" 
      | IDule.M_Trim (m1, r2) -> 
	  "M_Trim(" ^ pp_m m1 ^ ", " ^ pp_s r2 ^ ")" 
      | IDule.M_Accord (lr, lm, ls) ->
	  "M_Accord(" ^ pp_ls lr ^ ", " ^ pp_lm lm ^ " , " ^ pp_ls ls ^ ")"
      | IDule.M_Concord (lr, lm, ls) ->
	  "M_Concord(" ^ pp_ls lr ^ ", " ^ pp_lm lm ^ " , " ^ pp_ls ls ^ ")"
      | IDule.M_Link (lr, lm, ls) -> 
	  "M_Link(" ^ pp_ls lr ^ ", " ^ pp_lm lm ^ " , " ^ pp_ls ls ^ ")"
      | IDule.M_Ind (lr, lm, ls) ->
	  "M_Ind(" ^ pp_ls lr ^ ", " ^ pp_lm lm ^ " , " ^ pp_ls ls ^ ")"
      | IDule.M_CoInd (lr, lm, ls) ->
	  "M_CoInd(" ^ pp_ls lr ^ ", " ^ pp_lm lm ^ " , " ^ pp_ls ls ^ ")"
      | IDule.M_Memo (n, m) -> 
	  "M_Memo(" ^ n ^ ")"
      | IDule.M_Finish t -> 
	  "M_Finish()"
    and pp_lm lm =
      let s = bfold ""
	  (fun (i, f) s -> 
	    " " ^ IdIndex.t2s i ^ " = " ^ pp_m f ^ ";" ^ s) lm
      in
      (if s = "" then s else
      String.sub s 1 (String.length s - 2))
end

module PpIDule = PpIDule' (IdIndex) (IList) (IDule)


module type ElabIDule =
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Location : Location
    module ErrorRepLib : ErrorRepLib
    with module Location = Location
    module Cat : T
    module Funct : T
    module Trans : T
    module IDule : IDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t
    with module Trans = Trans
    with module Location = Location
    val el_sign : IDule.sign -> 
      [`OK of Funct.t|`Error of ErrorRepLib.error]
    val el_dule : IDule.dule -> 
      [`OK of Trans.t
      |`Error of ErrorRepLib.error]
  end

module ElabIDule' 
    (IdIndex : IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Location : Location)
    (ErrorRepLib : ErrorRepLib
    with module Location = Location)
    (Cat : T)
    (Funct : T)
    (Trans : T)
    (OkError : OkError
    with module IdIndex = IdIndex
    with module IList = IList) 
    (SemWSign : SemWSign
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Location = Location
    with module ErrorRepLib = ErrorRepLib
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (SemWDule : SemWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Location = Location
    with module ErrorRepLib = ErrorRepLib
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (SemLDule : SemLDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (SemIDule : SemIDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (BCore : BCore 
    with module IdIndex = IdIndex
    with module IList = IList
    with module Location = Location)
    (ElabBCore : ElabBCore
    with module IdIndex = IdIndex
    with module IList = IList
    with module Location = Location
    with module ErrorRepLib = ErrorRepLib
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t
    with module BCore = BCore)
    (IDule : IDule 
    with module IdIndex = IdIndex
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t
    with module Location = Location
    with module BCore = BCore)
    (CacheFFunct : Cache
    with module Location = Location
    with module ErrorRepLib = ErrorRepLib
    with type Value.t = Funct.t)
    (CacheFTrans : Cache
    with module Location = Location
    with module ErrorRepLib = ErrorRepLib
    with type Value.t = Trans.t)
    : (ElabIDule
    with module IdIndex = IdIndex
    with module IList = IList
    with module Location = Location
    with module ErrorRepLib = ErrorRepLib
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t
    with module IDule = IDule) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Cat = Cat
    module Funct = Funct
    module Trans = Trans
    module IDule = IDule
    module Location = Location
    module ErrorRepLib = ErrorRepLib
    open IList
    open OkError
    open SemWSign
    open SemWDule
    open SemLDule
    open SemIDule
    open ErrorRepLib

    let mem_f = CacheFFunct.create () (* FIXME: memory leak! *)
    let mem_t = CacheFTrans.create () (* FIXME: memory leak! *)

    let loc_error l ok_error =
      match ok_error with
      |`OK m ->
	  `OK m
      |`Error s -> 
	  `Error 
	    (modBackError#instance 
	       [Loc l; 
		Msg s])

    let rec el_sign s =
      let l = IDule.loc_sign s in
      match IDule.term_sign s with
      | IDule.S_Pp ls ->
	  (match vmap1ok el_sign ls with
	  |`OK ls -> 
	      loc_error l (s_Pp ls)
	  |`Error er -> `Error er)
      | IDule.S_Bb (r, lc, lf) -> 
	  (match el_sign r with
	  |`OK r -> 
	      loc_error l (s_Bb r lc lf)
	  |`Error er -> `Error er)
      | IDule.SC_Bb (r, lc, lf) ->
	  (match el_sign r with
	  |`OK r -> 
	      let lc = vmap ElabBCore.elab_kind lc in
	      let l_c2f = vmap (fun f -> fun c -> ElabBCore.elab_typ c f) lf in
	      sc_Bb r lc l_c2f
	  |`Error er -> `Error er)
      | IDule.S_Ww (m1, s2) ->
	  (match el_dule m1 with
	  |`OK m1 ->
	      (match el_sign s2 with
	      |`OK s2 ->
		  loc_error l (s_Ww m1 s2)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.S_Mm (n, s) -> CacheFFunct.el_thunk mem_f (fun () -> el_sign s) n
      | IDule.S_Ff s -> el_sign s
      | IDule.S_Var _ -> loc_error l (s_Pp nil)

    and el_dule m =
      let l = IDule.loc_dule m in
      match IDule.term_dule m with
      | IDule.M_Id s ->
	  (match el_sign s with
	  |`OK s -> 
	      `OK (m_Id s)
	  |`Error er -> `Error er)
      | IDule.M_Comp (m1, m2) ->
	  (match el_dule m1 with
	  |`OK m1 ->
	      (match el_dule m2 with
	      |`OK m2 ->
		  `OK (m_Comp m1 m2)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_Pr (lr, i) ->
	  (match vmap1ok el_sign lr with
	  |`OK lr -> 
	      loc_error l (m_Pr lr i)
	  |`Error er -> `Error er)
      | IDule.M_Record (r, lm, ls) ->
	  (match el_sign r with
	  |`OK r ->
	      (match vmap1ok el_dule lm with
	      |`OK lm -> 
		  (match vmap1ok el_sign ls with
		  |`OK ls -> 
		      loc_error l (m_Record r lm ls)
		  |`Error er -> `Error er)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_Base (r, s, lg, lt) ->
	  (match el_sign r with
	  |`OK r ->
	      (match el_sign s with
	      |`OK s ->
		  loc_error l (m_Base r s lg lt)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.MC_Base (r, s, lg, lt) ->
	  (match el_sign r with
	  |`OK r ->
	      (match el_sign s with
	      |`OK s ->
		  let l_c2g = vmap (fun g -> fun c -> 
		    ElabBCore.elab_typ c g) lg in
		  let l_fh2t = vmap (fun t -> fun f -> fun h ->
		    ElabBCore.elab_valu f h t) lt in
		  mc_Base r s l_c2g l_fh2t l
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_Inst (m1, m2) ->
	  (match el_dule m1 with
	  |`OK m1 ->
	      (match el_dule m2 with
	      |`OK m2 ->
		  loc_error l (m_Inst m1 m2)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_Trim (m1, r2) ->
	  (match el_dule m1 with
	  |`OK m1 ->
	      (match el_sign r2 with
	      |`OK r2 ->
		  loc_error l (m_Trim m1 r2)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_Accord (lr, lm, ls) ->
	  (match vmap1ok el_sign lr with
	  |`OK lr -> 
	      (match vmap1ok el_dule lm with
	      |`OK lm -> 		      
		  (match vmap1ok el_sign ls with
		  |`OK ls -> 
		      loc_error l (m_Accord lr lm ls)
		  |`Error er -> `Error er)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_Concord (lr, lm, ls) ->
	  (match vmap1ok el_sign lr with
	  |`OK lr -> 
	      (match vmap1ok el_dule lm with
	      |`OK lm -> 		      
		  (match vmap1ok el_sign ls with
		  |`OK ls -> 
		      loc_error l (m_Concord lr lm ls)
		  |`Error er -> `Error er)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_Link (lr, lm, ls) -> 
        (* assumption : there is no cyclic dependency in lm (with m_Link) or
           every module depends only on previous ones (with m_Link_ordered) *)
	  (match vmap1ok el_sign lr with
	  |`OK lr -> pri "{";
	      (match bmap1ok (fun (i, m) ->
		pri (" S:" ^ IdIndex.t2s i ^ " ");   
		el_dule m) lm with
	      |`OK lm -> pri "}";
		  (match vmap1ok el_sign ls with
		  |`OK ls -> 
		      let lm = bfold nil (fun iv r -> 
			cons iv r) lm in (* this is just rev! *)
		      loc_error l (m_Link_ordered lr lm ls) (* or m_Link *)
		  |`Error er -> `Error er)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_Ind (lr, lm, ls) -> 
        (* here circularity in lm permitted! *)
	  (match vmap1ok el_sign lr with
	  |`OK lr -> 
	      (match vmap1ok el_dule lm with
	      |`OK lm -> 
		  (match vmap1ok el_sign ls with
		  |`OK ls -> 
		      loc_error l (m_Ind lr lm ls)
		  |`Error er -> `Error er)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_CoInd (lr, lm, ls) -> 
        (* here circularity in lm permitted! *)
	  (match vmap1ok el_sign lr with
	  |`OK lr -> 
	      (match vmap1ok el_dule lm with
	      |`OK lm -> 
		  (match vmap1ok el_sign ls with
		  |`OK ls -> 
		      loc_error l (m_CoInd lr lm ls)
		  |`Error er -> `Error er)
	      |`Error er -> `Error er)
	  |`Error er -> `Error er)
      | IDule.M_Memo (n, m) ->
	  CacheFTrans.el_thunk mem_t (fun () -> el_dule m) n
      | IDule.M_Finish t -> `OK t
  end

module CacheFFunct = Cache' (Location) (ErrorRepLib)
    (struct type t = AFunct.t end)

module CacheFTrans = Cache' (Location) (ErrorRepLib)
    (struct type t = ATrans.t end)

module ElabIDule = ElabIDule' (IdIndex) (IList) (Location) (ErrorRepLib)
    (ACat) (AFunct) (ATrans)
    (OkError) (SemWSign) (SemWDule) (SemLDule) (SemIDule)
    (BCore) (ElabBCore) (IDule) (CacheFFunct) (CacheFTrans)


(* Ordinary product operations (not the name-driven sharing). *)
(* Just for the theory. *)
module type SemPDule =
  sig
    module IdIndex : IdIndex
    module IList : IList with type Index.t = IdIndex.t
    module Cat : T
    module Funct : T
    module Trans : T

    (* ordinary product *)
    val s_Pp_ordinary : Funct.t IList.t -> Funct.t
    val m_Pr_ordinary : Funct.t IList.t -> IdIndex.t -> Trans.t
    val m_Record_ordinary : 
	Funct.t -> Trans.t IList.t -> Funct.t IList.t -> Trans.t

    (* general equalizer (* *)
    val s_Qq : Trans.t IList.t ->
      [`OK of Funct.t|`Error of string]
    val m_Equate : Trans.t IList.t ->
      [`OK of Trans.t|`Error of string]
    val m_Verify : Trans.t IList.t -> Trans.t -> 
      [`OK of Trans.t|`Error of string]
     *)
  end

module SemPDule'
    (IdIndex : IdIndex)
    (IList : IList with type Index.t = IdIndex.t)
    (Cat : SemFCat with module IdIndex = IdIndex and module IList = IList)
    (Funct : ConPFunct
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t)
    (Trans : ConPTrans
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t)
    (SrcFCore : SrcFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with module Cat = Cat
    with type Funct.t = Funct.t)
    (DomFCore : DomFCore
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    (EqFFunct : EqFFunct with type Funct.t = Funct.t)
    (PpFFunct : PpFFunct 
    with type Funct.t = Funct.t
    with module Funct.IdIndex = IdIndex
    with module Funct.IList = IList)
    (OkError : OkError
    with module IdIndex = IdIndex
    with module IList = IList)
    (ToolWDule : ToolWDule
    with module IdIndex = IdIndex 
    with module IList = IList
    with type Cat.t = Cat.t
    with type Funct.t = Funct.t
    with type Trans.t = Trans.t)
    : (SemPDule
    with module IdIndex = IdIndex
    with module IList = IList
    with module Cat = Cat
    with module Funct = Funct
    with module Trans = Trans) =
  struct
    module IdIndex = IdIndex
    module IList = IList
    module Location = Location
    module ErrorRepLib = ErrorRepLib
    module Cat = Cat
    module Funct = Funct
    module Trans = Trans
    open IList
    open Cat
    open Funct
    open Trans
    open SrcFCore
    open ErrorRepLib

    let footPp lc (i, s) =
      f_PR lc i

    let legPp lc (i, s) = 
      f_COMP (footPp lc (i, s)) s

    let lsrcPp ls = vmap src ls

    let s_Pp_ordinary ls = 
      let lc = lsrcPp ls in
      let legs = bmap (legPp lc) ls in
      let c = c_PP lc in
      let body = f_pp c legs in
      body

    let m_Pr_ordinary lr i = (* : S_Pp_ordinary lr -> s *)
      let s = find i lr in
      let lc = lsrcPp lr in
      let foot_i = footPp lc (i, s) in
      let legs = bmap (legPp lc) lr in
      let t = t_pr legs i in
      ToolWDule.pack (foot_i, t)

    let m_Record_ordinary r lm ls = (* : r -> S_Pp_ordinary ls *)
      let lft = vmap ToolWDule.unpack lm in (* : r -> s_i *)
      let lf = vmap (fun (f, t) -> f) lft in
      let lt = vmap (fun (f, t) -> t) lft in
      let c = src r in
      let f = f_RECORD c lf in 
      let t = t_record r lt in
      ToolWDule.pack (f, t)
  end

module SemPDule = SemPDule' (IdIndex) (IList)
    (ConPCat) (ConPFunct) (ConPTrans) (SrcFCore) (DomFCore) (EqFFunct)
    (PpFFunct) (OkError) (ToolWDule)
