/* Patryk Czarnik, sieci komp. zadanie 1 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "mud_common.h"

/* Domyslne nazwy plikow z uzytkownikami i z plansza. */
#define NAZWA_PLIKU_UZYTKOWNIKOW "users.mud"
#define NAZWA_PLIKU_PLANSZY "board.mud"

/* Typy rekordowe dla lokacji na planszy i dla uzytkownika. */
typedef struct {
  char *opis;  /* Tekstowy opis lokacji */
  int N, S, E, W; /* Numer sasiedniej lokacji w danym kierunku lub -1 jesli nie ma. */
} Lokacja;

typedef struct {
  char nazwa[DL_NAZWY], haslo[DL_NAZWY];
  int pozycja;  /* Numer lokacji lub -1 jesli jest niezalogowany */
  int fd;  /* Deskryptor pliku uzywan do komunikacji z tym uzytkownikiem
              za pomaca TCP, ustawiany przy laczeniu z nim. */
  struct sockaddr_in adres_udp; /* Adres udp danego klienta. */
  int dl_adresu;  /* Dlugosc tego adresu. */
} Uzytkownik;

/* Plansza i tablica uzytkownikow sa zapisane w zmiennych globalnych,
 * ktore sa inicjowane w funkcji wczytaj_dane. */
static int rozmiar_planszy;
static int domyslna_pozycja = 0;  /* na tej pozycji umieszczani sa nowi gracze */
static Lokacja *plansza;
static int liczba_wszystkich_graczy; /* w tym niezalogowanych */
static Uzytkownik *gracze;

/* Zbior deskryptorow, na ktorych serwer ma nasluchiwac.
 * To zmienna globalna, gdyz jest modyfikowana w wyloguj_gracza. */
static fd_set aktywne_gniazda;
static int maks_deskr;

/* Globalne sa takze deskryptory gniazd TCP i UDP */
static int desk_tcp=-1;
static int desk_udp=-1;

/** Zwalnia zajete zasoby, bedzie wywolywana przez sygnal SIGINT */
void porzadki(int sygnal)
{
  int i;

  if(desk_tcp >=0)
    close(desk_tcp);
  if(desk_udp >=0)
    close(desk_udp);
  for(i=0; i< liczba_wszystkich_graczy; i++){
    if(gracze[i].pozycja >= 0 && gracze[i].fd > 0)
      close(gracze[i].fd);
  }
  free(gracze);
  for(i=0; i<rozmiar_planszy; i++)
    free(plansza[i].opis);
  free(plansza);
}

/** Wczytuje dane o planszy i uzytkownikach z domyslnych plikow */
void wczytaj_dane(void)
{
  FILE *plik;
  char bufor[MAX_DL_OPISU];
  int i, j;
  
  /* Wczytanie danych o uzytkownikach */
  if((plik = fopen(NAZWA_PLIKU_UZYTKOWNIKOW, "r")) == NULL)
    blad("Nie da sie odczytac pliku uzytkownikow\n");
  if(fscanf(plik, "%d\n", &liczba_wszystkich_graczy) != 1)
    blad("Blad w pliku uzytkownikow (1)\n");
  if((gracze = (Uzytkownik *)malloc(liczba_wszystkich_graczy * sizeof(Uzytkownik))) == NULL)
    blad("Brakuje pamieci na tablice graczy\n");
  for(i=0; i<liczba_wszystkich_graczy; i++){
    if(fscanf(plik, "%[^#]#%[^\n]\n", gracze[i].nazwa, gracze[i].haslo) != 2)
      blad("Blad w pliku uzytkownikow (2)\n");
    gracze[i].pozycja = -1;  /* oznacza ze gracz nie jest zalogowany */
  }
  fclose(plik);
  
  /* Wczytanie danych o planszy */
  if((plik = fopen(NAZWA_PLIKU_PLANSZY, "r")) == NULL)
    blad("Nie da sie odczytac pliku planszy\n");
  if(fscanf(plik, "%d", &rozmiar_planszy) != 1)
    blad("Blad w pliku planszy (1)\n");
  if((plansza = (Lokacja *)malloc(rozmiar_planszy * sizeof(Lokacja))) == NULL)
    blad("Brakuje pamieci na plansze\n");
  for(i=0; i<rozmiar_planszy; i++){
    if((fscanf(plik, "%d%d%d%d%d", &j, &plansza[i].N, &plansza[i].S,
               &plansza[i].E, &plansza[i].W) != 5)
       || (i != j))
      blad("Blad w pliku planszy (2)\n");
    if(fscanf(plik, "%[^#]#", bufor) != 1)
      blad("Blad w pliku planszy (3)\n");
    if((plansza[i].opis = (char*)malloc(strlen(bufor) * sizeof(char))) == NULL)
      blad("Brakuje pamieci na opis lokacji\n");
    strcpy(plansza[i].opis, bufor);
  }
  fclose(plik);
}

/** Wyznacza i ustawia maksymalny nr deskryptora wsrod aktywnych */
void wyznacz_maks_deskr(void)
#define MAX_NR_DESKR 256
{
  int i, max;

  for(i=0; i< MAX_NR_DESKR; i++)
    if(FD_ISSET(i, &aktywne_gniazda))
      max = i;
  maks_deskr = max;
}
#undef MAX_NR_DESKR

/** Wysyla do gracza o podanym numerze komunikat KOM_PISZ z tekstem tekst */
void wyslij(int nr_gracza, const char *tekst)
{
  int fd;
  char do_wyslania[MAX_DL_KOMUNIKATU];

  if(gracze[nr_gracza].pozycja >= 0){
    fd = gracze[nr_gracza].fd;
    if(fd > 0){
      do_wyslania[0] = KOM_PISZ;
      strcpy(do_wyslania+1, tekst);
      if(write(fd, do_wyslania, strlen(do_wyslania+1)+2) < 0)
        pisz_blad("Blad przy write do TCP\n");
    }
  }
}

/** Wysyla do gracza o podanym numerze opis pola, na ktorym on sie znajduje. */
void wyslij_opis(int nr_gracza)
{
  int poz;
  
  if((poz = gracze[nr_gracza].pozycja) >=0){
    /* wysylam przez UDP */
    if(sendto(desk_udp, plansza[poz].opis, strlen(plansza[poz].opis)+1,
           0, (struct sockaddr*)&gracze[nr_gracza].adres_udp, gracze[nr_gracza].dl_adresu) <0)
      pisz_blad("Blad przy wysylaniu opisu\n");
  }
}

/** Wylogowuje gracza o podanym numerze. */
void wyloguj_gracza(int nr_gracza)
{
  int fd, i;
  char pozegnanie[MAX_DL_KOMUNIKATU];

  if(gracze[nr_gracza].pozycja >= 0){
    pozegnanie[0] = '\0';
    strcat(pozegnanie, "Uzytkownik ");
    strcat(pozegnanie, gracze[nr_gracza].nazwa);
    strcat(pozegnanie, " opuscil ten swiat\n");
    for(i=0; i<liczba_wszystkich_graczy; i++)
      if(gracze[i].pozycja == gracze[nr_gracza].pozycja)
        wyslij(i, pozegnanie);

    fd = gracze[nr_gracza].fd;
    /* zamykam gniazdo TCP */
    close(fd);
    /* usuwam deskryptor ze zbioru sluchanych */
    if(FD_ISSET(fd, &aktywne_gniazda)){
      FD_CLR(fd, &aktywne_gniazda);
      wyznacz_maks_deskr();
    }
    /* zaznaczam, ze niezalogowany */
    gracze[nr_gracza].pozycja= -1;
 
    printf("Wylogowano gracza %s\n",gracze[nr_gracza].nazwa);
  }
}

/** Pojawil sie nowy uzytkownik. *
  * Nowy uzytkownik pisze przez deskryptor, bufor to jego komunikat,
  * ktory ma zapisane dlugosc adresu, adres i cos postaci nazwa#haslo\0".
  * Funkcja sprawdza jego dane, jesli sa zle, to odpowiada, ze
  * sie nie zalogowal, jesli poprawne to ewentualnie wylogowuje
  * uzytkownika i loguje go jako nowego.
  */
void nowy_gracz(int deskryptor, char *bufor)
{
  char nazwa[DL_NAZWY], haslo[DL_NAZWY], *napis_haslo;
  int i, dlug;

  memcpy(&dlug, bufor+1, sizeof(int));
  napis_haslo = bufor + 1 + sizeof(int) + dlug;
  nazwa[0] = '\0';
  haslo[0] = '\0';
  if(sscanf(napis_haslo,"%[^#]#%s", nazwa, haslo) >= 1)
    for(i=0; i<liczba_wszystkich_graczy; i++)
      if(strcmp(gracze[i].nazwa, nazwa) == 0){
        /* znalazlem tego uzytkownika */
        if(strcmp(gracze[i].haslo, haslo) == 0){
	  /* podal poprawne haslo */
	  if(gracze[i].pozycja >= 0) /* jest jeszcze zalogowany */
	    wyloguj_gracza(i);
	  gracze[i].fd = deskryptor; /* gracz pisze przez ten deskryptor */
	  gracze[i].pozycja = domyslna_pozycja;
	  /* odczytuje adres UDP klienta */
	  gracze[i].dl_adresu = dlug;
	  memcpy(&gracze[i].adres_udp, bufor+1+sizeof(int), dlug);
          
	  /* wysylam pierwszy komunikat do klienta */
	  nazwa[0] = KOM_ZALOG;
	  memcpy(nazwa+1, &i, sizeof(int));
	  write(deskryptor, nazwa, sizeof(int)+1);
	  printf("Zalogowalem gracza %d %s\n",i, gracze[i].nazwa);
	  return;  /* bez bledu */
	}
      break; /* blad */
      }
  
  /* byl blad, odmawiam zalogowania */
  nazwa[0] = KOM_BLAD;
  strcpy(nazwa+1, "Niepoprawne login lub haslo\n");
  write(deskryptor, nazwa, strlen(nazwa)+1);
  close(deskryptor);
  if(FD_ISSET(deskryptor, &aktywne_gniazda)){
    FD_CLR(deskryptor, &aktywne_gniazda);
    wyznacz_maks_deskr();
  }
}

/** Wykonanie komendy chat. */
void chat(int nr_gracza, char *tekst)
{
  int i;
  char do_wyslania[MAX_DL_KOMUNIKATU];
  
  /* do wsystkich zalogowanych graczy wysyla tekst */
  if(gracze[nr_gracza].pozycja >=0)
   for(i=0; i< liczba_wszystkich_graczy; i++)
    if(gracze[i].pozycja >= 0){
      do_wyslania[0] = KOM_PISZ;
      do_wyslania[1] = '\0';
      strcat(do_wyslania+1, "[CHAT] ");
      strcat(do_wyslania+1, gracze[nr_gracza].nazwa);
      strcat(do_wyslania+1, " mowi:\n");
      strcat(do_wyslania+1, tekst);
      wyslij(i, do_wyslania);
    }
}

/** Wykonanie komendy say. */
void say(int nr_gracza, char *tekst)
{
  int i;
  char do_wyslania[MAX_DL_KOMUNIKATU];
  
  /* do wszystkich graczy w tej samej lokacji, oprocz wysylajacego,
   * wysyla teskt */
  if(gracze[nr_gracza].pozycja >=0)
   for(i=0; i< liczba_wszystkich_graczy; i++)
    if(i != nr_gracza && gracze[i].pozycja == gracze[nr_gracza].pozycja){
      do_wyslania[0] = KOM_PISZ;
      do_wyslania[1] = '\0';
      strcat(do_wyslania+1, "[SAY] ");
      strcat(do_wyslania+1, gracze[nr_gracza].nazwa);
      strcat(do_wyslania+1, " mowi:\n");
      strcat(do_wyslania+1, tekst);
      wyslij(i, do_wyslania);
    }
}

/** Wykonanie komendy N, S, E lub W w zaleznosci od parametru kierunek. */
void przejdz(int nr_gracza, char kierunek)
{
  int poz;
  
  poz = gracze[nr_gracza].pozycja;
  switch(kierunek){
    case 'N' : poz = plansza[poz].N; break;
    case 'S' : poz = plansza[poz].S; break;
    case 'E' : poz = plansza[poz].E; break;
    case 'W' : poz = plansza[poz].W; break;
    default : return;
  }
  if(poz < 0)
    wyslij(nr_gracza, "Nie ma takiego wyjscia\n");
  else {
    /* zmiana pozycji */
    gracze[nr_gracza].pozycja = poz;
    /* od razu wysylam opis pozycji */
    wyslij_opis(nr_gracza);
  }
}

/** Interpretacja komunikatu, ktory nadszedl przez TCP od jednego
 * z graczy. */
int interpretuj(int deskryptor, char *bufor, int rozmiar)
/* rozmiar >= 0 */
{
  if(rozmiar == 0){  /* koniec pliku - awaryjnie zamykam to gniazdo */
    pisz_blad("Wykrylem koniec pliku, zamykam gniazdo\n");
    close(deskryptor);
    if(FD_ISSET(deskryptor, &aktywne_gniazda)){
      FD_CLR(deskryptor, &aktywne_gniazda);
      wyznacz_maks_deskr();
    }
  }
  else { /* rozmiar > 0 */
    switch(bufor[0]){
       /* na pozycji bufor+1 znajduje sie liczba int, nr portu udp lub
        * nr gracza, to dziwne rzutowanie sluzy odczytaniu tej liczby */
      case KOM_NOWY :
        if(rozmiar < 3+ sizeof(int))
	  return -1;
	nowy_gracz(deskryptor, bufor);
	break;
      case KOM_QUIT :
        if(rozmiar < 1+ sizeof(int))
	  return -1;
	wyloguj_gracza(*(int*)(bufor+1));
	break;
      case KOM_LOOK :
        if(rozmiar < 1+ sizeof(int))
	  return -1;
	wyslij_opis(*(int*)(bufor+1));
	break;
      case KOM_CHAT :
        if(rozmiar < 2+ sizeof(int))
	  return -1;
	chat(*(int*)(bufor+1), bufor+1+sizeof(int));
	break;
      case KOM_SAY :
        if(rozmiar < 2+ sizeof(int))
	  return -1;
	say(*(int*)(bufor+1), bufor+1+sizeof(int));
	break;
      case KOM_N :
        if(rozmiar < 1+ sizeof(int))
	  return -1;
	przejdz(*(int*)(bufor+1), 'N');
	break;
      case KOM_S :
        if(rozmiar < 1+ sizeof(int))
	  return -1;
	przejdz(*(int*)(bufor+1), 'S');
	break;
      case KOM_E :
        if(rozmiar < 1+ sizeof(int))
	  return -1;
	przejdz(*(int*)(bufor+1), 'E');
	break;
      case KOM_W :
        if(rozmiar < 1+ sizeof(int))
	  return -1;
	przejdz(*(int*)(bufor+1), 'W');
	break;

      default : return -1;
    }
  }
  return 0;
}

/** Wlasciwe dzialanie serwera */
void serwuj(void)
{
  int nowy, i, dlug, len;
  struct sockaddr_in adres_serwera, adres_udp; /* adresy serwera */
  fd_set zbior;
  char bufor[MAX_DL_KOMUNIKATU];

  /* Tworzenie jednego gniazda UDP do wysylania opisow */
  if((desk_udp = socket(AF_INET, SOCK_DGRAM, 0)) <0)
    blad("Blad przy otwieraniu gniazda UDP\n");
  bzero(&adres_udp, sizeof(adres_udp));
  adres_udp.sin_family = AF_INET;
  adres_udp.sin_addr.s_addr = htonl(INADDR_ANY);
  adres_udp.sin_port = htons(NR_PORTU_UDP);
  if (bind(desk_udp, (struct sockaddr*) &adres_udp, sizeof adres_udp) < 0)
    blad("Blad przy bindowaniu UDP\n");

  /* Tworzenie jednego gniazda TCP do otrzymywania nowych zgloszen */
  desk_tcp = socket(AF_INET, SOCK_STREAM, 0);
  if (desk_tcp < 0)
    blad("Blad przy otwieraniu gniazda TCP\n");

  bzero(&adres_serwera, sizeof(adres_serwera));
  adres_serwera.sin_family = AF_INET;
  adres_serwera.sin_addr.s_addr = INADDR_ANY;
  adres_serwera.sin_port = htons(NR_PORTU);
  len = sizeof(adres_serwera);
  if (bind (desk_tcp, (struct sockaddr*)&adres_serwera, len) < 0) 
    blad("Blad przy bindowaniu TCP\n");
  /* jaki mam adres ? */
  if(getsockname(desk_tcp, (struct sockaddr*)&adres_serwera, &len)<0)
    blad("Blad przy getsockname\n");
  printf("Numer portu TCP: %d\n", ntohs(adres_serwera.sin_port));
  /* Inicjowanie zbioru deskryptorow gniazd, ktorych bede sluchal */
  FD_ZERO(&aktywne_gniazda);
  FD_SET(desk_tcp, &aktywne_gniazda);
  maks_deskr = desk_tcp;
  
  listen(desk_tcp, 5);

  /* Przerwanie SIGINT zrobi porzadki */
  if(signal(SIGINT, porzadki) == SIG_ERR)
    blad("Blad signal");

/* No i sie krecimy */
  while(1) {
    /* Kopiuje zbior deskryptorow do sluchania na zmienna lokalna */
    memcpy(&zbior, &aktywne_gniazda, sizeof(zbior));
    /* czekam az ktos cos przysle */
    if (select(maks_deskr+1, &zbior, NULL, NULL, NULL) < 0)
      blad("Blad select\n");
    if (FD_ISSET(desk_tcp, &zbior)) {
      /* Pojawia sie nowy klient */
      if((nowy = accept(desk_tcp, NULL, NULL)) == -1)
	pisz_blad("accept");
      else {
        /* Bede sluchal takze na tym deskryptorze */
        FD_SET(nowy, &aktywne_gniazda);
	if(nowy > maks_deskr)
	  maks_deskr = nowy;
      }
    } else {
      /* Komunikat od zalogowanego uzytkownika */
      for(i=0; i<=maks_deskr; i++)
        if(i != desk_tcp && FD_ISSET(i, &zbior)){
          if((dlug = read(i, bufor, MAX_DL_KOMUNIKATU)) <0)
	    pisz_blad("blad przy read z TCP\n");
	  else
            interpretuj(i, bufor, dlug);
        }
    }
  }
}

int main()
{
  wczytaj_dane();
  serwuj();
  return 0;
}
